class Map
{  
  // Constructor: tmpElementSize is the width/height of one block in pixel
  Map( int tmpElementSize )
  {
    elementSize = tmpElementSize;
    images = new PImage[26];
  }
  
  // Constructor: Loads a map file
  Map( String mapFile )
  {
    images = new PImage[26];
    loadFile( mapFile );
  }
    
  int widthPixel()
  {
    return w * elementSize;
  }
  
  int heightPixel()
  {
    return h * elementSize;
  }
  
  // Returns the character at block position x,y. '_' for invalid positions (out of range)
  char at( int x, int y )
  {
    if ( x < 0 || y < 0 || x >= w || y >= h )
      return '_';
    else
      return map[y].charAt(x);
  }
  
  // Replaces s.charAt(index) with ch
  String replace (String s, int index, char ch)
  {
    return s.substring(0, index)+ch+s.substring(index+1,s.length());
  }
  
  // If the dimension of the map is below width times height
  // _ are appended in each line and full lines are appended
  // such that it is width times height.
  void extend (int width, int height) {
    while (height>h) {
      map = append(map, "");
      h++;
    }
    if (w<width) w = width;
    for (int y=0; y<h; y++) {
      while (map[y].length()<w) 
        map[y] = map[y] + "_";
    }
  }
  
  // Sets the character at block position x,y
  // Coordinates below 0 are ignored, for coordinates
  // beyond the map border, the map is extended
  void set (int x, int y, char ch) {
    if ( x < 0 || y < 0 ) return;
    extend (x+1, y+1);
    map[y] = replace (map[y], x, ch);
  }
    
  // True if the rectangle given by centerX, centerY, w, h (partially) contains an element with a character
  // from list
  boolean testCharacterInRect( float centerX, float centerY, float w, float h, String list )
  {
    int startX = floor((centerX-w/2) / elementSize),
        startY = floor((centerY-h/2) / elementSize),
        endX   = floor((centerX+w/2) / elementSize),
        endY   = floor((centerY+h/2) / elementSize);
        
    for ( int x = startX; x <= endX; ++x )
    {
      for ( int y = startY; y <= endY; ++y )
      {
        if ( list.indexOf( at(x, y) ) != -1 )
          return true;
      }
    }
    return false;
  }
  
  // True if the rectangle is completely inside one element and it is one of the characters from the list
  boolean testCharacterFullyInsideRect( float centerX, float centerY, float w, float h, String list )
  {
    int startX = floor((centerX-w/2)/ elementSize),
        startY = floor((centerY-h/2)/ elementSize),
        endX   = floor((centerX+w/2) / elementSize),
        endY   = floor((centerY+h/2) / elementSize);
        
    return ( startX == endX && startY == endY && list.indexOf( at(startX, startY) ) != -1 );
  }
  
  // Draws the map on the screen, starting at offsetX,offsetY in pixels. Offset may be negative.
  // Use this to shift the visible pat of the map
  void draw( float offsetX, float offsetY )
  {
    int startX = floor(offsetX / elementSize),
        startY = floor(offsetY / elementSize);
    for ( int y = startY; y < startY + height/elementSize + 2; ++y )
    {
      for ( int x  = startX; x < startX + width/elementSize + 2; ++x )
      {
        PImage img = null;
        char element = at( x, y );
        if ( element == '_' )
          img = outsideImage;
        else if ('A'<=element && element<='Z')
          img = images[at( x, y ) - 'A'];
        if ( img != null )
          image( img, 
                 x*elementSize - offsetX, 
                 y*elementSize - offsetY, 
                 elementSize, elementSize );
      }
    }
  }
  
  PImage tryLoadImage (String imageFilename) {
    //println("Trying "+imageFilename);
    if (createInput(imageFilename)!=null) {
      //println("Found");
      return loadImage (imageFilename);
    }
    else return null;
  }
  
  // Loads an image named imageName from a locatation relative
  // to the map file mapFile. It must be either in the same
  // directory, or in a subdirectory images, or in a parallel
  // directory images.
  PImage loadImageRelativeToMap (String mapFile, String imageName) {
    File base = new File(mapFile);
    File parent = base.getParentFile();
    PImage img;
    img = tryLoadImage (new File (parent,imageName).getPath());
    if (img!=null) return img;
    img = tryLoadImage (new File (parent, "images/"+imageName).getPath());
    if (img!=null) return img;
    img = tryLoadImage (new File (parent, "../images/"+imageName).getPath());
    return img;
  }
  
  // Goes through all images loaded and determine selementsize as amx
  // If image sizes are not square and equal a warning message is printed
  void determineElementSize () {
    elementSize = 0;
    PImage[] allImages = (PImage[]) append (images, outsideImage);
    for (int i=0; i<allImages.length; i++) if (allImages[i]!=null) {
      if (elementSize>0 && 
          (allImages[i].width!=elementSize || allImages[i].height!=elementSize))
          println ("WARNING: Images are not square and of same size");
      if (allImages[i].width>elementSize)  elementSize = allImages[i].width;
      if (allImages[i].height>elementSize) elementSize = allImages[i].height;      
    }
    if (elementSize==0) throw new Error ("No image could be loaded.");
  }

 
  // Loads a map file
  // element size is obtained from the first image loaded
  void loadFile( String mapFile )
  {
    map = loadStrings( mapFile );
    if (map==null) 
      throw new Error ("Map "+mapFile+" not found.");
    while (map.length>0 && map[map.length-1].equals(""))
       map = shorten(map);
    h = map.length;
    if ( h == 0 ) 
      throw new Error("Map has zero size");
    w = map[0].length();
    
    // Load images
    for (char c='A'; c<='Z'; c++) 
         images[c - 'A'] = loadImageRelativeToMap (mapFile , c + ".png" );        
    outsideImage = loadImageRelativeToMap (mapFile, "_.png");
    
    for ( int y = 0; y < h; ++y )
    {
      String line = map[y];
      if ( line.length() != w )
        throw new Error("Not every line in map of same length");
        
      for ( int x = 0; x < line.length(); ++x )
      {
        char c = line.charAt(x);
        if (c==' ' || c=='_') {}
        else if ('A'<=c && c<='Z') {
          if (images[c - 'A'] == null) 
            throw new Error ("Image for "+c+".png missing");        
        }
        else throw new Error("map must only contain A-Z, space or _");      
      }
    }
    
    
    determineElementSize ();
  }
  
  // Saves the map into a file
  void saveFile (String mapFile) {
    saveStrings (mapFile, map);
  }

  // *** variables ***
  // element x, y is map[y].charAt(x)
  String map[];
  // images[c-'A'] is the image for element c
  PImage images[];
  // special image drawn outside the map
  PImage outsideImage;
  // map dimensions in elements
  int w, h;
  // width and height of an element in pixels
  int elementSize;
}

