import java.awt.*;
import java.awt.event.*;
import java.util.*;

//! Class containing both the coal cart simulation and the GUI
public class CoalCart2D extends Frame implements KeyListener
{
    static final long serialVersionUID = 1;
    
    //! True position of the center between the carts back wheels [m]
    double cartX, cartY;
    //! True orientation of the carts forward direction with respect
    //! to world X frame [rad]
    double cartPhi = 0;

    CoalCartParameter parameter = new CoalCartParameter();    

    //! Scale used when starting the system.
    double defaultScale = 8;    
    
    //! Scale [pixel/m] used for rendering all graphics
    double scale = defaultScale;

    //! Margin around the workspace in pixel
    int margin = 10;    
    
    //! The random number generator used for simulation
    Random rnd = new Random ();

    //! \c lastObservations[i] is the last angle to beacon i measured the last
    //! time or NaN if not existing.
    double[] lastObservations; 

    KalmanFilter2D ekf = new KalmanFilter2D (parameter);    

    //! Component showing the coal cart
    class CoalCartComponent extends Component implements ComponentListener
    {    
        static final long serialVersionUID = 1;
	CoalCartComponent () 
	{
	    super ();
	    setBackground (Color.lightGray);	    
	    addComponentListener (this);	    
	}
	
	int xOf(double x) 
	{
	    return (int) (x*scale)+margin;
	}

	int yOf(double y) 
	{
	    return getHeight()-margin - (int) (y*scale);
	}

	void paintLineInWorldCoordinates (Graphics g, double x0, double y0, double x1, double y1)
	{
	    g.drawLine (xOf (x0), yOf(y0), xOf (x1), yOf(y1));	    
	}	

	void paintLineInCartCoordinates (Graphics g, double cartX, double cartY, double cartPhi,
					 double x0, double y0, double x1, double y1)
	{
	    double c = Math.cos(cartPhi), s = Math.sin(cartPhi);
	    paintLineInWorldCoordinates (g, cartX + x0*c - y0*s, cartY + x0*s + y0*c, 
					 cartX + x1*c - y1*s, cartY + x1*s + y1*c);	    
	}	


	void paintCoalCart (Graphics g, Color color, double cartX, double cartY, double cartPhi, boolean doFill)
	{
	    double[][] cs = parameter.cartShape;	    
	    int[] xx = new int[cs.length];
	    int[] yy = new int[cs.length];
	    double c = Math.cos(cartPhi), s = Math.sin(cartPhi);
	    for (int i=0; i<cs.length; i++) {
		xx[i] = xOf (cartX + c*cs[i][0] - s*cs[i][1]);
		yy[i] = yOf (cartY + s*cs[i][0] + c*cs[i][1]);
	    }
	    Polygon p = new Polygon(xx, yy, cs.length);
	    g.setColor (color);
	    if (doFill) {
		g.fillPolygon (p);
		g.setColor (Color.white);
		paintCross (g, cartX, cartY);		
	    }	    
	    else {
		g.drawPolygon (p);	    
		g.setColor (Color.black);
		paintCross (g, cartX, cartY);		
	    }	   	    
	}


	void paintCross (Graphics g, double x, double y)
	{
	    int ix = xOf (x), iy = yOf(y);
	    g.drawLine (ix-5, iy, ix+5, iy);
	    g.drawLine (ix, iy-5, ix, iy+5);	    
	}


	void paintBeacon (Graphics g, double x, double y)
	{
	    g.setColor (Color.red);
	    paintCross (g, x, y);
	}	


	void paintBorder (Graphics g)
	{
	    double w =  parameter.workspaceLength;	    
	    g.setColor (Color.black);
	    paintLineInWorldCoordinates (g, 0, 0, w, 0);
	    paintLineInWorldCoordinates (g, w, 0, w, w);	    
	    paintLineInWorldCoordinates (g, w, w, 0, w);	    
	    paintLineInWorldCoordinates (g, 0, w, 0, 0);
	}


	//! Paints an phi+/-deltaPhi sector at x, y
	void paintOrientationSector (Graphics g, double x, double y, double phi, double deltaPhi)
	{
	    int l = 60;	    
	    int xx = xOf(x), yy = yOf(y);	    
	    double phim = phi-deltaPhi;	    
	    g.drawArc (xx-l, yy-l, 2*l, 2*l, (int) (180/Math.PI*phim), (int) (180/Math.PI*2*deltaPhi));
	    g.drawLine (xx, yy, xx+(int) (l*Math.cos(phi-deltaPhi)), yy-(int) (l*Math.sin(phi-deltaPhi)));
	    g.drawLine (xx, yy, xx+(int) (l*Math.cos(phi+deltaPhi)), yy-(int) (l*Math.sin(phi+deltaPhi)));	    
	}


	void paintXYUncertaintyEllipse (Graphics g, double x, double y, double s0, double s1, double angle)
	{
	    double c = Math.cos(angle), s = Math.sin(angle);
	    paintLineInWorldCoordinates (g, x+c*s0, y+s*s0, x-c*s0, y-s*s0);
	    paintLineInWorldCoordinates (g, x-s*s1, y+c*s1, x+s*s1, y-c*s1);
	    double xOld=0, yOld=0;
	    boolean first = true;	    
	    for (double alpha=0; alpha<2*Math.PI+0.0001; alpha += 0.05) {
		double x1 = Math.cos(alpha)*s0;
		double y1 = Math.sin(alpha)*s1;
		double x2 = x + c*x1 -s*y1;
		double y2 = y + s*x1 +c*y1;
		if (!first) paintLineInWorldCoordinates (g, xOld, yOld, x2, y2);
		first = false;		
		xOld = x2;		
		yOld = y2;		
	    }	    
	}
	
	

	void paintEstimate (Graphics g)
	{
	    if (ekf==null) return;
	    KalmanFilter2D.Estimate es = ekf.computeEstimateForGUI ();
	    if (es==null) return;
	    paintCoalCart (g, Color.green, es.x, es.y, es.phi, false);
	    g.setColor (Color.red);	    
	    paintOrientationSector (g, es.x, es.y, es.phi, es.sigmaPhi);
	    paintXYUncertaintyEllipse (g, es.x, es.y, es.sigmaXYLarge, es.sigmaXYSmall, es.sigmaXYAngle);
	}	


        public void paint (Graphics g)
        {	    
	    g.setColor(getBackground());
	    g.fillRect(0, 0, getWidth(), getHeight());

            g.setColor (Color.black);
	    paintBorder (g);
	    double[][] b = parameter.beacon;	    
	    for (int i=0; i<b.length; i++) paintBeacon (g, b[i][0], b[i][1]);
	    paintCoalCart (g, Color.black, cartX, cartY, cartPhi, true);
	    g.setColor (Color.red);	    
	    double farAway = 5*parameter.workspaceLength;	    
	    if (lastObservations!=null)
		for (int i=0; i<lastObservations.length; i++) if (!Double.isNaN(lastObservations[i])){
		    double c = Math.cos (lastObservations[i]), s = Math.sin (lastObservations[i]);
		    paintLineInCartCoordinates (g, cartX, cartY, cartPhi, 0, 0, c*farAway, s*farAway);
		}
	    paintEstimate (g);
	}

        public Dimension getPreferredSize() 
        {
            return new Dimension ((int) (defaultScale*parameter.workspaceLength+2*margin), (int) (defaultScale*parameter.workspaceLength+2*margin));
        }	

	public void	componentHidden(ComponentEvent e) 
	{}		
	
	public void	componentMoved(ComponentEvent e) 
	{
	}
	
	public void	componentResized(ComponentEvent e)
	{
	    int w = getWidth() - 2*margin;
	    int h = getHeight() - 2*margin;
	    if (w>h) scale = h/parameter.workspaceLength;
	    else scale = w/parameter.workspaceLength;	
//	    System.out.println ("Resized w="+w+"h="+h+"scale="+scale);	    
	    repaint ();
	}
	
	public void	componentShown(ComponentEvent e)
	{
	}
    }    

    void printFlags ()
    {
	System.out.println ("autoMeasure="+autoMeasure+", doMeasureManyBeacons="+doMeasureManyBeacons+
                            ", doUseNoise="+doUseNoise);
    }
    

    //! The upper component that shows the cart and the beacons sensor
    CoalCartComponent ccc;

    //! Label for showing text messages
    Label statusText;    

    //! Whether to make a measurement after each update step
    boolean autoMeasure = false;

    //! Whether to measure several beacons each time
    boolean doMeasureManyBeacons = false;

    //! Whether to corrupt the simulated measurements
    boolean doUseNoise = true;    

    CoalCart2D ()
    {
        super ("Tracking a Coal Cart with an Extended Kalman Filter in 2D");

        MenuItem mi;
        
	MenuBar menuBar = new MenuBar();
	setMenuBar (menuBar);

	Menu fileMenu = new Menu("File");
	menuBar.add (fileMenu);

	mi = new MenuItem ("Restart");
	mi.addActionListener (new ActionListener() {
                public void actionPerformed(ActionEvent e) {
		    init ();		    
		}		
	    });	
	fileMenu.add (mi);

	mi = new MenuItem ("Quit");
	mi.addActionListener (new ActionListener() {
                public void actionPerformed(ActionEvent e) {
		    System.exit  (0);		    
		}		
	    });	
	fileMenu.add (mi);

	Menu optionsMenu = new Menu("Options");
	menuBar.add (optionsMenu);

        CheckboxMenuItem cmi = new CheckboxMenuItem("Auto measure", autoMeasure);
        cmi.addItemListener (new ItemListener() {
		public void itemStateChanged(ItemEvent e) {
		    CheckboxMenuItem tcmi = (CheckboxMenuItem) e.getSource();
		    autoMeasure = tcmi.getState();
		    printFlags ();		    
		}
            });
        optionsMenu.add (cmi);
        
        cmi = new CheckboxMenuItem("Measure multiple beacons", doMeasureManyBeacons);
        cmi.addItemListener (new ItemListener() {
		public void itemStateChanged(ItemEvent e) {
		    CheckboxMenuItem tcmi = (CheckboxMenuItem) e.getSource();
		    doMeasureManyBeacons = tcmi.getState();
		    printFlags ();		    
                    }
            });
        optionsMenu.add (cmi);

        cmi = new CheckboxMenuItem("activate measurement noise", doUseNoise);
        cmi.addItemListener (new ItemListener() {
		public void itemStateChanged(ItemEvent e) {
		    CheckboxMenuItem tcmi = (CheckboxMenuItem) e.getSource();
		    doUseNoise = tcmi.getState();
		    printFlags ();		    
		}
            });
        optionsMenu.add (cmi);

	Menu  checkMenu= new Menu("Debug");
	menuBar.add(checkMenu);
	
	mi = new MenuItem ("Check df(x,u)/dx");
	mi.addActionListener (new ActionListener() {
                public void actionPerformed(ActionEvent e) {
		    if (ekf!=null) ekf.checkDynamicModelJacobianX ();
		}		
	    });	
	checkMenu.add (mi);

	mi = new MenuItem ("Check df(x,u)/du");
	mi.addActionListener (new ActionListener() {
                public void actionPerformed(ActionEvent e) {
		    if (ekf!=null) ekf.checkDynamicModelJacobianU ();
		}		
	    });	
	checkMenu.add (mi);

	mi = new MenuItem ("Check dg(x)/du");
	mi.addActionListener (new ActionListener() {
                public void actionPerformed(ActionEvent e) {
		    if (ekf!=null) ekf.checkMeasurementModelJacobian ();
		}		
	    });	
	checkMenu.add (mi);
	

	ccc = new CoalCartComponent();	
	add (ccc, "Center");


    	statusText = new Label ();
	add (statusText, "South");	

	cartX = cartY = parameter.workspaceLength/2;
	cartPhi = 0;	
        addKeyListener(this);
        
        pack();        
        setVisible(true);        
    }    

    public void keyTyped(KeyEvent evt) {
    }
    
    public void keyReleased(KeyEvent evt) {
    }


    public void init ()
    {
	cartX   = parameter.initialCartX;
	cartY   = parameter.initialCartY;
	cartPhi = parameter.initialCartPhi;	
	ekf.initialize (parameter);
	repaint ();	
    }    


    public void move (double translation, double rotation)
    {
	// Update the true position according to exact motion
	double c = Math.cos(cartPhi), s = Math.sin(cartPhi);
	cartX   += c*translation;
	cartY   += s*translation;
	cartPhi += rotation;
	// Compute exact measurement
	double uLeft  = translation - rotation*parameter.cartWidth/2;
	double uRight = translation + rotation*parameter.cartWidth/2;
	// Distort it by noise
	if (doUseNoise) {	    
	    uLeft  += parameter.dynamicNoiseSigma*Math.sqrt(Math.abs(uLeft))*rnd.nextGaussian ();
	    uRight += parameter.dynamicNoiseSigma*Math.sqrt(Math.abs(uRight))*rnd.nextGaussian ();	
	}	
	// Pass it to EKF
	if (ekf!=null) ekf.dynamic (uLeft, uRight);
	repaint ();
    }
    

    public void measure ()
    {
	// Measure all observable beacons
	double[][] b = parameter.beacon;	
	lastObservations = new double[b.length];	
	int ctr=0;	
	for (int i=0; i<b.length; i++) {
	    double dX = b[i][0] - cartX;
	    double dY = b[i][1] - cartY;
	    double angle = Math.atan2 (dY, dX) - cartPhi;
	    angle = angle % (2*Math.PI);
	    if (angle>Math.PI) angle -= 2*Math.PI;
	    else if (angle<-Math.PI) angle += 2*Math.PI;	    
	    if (doUseNoise) angle += parameter.measurementNoiseSigma * rnd.nextGaussian();	    
	    if (-parameter.fieldOfView <= angle && angle <= parameter.fieldOfView) {
		lastObservations[i] = angle;
		ctr++;
	    }	    
	    else lastObservations[i] = Double.NaN;	    
	}

	// If only one at a time is desired remove all except a random one
	if (!doMeasureManyBeacons && ctr>0) {
	    int whichMeasurement = rnd.nextInt (ctr);
	    for (int i=0; i<lastObservations.length; i++) 
		if (!Double.isNaN(lastObservations[i])) {
		    if (whichMeasurement!=0) lastObservations[i]=Double.NaN;
		    whichMeasurement--;
		}		    
	}	


	// Pass measurements to EKF
	if (ekf!=null) 
	    for (int i=0; i<lastObservations.length; i++) 
		if (!Double.isNaN(lastObservations[i]))
		ekf.measurement (lastObservations[i], i);	
	repaint ();	
    }

    void clearMeasurement ()
    {
	lastObservations = null;	
    }    
    
    
    public void keyPressed(KeyEvent evt) {
      int key = evt.getKeyCode();  // keyboard code for the key that was pressed
      
      if (key == KeyEvent.VK_LEFT) {
	  move (0, Math.PI/8);	  
          if (autoMeasure) measure();
	  else clearMeasurement ();	  
      }      
      else if (key == KeyEvent.VK_RIGHT) {
	  move (0, -Math.PI/8);
          if (autoMeasure) measure();
	  else clearMeasurement ();	  
      }
      else if (key == KeyEvent.VK_UP) {
	  move (2.5, 0);	  
          if (autoMeasure) measure();
	  else clearMeasurement ();	  
      }
      else if (key == KeyEvent.VK_DOWN) {
	  move (-2.5, 0);	  
          if (autoMeasure) measure();
	  else clearMeasurement ();	  
      }
      else if (key== KeyEvent.VK_SPACE) {
	  measure ();	  
      }
      else if (key== KeyEvent.VK_A) {
	  autoMeasure = !autoMeasure;
	  printFlags ();
      }
      else if (key== KeyEvent.VK_M) {
	  doMeasureManyBeacons = !doMeasureManyBeacons;
	  printFlags ();
      }
      else if (key== KeyEvent.VK_N) {
	  doUseNoise = !doUseNoise;
	  printFlags ();
      }      
    }
    

    public static void main(String[] args) 
    {
	CoalCart2D cc = new CoalCart2D();
        cc.addWindowListener(new java.awt.event.WindowAdapter() {
                public void windowClosing(WindowEvent winEvt) {
                    System.exit(0); 
                }
            });
    }

}
