import java.io.*;

// ATTENTION
// Matrices and vectors are stored as Jama.Matrix objects.
// The documentation of the Jama matrix package can be found at
// http://math.nist.gov/javanumerics/jama/doc/
import Jama.*;

// ATTENTION
// You can create matrices on the fly, i.e. by writing down terms for coefficients
// with the following or similar expressions:
// new Matrix (new double[][] 
//               {{ a00, a01, a02}, 
//                {a10, a11, a12}}
//             )
// A shortcut for vectors, i.e. n*1 matrices is
// newVector (new double[] {v0, v1, v2})


class KalmanFilter2D 
{
    //! The parameter of the cart used in the filter
    CoalCartParameter parameter;
    
    //! The filter's belief on the current state, actually a vector
    Matrix muX;
    //! and state covariance
    Matrix covX;

    //! Auxiliary function that creates a column vector
    //! as a Jama.Matrix object from a 1D double array
    static Matrix newVector (double[] v)
    {
	return new Jama.Matrix (v, v.length);
    }
    
    
    KalmanFilter2D (CoalCartParameter p)
    {
	initialize (p);	
    }
    
    //! Initialize the mean and covariance to roughly the robot being at 50,50,0
    void initialize (CoalCartParameter p)
    {
	parameter = p;	

	// ADD CODE HERE

	if (muX!=null) checkConsistency ();	
    }


    //! Function f returning the state obtained after starting in state
    //! \c x and moving with dynamic measurement \c u. Without taking
    //! uncertainty into account.
    Matrix dynamicModel (Matrix x, Matrix u)
    {
	// ADD CODE HERE
	return null;	
    }

    //! Jacobian of f (\c dynamicModel) with respect to X
    Matrix dynamicModelJacobianX (Matrix x, Matrix u)
    {
	// ADD CODE HERE
	return null;	
    }


    //! Jacobian of f (\c dynamicModel) with respect to U
    Matrix dynamicModelJacobianU (Matrix x, Matrix u)
    {
	// ADD CODE HERE
	return null;	
    }


    //! Measurement function g for the observation of beacon[beaconId]
    //! This function returns the value that should have been measured
    //! concerning beacon \c beaconId when the robot were in state \c x.
    Matrix measurementModel (Matrix x, int beaconId)
    {
	// ADD CODE HERE
	return null;	
    }

    //! Jacobian of g (\c measurementModel) with respect to X
    Matrix measurementModelJacobian (Matrix x, int beaconId)
    {
	// ADD CODE HERE
	return null;	
    }
    
    
    //! Update the EKF after one dynamic step, i.e. with a
    //! dynamic measurement u
    void dynamic (Matrix u)
    {
	// ADD CODE HERE
	if (covX!=null) {	    
	    System.out.println ("Covariance after dynamic");	
	    covX.print(8, 4);
	    checkConsistency ();
	}	
    }

    //! Wrapper for calling dynamic with two doubles
    //! namely the distance traveled by the left and right wheel
    //! respectively
    void dynamic (double uLeftWheel, double uRightWheel)
    {
	dynamic (newVector (new double[] {uLeftWheel, uRightWheel}));
    }

    //! Normalizes an angle to the interval [-PI..+PI]
    double normalizedAngle (double angle)
    {
	// ADD CODE HERE
	return 0;	
    }    

    
    //! Integrate a single measurement of beacon \c beaconId
    //! into the estimation
    //! If several beacons are observed at the same time the
    //! routine is called several times. When debugging, it
    //! might be helpful to only call \c measurement once to
    //! facilitate finding bugs.
    void measurement (Matrix z, int beaconId)
    {
	// ADD CODE HERE
	
	if (covX!=null) {	    
	    System.out.println ("Covariance after measurement");
	    covX.print(8, 4);	
	    checkConsistency ();	
	}	
    }

    
    //! Wrapper for calling \c measurement with one double
    //! namely the actual angle observed to beacon \c beaconId
    void measurement (double angle, int beaconId)
    {
	measurement (newVector (new double[] {angle}), beaconId);	
    }


    //*** The code below is not part of the Kalman Filter algorithm.
    
    //! muX and covX in a way that can be visualized by the GUI
    class Estimate
    {
	//! Position and orientation of the cart
	double x, y, phi;
	//! rms phi error and rms XY error
	//! Derived from covariance
	double sigmaPhi, sigmaXY;
	//! Largest and smallest uncertainty in the 2*2
	//! covariance matrix of (X,Y) 
	double sigmaXYLarge, sigmaXYSmall;	
	//! Angle of the large (X,Y) uncertainty
	double sigmaXYAngle;
    };

    //! Converts muX and covX into a format that can be used
    //! by the GUI
    Estimate computeEstimateForGUI ()
    {
	if (muX==null) return null;	
	Estimate es = new Estimate();
	es.x        = muX.get (0, 0);
	es.y        = muX.get (1, 0);
	es.phi      = muX.get (2, 0);

	if (covX==null) return es;	
	es.sigmaPhi = Math.sqrt (covX.get(2,2));

	// Find the 2 eigenvalues/vectors of the XY subpart of covX
	double ll0, ll1;
	es.sigmaXYAngle = Math.atan2(-2*covX.get(1,0), covX.get(1,1)-covX.get(0,0))/2;
	if (Double.isNaN (es.sigmaXYAngle)) es.sigmaXYAngle=0;
	double c = Math.cos(es.sigmaXYAngle);
	double s = Math.sin(es.sigmaXYAngle);
	double c2 = c*c, s2=s*s, cs = c*s;
    
	ll0 = c2*covX.get(0,0)+2*cs*covX.get(1,0)+s2*covX.get(1,1);
	ll1 = s2*covX.get(0,0)-2*cs*covX.get(1,0)+c2*covX.get(1,1);
	if (ll0<ll1) {
	    if (es.sigmaXYAngle>0) es.sigmaXYAngle-=Math.PI/2;
	    else es.sigmaXYAngle+=Math.PI/2;
	    es.sigmaXYLarge = Math.sqrt(ll1);
	    es.sigmaXYSmall = Math.sqrt(ll0);
	}
	else {
	    es.sigmaXYLarge = Math.sqrt(ll0);
	    es.sigmaXYSmall = Math.sqrt(ll1);
	}
	return es; 
    }    

    //! Checks, whether covX is symmetric and positive definit
    //! If this routine fails, than usually one of your matrices
    //! or matrix formulas is broken.
    void checkConsistency () 
    {
	// Test matrix format
	int n = muX.getRowDimension();	
	if (muX.getColumnDimension()!=1) throw new Error ("muX is no vector");
	if (covX.getColumnDimension()!=n) throw new Error ("covX is not square");	

	Matrix cov2 = covX.copy();	
	// Test symmetry
	for (int i=0;i<n;i++) for (int j=0;j<n;j++) {
	    cov2.set (i, j, (covX.get(i,j)+covX.get(j,i))/2);	
	    double delta = Math.abs(covX.get(i,j)-covX.get(j,i));
	    if (delta > 0.01) throw new Error ("Matrix is not symmetric");	    
	}	
	      
	// Assert positiv definiteness	
	if (!cov2.chol().isSPD()) throw new Error ("covX is not positive definite");
    }    


    Matrix sampleU = new Matrix (new double[][] {{1},{1.3}});    				 						 
					 

    //! Allows to checks whether dynamicModelJacobianU is correct by numerically
    //! differentiating dynamicModel
    void checkDynamicModelJacobianX ()
    {
	Matrix JAnalytical = dynamicModelJacobianX (muX, sampleU);	
	System.out.println ("Analytical Jacobian df(x,u)/dx");
	JAnalytical.print (8, 6);	

	double eps = 0.001;	
	Matrix JNum = new Matrix (JAnalytical.getRowDimension(), JAnalytical.getColumnDimension());
	for (int j=0; j<JNum.getColumnDimension(); j++) {
	    Matrix xPlus, xMinus;
	    xPlus = muX.copy();
	    xPlus.set (j,0, muX.get(j,0)+eps);
	    xPlus = dynamicModel (xPlus, sampleU);
	    
	    xMinus = muX.copy();
	    xMinus.set (j,0, muX.get(j,0)-eps);
	    xMinus = dynamicModel (xMinus, sampleU);
	    for (int i=0; i<JNum.getRowDimension(); i++) JNum.set(i, j, (xPlus.get(i,0)-xMinus.get(i,0))/(2*eps));	    
	}
	System.out.println ("Numerical Jacobian df(x,u)/dx");
	JNum.print (8, 6);	
    }    

    //! Same for Jacobian of the dynamic model with respect to X
    void checkDynamicModelJacobianU ()
    {
	Matrix JAnalytical = dynamicModelJacobianU (muX, sampleU);	
	System.out.println ("Analytical Jacobian df(x,u)/du");
	JAnalytical.print (8, 6);	

	double eps = 0.001;	
	Matrix JNum = new Matrix (JAnalytical.getRowDimension(), JAnalytical.getColumnDimension());
	for (int j=0; j<JNum.getColumnDimension(); j++) {
	    Matrix uPlus, uMinus;
	    uPlus = sampleU.copy();
	    uPlus.set (j,0, sampleU.get(j,0)+eps);
	    uPlus = dynamicModel (muX, uPlus);
	    
	    uMinus = sampleU.copy();
	    uMinus.set (j,0, sampleU.get(j,0)-eps);
	    uMinus = dynamicModel (muX, uMinus);
	    for (int i=0; i<JNum.getRowDimension(); i++) JNum.set(i, j, (uPlus.get(i,0)-uMinus.get(i,0))/(2*eps));	    
	}
	System.out.println ("Numerical Jacobian df(x,u)/du");
	JNum.print (8, 6);	
    }    

    //! Allows to checks whether dynamicModelJacobianU is correct by numerically
    //! differentiating dynamicModel
    void checkMeasurementModelJacobian ()
    {
	Matrix JAnalytical = measurementModelJacobian (muX, 0);	
	System.out.println ("Analytical Jacobian dg(x)/dx");
	JAnalytical.print (8, 6);	

	double eps = 0.001;	
	Matrix JNum = new Matrix (JAnalytical.getRowDimension(), JAnalytical.getColumnDimension());
	for (int j=0; j<JNum.getColumnDimension(); j++) {
	    Matrix xPlus, xMinus;
	    xPlus = muX.copy();
	    xPlus.set (j,0, muX.get(j,0)+eps);
	    xPlus = measurementModel (xPlus, 0);
	    
	    xMinus = muX.copy();
	    xMinus.set (j,0, muX.get(j,0)-eps);
	    xMinus = measurementModel (xMinus, 0);
	    for (int i=0; i<JNum.getRowDimension(); i++) JNum.set(i, j, (xPlus.get(i,0)-xMinus.get(i,0))/(2*eps));	    
	}
	System.out.println ("Numerical Jacobian dg(x)/dx");
	JNum.print (8, 6);	
    }    
};

