import java.awt.*;
import java.awt.event.*;
import java.util.*;
import java.text.*;


public class CoalCart extends Frame implements KeyListener
{
    static final long serialVersionUID = 1;
    
    //! True position of the cart
    double x;

    //! Last measurement
    /*! NaN if there is no measurement. */
    double z;

    //! Length of the track
    double length = 100;   

    //! Height of the cart on the ground
    double cartHeight = 2;

    //! Length of the cart
    double cartLength = 6;

    //! Height of the sensor
    double sensorHeight = 10;


    //! Height of the error plot
    int errorPlotHeight = 100;    

    //! dynamic noise used for simulation
    double dynamicNoiseSigma = 0.3;

    //! measurement noise used for the simulation
    double measurementNoiseSigma = 1*Math.PI/180;

    int ctr=0;        
    class Interval 
    {
        Interval (double value, double sigma)
        {
            this.value = value;
            this.sigma = sigma;
        }        
        double value;
        double sigma;
    };
    
    
    Interval[] errorLog = new Interval[0];    
    
    
    

    //! The filter tracking the cart
    KalmanFilter1D ukf = new KalmanFilter1D();

    //! The random number generator used for simulation
    Random rnd = new Random ();

    CheckboxMenuItem miAutoMeasure;    

    class CoalCartComponent extends Component
    {    
        static final long serialVersionUID = 1;

        double offsetX = 100;    
        double offsetY = 20;

        //! Pixel per meter for painting
        double scale = 10;

        CoalCartComponent ()
        {
            super (); 
	    setBackground (Color.gray);	    
        }
        

        void lineScale (Graphics g, double x1, double y1, double x2, double y2)
        {
            g.drawLine ((int) (x1*scale+offsetX), (int) getHeight()-10-(int)(y1*scale+offsetY), 
                        (int) (x2*scale+offsetX), (int) getHeight()-10-(int)(y2*scale+offsetY));
        }
        
        
        void fillRectScale (Graphics g, double x1, double y1, double x2, double y2)
        {
            g.fillRect ((int) (x1*scale+offsetX), (int) getHeight()-10-(int)(y2*scale+offsetY), 
                        (int) ((x2-x1)*scale), (int)((y2-y1)*scale));
        }
        
        
        void rectScale (Graphics g, double x1, double y1, double x2, double y2)
        {
            int xx1 = (int) (x1*scale+offsetX);
            int yy1 = (int) getHeight()-10-(int)(y1*scale+offsetY);
            int xx2 = (int) (x2*scale+offsetX);
            int yy2 = (int) (int) getHeight()-10-(int)(y2*scale+offsetY);
            g.drawLine (xx1, yy1, xx2, yy1);
            g.drawLine (xx2, yy1, xx2, yy2);
            g.drawLine (xx2, yy2, xx1, yy2);
            g.drawLine (xx1, yy2, xx1, yy1);        
        }
    
    

        public void paint (Graphics g)
        {
	    g.setColor(getBackground());
	    g.fillRect(0, 0, getWidth(), getHeight());

            g.setColor (Color.black);
            lineScale     (g, 0, 0, length, 0);
            lineScale     (g, 0, 0, 0, sensorHeight);
            fillRectScale (g, x-cartLength/2, 0, x+cartLength/2, cartHeight);
            lineScale     (g, x, cartHeight+1, x, -1);
            
            if (!Double.isNaN(z)) {                
                g.setColor (Color.red);
                lineScale (g, 0, sensorHeight, (sensorHeight-cartHeight)/Math.tan(z), cartHeight);
            }            
            
            g.setColor (Color.green);
            if (ukf!=null) {
                rectScale (g, ukf.muX-cartLength/2, 0, ukf.muX+cartLength/2, cartHeight);
                lineScale (g, ukf.muX-Math.sqrt(ukf.sigmaX2), -0.5, ukf.muX+Math.sqrt(ukf.sigmaX2), -0.5);
            }            
        }        

        public Dimension getPreferredSize() 
        {
            return new Dimension ((int) (scale*length+2*offsetX), (int) (scale*(sensorHeight)+44+offsetY));
        }
    }


    class ErrorLogComponent extends Component 
    {
        static final long serialVersionUID = 1;

        double scale = 10;
        double range = 10;        
        double unit = 2;        

        int offsetX = 100;        

        ErrorLogComponent ()
        {
            super ();            
        }

        public void paint (Graphics g)
        {
            g.setColor (Color.black);
            g.drawLine (0, getHeight()/2, getWidth(), getHeight()/2);
            g.drawLine (offsetX, 0, offsetX, getHeight());            
            int lastY=0;            
            int start =0;
            int w = getWidth()-offsetX-5;            
            if (errorLog.length>w) start = errorLog.length-w;            
            if (errorLog!=null) for (int i=start; i<errorLog.length; i++) {
                g.setColor (Color.red);
                int y = (int) (getHeight()/2 + errorLog[i].value*scale);
                if (i>0) g.drawLine (i-1+offsetX-start, lastY, i+offsetX-start, y);
                g.setColor (Color.green);
                g.drawLine (i+offsetX-start, (int) (getHeight()/2 + errorLog[i].sigma*scale),
                            i+offsetX-start, (int) (getHeight()/2 - errorLog[i].sigma*scale));                
                lastY = y;                
            }
            DecimalFormat format = new DecimalFormat("+00.0");
            for (double y=unit; y<=range; y+=unit) {
                int yy = (int) (getHeight()/2 + y*scale);                
                g.setColor (Color.cyan);            
                g.drawLine (0, yy, getWidth(), yy);
                g.setColor (Color.black);            
                g.drawString (format.format(y), offsetX-50, yy+5);                
                yy = (int) (getHeight()/2 - y*scale);                
                g.setColor (Color.cyan);            
                g.drawLine (0, yy, getWidth(), yy);
                g.setColor (Color.black);            
                g.drawString (format.format(-y), offsetX-50, yy+5);
            }            
        }        

        public Dimension getPreferredSize() 
        {
            return new Dimension ((int) (length+20), (int) (scale*2*range)+20);
        }
    }
    

   
    //! The upper component that shows the cart and the bearing sensor
    CoalCartComponent ccc;

    //! The low component that shows the error plot
    ErrorLogComponent elc;    

    //! Label for showing text messages
    Label statusText;    

    //! Whether to append the next log
    boolean doAppendNextLogState=false;    

    //! Whether to make a measurement after each update step
    boolean autoMeasure = true;    


    void logState (String operation, double value)
    {
        if (!doAppendNextLogState) {
            statusText.setText(null);
            doAppendNextLogState = true;
        }
        NumberFormat format = new DecimalFormat("0.00");
        String st = ctr + " " + operation + format.format(value) + " State "+format.format(ukf.muX) + " +/- "+format.format(Math.sqrt(ukf.sigmaX2));
        String st2 = statusText.getText();
        if (st2!=null) st = st2 + "; " + st;        
        statusText.setText (st);        
        System.out.println (st);
        if (ukf!=null && errorLog!=null) {
            int n = errorLog.length;            
            Interval[] nLog = new Interval[n+1];
            for (int i=0;i<n;i++) nLog[i] = errorLog[i];
            nLog[n] = new Interval (ukf.muX-x, Math.sqrt(ukf.sigmaX2));
            errorLog = nLog;            
        }
        ctr++;
    }
    

    void move (double deltaX)
    {
        x += deltaX;
        if (ukf!=null) {
            double u = deltaX+Math.sqrt(Math.abs(deltaX))*dynamicNoiseSigma*rnd.nextGaussian();        
            ukf.dynamic (u);        
            logState("Moved ", u);
        }
        z = Double.NaN;        
        repaint ();        
    }

    void measure ()
    {
        z = Math.atan ((sensorHeight-cartHeight)/x) + measurementNoiseSigma*rnd.nextGaussian();        
        if (ukf!=null) {
            ukf.measurement (z);        
            logState ("Measured ", z);
        }        
        repaint ();        
    }

    CoalCart ()
    {
        super ("Tracking a Coal Cart with an Extended Kalman Filter");

	MenuBar menuBar = new MenuBar();
	setMenuBar (menuBar);
	Menu optionsMenu = new Menu("Options");
	menuBar.add (optionsMenu);

        miAutoMeasure = new CheckboxMenuItem("Auto measure", autoMeasure);
        miAutoMeasure.addActionListener (new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    CheckboxMenuItem cmi = (CheckboxMenuItem) e.getSource();
                    autoMeasure = cmi.getState();
                    System.out.println ("Switched to auto "+autoMeasure);
                }
            });
        optionsMenu.add (miAutoMeasure);
        
	ccc = new CoalCartComponent();	
	add (ccc, "North");

        elc = new ErrorLogComponent();
        add (elc, "Center");        

    	statusText = new Label ();
	add (statusText, "South");	

        x = length/2 + length/10*rnd.nextGaussian();
        z = Double.NaN;
        addKeyListener(this);
        
        pack();        
        setVisible(true);        
    }    

    public void keyTyped(KeyEvent evt) {
    }
    
    public void keyReleased(KeyEvent evt) {
    }
    
    public void keyPressed(KeyEvent evt) {
      int key = evt.getKeyCode();  // keyboard code for the key that was pressed
      
      if (key == KeyEvent.VK_LEFT) {
          if (x>=1) move (-1);
          if (autoMeasure) measure();
          doAppendNextLogState = false;          
      }      
      else if (key == KeyEvent.VK_RIGHT) {
          if (x<=length-1) move (1); 
          if (autoMeasure) measure();
          doAppendNextLogState = false;          
      }
      else if (key== KeyEvent.VK_SPACE) {
          measure ();      
          doAppendNextLogState = false;          
      }
      else if (key== KeyEvent.VK_A) {
	  autoMeasure = !autoMeasure;
	  miAutoMeasure.setState (autoMeasure);	  
          System.out.println ("Switched to auto "+autoMeasure);
      }      
    }
    

    public static void main(String[] args) 
    {
	CoalCart cc = new CoalCart();
        cc.addWindowListener(new java.awt.event.WindowAdapter() {
                public void windowClosing(WindowEvent winEvt) {
                    System.exit(0); 
                }
            });
    }
    
};

