//
// boundbox.cc  --  CollageSystem
//
// BoundingBox
//
// Author: N. Schwabe
// Date of creation : 10/94
// Last modification: 11/94
//
// University of Bremen, Germany
//

#include "boundbox.h"
#include <values.h>


// ------ BoundingBox ---------------------------------------------------------

BoundingBox::BoundingBox (uint dim) : MinMaxPairList()
{
	MinMaxPair pair(MAXDOUBLE,MINDOUBLE);
	for (int i=0; i < dim; i++)
		Append (pair);
}

void BoundingBox::Calc (const Point &p)
{
	if (p.GetDim() != Length()) {
		static Boolean warned = False;
		if (! warned)
		{
			cerr << "CollageSystem: Warning: No bounding boxes are calculated in this dimension mode.\n";
			cerr << "               This warning will not be repeated, even if the grammar changes.\n";
			warned = True;
		}
		return;
	}
	int d=0;
	MinMaxPair *cur = GetAt(0);
	while (cur) {
		Coordinate c = p.Get(d++);
		if (c < cur->min)
			cur->min = c;
		if (c > cur->max)
			cur->max = c;
		cur = GetNext();
	}
}



// ------- PictGroupSet -------------------------------------------------------

PictGroupSet::~PictGroupSet ()
{
	// Traverse the list and free all elements:

	PictInterval *p = list.GetAt (0);
	while (p) {
		list.Delete ();
		delete p;
		p = list.GetAt (0);
	}
}

void PictGroupSet::NewInterval (int from, int to)
{
	PictInterval *n = new PictInterval (from, to);
	list.Append (*n);
}

Boolean PictGroupSet::IsIn (int pic)
{
	PictInterval *p = list.GetAt (0);
	while (p) {
		if (p->IsIn (pic)) return True;
		p = list.GetNext ();
	}
	return False;
}



// ------- PictSet ------------------------------------------------------------

PictSet::~PictSet ()
{
	// Traverse the list and free all elements:

	PictGroupSet **p = list.GetAt (0);
	while (p) {
		list.Delete ();
		delete *p;
		p = list.GetAt (0);
	}
}

void PictSet::NewGroup (char group)
{
	current = new PictGroupSet (group);
	list.Append (current);
}

void PictSet::NewInterval (int from, int to)
{
	if (! current) {
		cerr << "PictSet::NewInterval: No current group." << endl;
		return;
	}
	current->NewInterval (from, to);
}

Boolean PictSet::IsIn (char g, int pic, Boolean isLast)
{
	PictGroupSet **p = list.GetAt (0);
	if (! p) return True; // empty set =^= set containg all members
	while (p) {
		if ((*p)->Group() == g)
		{
			if (   (*p)->IsIn (pic)
			    || (isLast && (*p)->IsIn (BBLastPic)))
				return True;
		}
		p = list.GetNext ();
	}
	return False;
}



// ------- BoundingBoxCalculator ----------------------------------------------

BoundingBoxCalculator::BoundingBoxCalculator ()
{
	options = 0;
}

void BoundingBoxCalculator::SetOptions (OptionBBoxList& o)
{
	options = &o;
}

void BoundingBoxCalculator::SetPicture 
	(char group, int picture, BoundingBox& bb)
{
	PictBoundingBox *n = new PictBoundingBox (bb.Dim());
	n->group = group;
	n->picture = picture;
	n->bbOrg = bb;
	boxes.Append (*n);
}

void BoundingBoxCalculator::MinOrMax
	(	OptionBBox  *option,
		BoundingBox *result,
		Boolean     minim,
		Boolean     swapLastCmp)
{
	int actPos = boxes.GetIndex(); // save actual list position
	Coordinate llx=0, lly=0, urx=0, ury=0; // init just to avoid compiler warnings
	Coordinate llx_, lly_, urx_, ury_;
	Boolean first = True, isLast;

	// Implementation note: Conceptually, we would have to iterate
	// through the 'list' of operand pictures. Because there is no
	// explicit list containing these pictures, we iterate through
	// the list of all created pictures instead and determine for
	// each picture if it is in the set of operand intervalls of this
	// bounding box option.

	PictGroupSet **lastOperandSet = option->operands.List().GetLast();

	PictBoundingBox *b = boxes.GetAt(0), *next = 0;
	while (b) 
	{
		isLast = False;
		next = boxes.GetNext ();

		if (option->operands.IsIn (b->group, b->picture, 
		                           (! next) || next->group != b->group))
			// picture is operand
		{
			if (swapLastCmp) 
			{
				// Determine whether b is the last picture described by
				// the last operand set. This implementation of the test
				// assumes that pictures of one group are created 
				// sequentially. Therefore, picture b is the last picture
				// of the last operand set if
				// - it belongs to lastOperandSet
				// - and the next picture doesn't belong to lastOperandSet
				//   or does not exist.

				if (lastOperandSet) 
				{
					isLast = (   (*lastOperandSet)->Group() == b->group 
					    && (*lastOperandSet)->IsIn (b->picture) 
					    && ((! next) || (*lastOperandSet)->Group() != next->group  
					                 || ! (*lastOperandSet)->IsIn (next->picture)) );
				}
				else
				{
					// no last operand set => empty operand set =^= full set
					isLast = ! next;
				}
			}
		
			llx_ = b->bbOrg.GetAt(0)->min;
			urx_ = b->bbOrg.GetAt(0)->max;
			lly_ = b->bbOrg.GetAt(1)->min;
			ury_ = b->bbOrg.GetAt(1)->max;

			if (first) 
			{
				llx = llx_;
				lly = lly_;
				urx = urx_;
				ury = ury_;
				first = False;
			}
			else 
			{
				if (   (minim && ((! swapLastCmp) || (! isLast)))
				    || ((! minim) && swapLastCmp && isLast) ) 
				{
					if (llx_ > llx) llx = llx_;
					if (lly_ > lly) lly = lly_;
					if (urx_ < urx) urx = urx_;
					if (ury_ < ury) ury = ury_;
				}
				else 
				{
					if (llx_ < llx) llx = llx_;
					if (lly_ < lly) lly = lly_;
					if (urx_ > urx) urx = urx_;
					if (ury_ > ury) ury = ury_;
				}
			}
			
		}
		b = next;
	}

	if (first)
		cerr << "BoundingBoxCalculator::Min : No operand pictures." << endl;
	else 
	{
		result->GetAt(0)->min = llx;
		result->GetAt(0)->max = urx;
		result->GetAt(1)->min = lly;
		result->GetAt(1)->max = ury;
	}

	// restore last list position:
	if (actPos >= 0) 
		(void) boxes.GetAt (actPos);
}


void BoundingBoxCalculator::Avg
	(	OptionBBox  *option,
		BoundingBox *result)
{
	int actPos = boxes.GetIndex(); // save actual list position
	Coordinate llx=0, lly=0, urx=0, ury=0;
	int count=0;

	PictBoundingBox *b = boxes.GetAt(0), *next = 0;
	while (b) {
		
		next = boxes.GetNext ();

		if (option->operands.IsIn (b->group, b->picture, 
		                           (! next) || next->group != b->group))
			// picture is operand
		{
			count++;
			llx += b->bbOrg.GetAt(0)->min;
			urx += b->bbOrg.GetAt(0)->max;
			lly += b->bbOrg.GetAt(1)->min;
			ury += b->bbOrg.GetAt(1)->max;
		}
		b = next;
	}

	if (count==0)
		cerr << "BoundingBoxCalculator::Avg : No operand pictures." << endl;
	else {
		result->GetAt(0)->min = llx/count;
		result->GetAt(0)->max = urx/count;
		result->GetAt(1)->min = lly/count;
		result->GetAt(1)->max = ury/count;
	}

	// restore last list position:
	if (actPos >= 0) 
		(void) boxes.GetAt (actPos);
}


OptionBBox *BoundingBoxCalculator::SelectedOption 
	(char group, int picture, Boolean isLast, Boolean didX, Boolean didY)

// Returns the (first matching) option in which group/picture is selected. 

{
	// Iterate through the options list:

	OptionBBox **opt = options->GetAt (0);
	while (opt)
	{
		if ((*opt)->selected.IsIn (group, picture, isLast)) 
		{
			if ((*opt)->option == OptBBox && didX == False && didY == False)
				return *opt;
			else if ((*opt)->option == OptXBBox && didX == False)
				return *opt;
			else if ((*opt)->option == OptYBBox && didY == False)
				return *opt;
		}
		opt = options->GetNext ();
	}
	return 0;
}



Boolean BoundingBoxCalculator::Calc 
	(	char group, int picture, BoundingBox *result, Boolean *calc)
{
	int actPos = boxes.GetIndex ();
	PictBoundingBox *b = boxes.GetAt (0), *next;
	OptionBBox *selOption;
	Boolean didX = False, didY = False;

	while (b && (b->group != group || b->picture != picture))
		b = boxes.GetNext ();
	if (! b) return False;
	next = boxes.GetNext ();
	if (actPos >= 0) 
		(void) boxes.GetAt (actPos);

	*result = b->bbOrg; // default

	while ((selOption = SelectedOption (group, picture, 
	                                    (! next) || next->group != b->group,
	                                    didX, didY)) != 0)
	{
		BoundingBox temp(2);

		if (calc) *calc = True;
		switch (selOption->operation) {
		case BMin:
			MinOrMax (selOption, &temp, True, False);
			break;
		case BMax:
			MinOrMax (selOption, &temp, False, False);
			break;
		case BMinMax:
			MinOrMax (selOption, &temp, True, True);
			break;
		case BMaxMin:
			MinOrMax (selOption, &temp, False, True);
			break;
		case BAvg:
			Avg (selOption, &temp);
			break;
		default:
			cerr << "BoundingBoxCalculator::Calc: Unknown operation." << endl;
			return False;
		}
		
		if (selOption->option == OptBBox)
		{
			*result = temp;
			didX = True;
			didY = True;
		}
		else if (selOption->option == OptXBBox) 
		{
			result->GetAt(0)->min = temp.GetAt(0)->min;
			result->GetAt(0)->max = temp.GetAt(0)->max;
			didX = True;
		}
		else if (selOption->option == OptYBBox) 
		{
			result->GetAt(1)->min = temp.GetAt(1)->min;
			result->GetAt(1)->max = temp.GetAt(1)->max;
			didY = True;
		}
		else
			cerr << "BoundingBoxCalculator::Calc: Unknown option." << endl;
	}

	if (didX == False && didY == False)
	{
		// return default:
		if (calc) *calc = False;
		*result = b->bbOrg;
	}
	return True;
}


