#ifndef __BOUNDBOX_H__
#define __BOUNDBOX_H__
//
// boundbox.h  --  CollageTwo
//
// Bounding-Box
//
// Author               : Nils Schwabe
// Date of creation     : Oct 94
// Last modification    : 16 Nov 95 ns
//
// University of Bremen, Germany
//

#ifndef __BASIC_H__
#include "basic.h"
#endif

#ifndef __POINT_H__
#include "point.h"
#endif


// ------ BoundingBox ---------------------------------------------------------

	// A bounding box specifies a cube in n-dimensional euclidean space
	// that contains a given set of points, i.e. for each dimension, it
	// specifies the minimum and the maximum of all coordinate values of
	// all points of the set.

class MinMaxPair 
{
public:
	MinMaxPair (Coordinate mi, Coordinate ma) {min=mi; max=ma;}
	Coordinate min, max;
};

typedef SList<MinMaxPair> MinMaxPairList;

class BoundingBox : public MinMaxPairList
{
public:
	BoundingBox (uint dim);
	BoundingBox& operator= (BoundingBox &ass)
	{
		MinMaxPairList::operator= (MinMaxPairList(ass)); 
		return *this;
	}

	void Calc (const Point &p);

	// Checks if p is within the bounding box and extends the box if not.
	// Each MinMaxPair gives the minimum and maximum Coordinate value
	// for each dimension. (Length of list = number of dimensions)

	uint Dim ()                                          {return Length();}

	// Returns the dimension of the bounding box.
};


// ------- Bounding box calculation base --------------------------------------

#define BBLastPic 2147483647

// PictInterval: An interval of picture numbers (from n, to m):
// 
// Possible values for n and m:
// n = m, n,m >= 0       : Only the single picture n
// n < m, n,m >= 0       : From picture n upto picture m (inclusive)
// n >= 0, m = BBLastPic : From picture n upto the last picture (inclusive)
 
class PictInterval
{
public:
	PictInterval (int from, int to)                   {from_=from; to_=to;}

	// Constructs the picture number interval with values 'from' and 'to'.

	int From ()                                             {return from_;}
	int To ()                                                 {return to_;}

	// Returns the 'from' resp. 'to' value of the picture number interval.

	Boolean IsIn (int p)      
	                                       {return p >= from_ && p <= to_;}

	// Returns True if picture number 'p' is within the interval.

private:
	int from_;                       // 0..n 
	int to_;                         // 0..n or BBLastPic
};

typedef SList<PictInterval> PictIntervalList;


// PictGroupSet: A set of picture number intervals that is assigned to one
//               group of pictures. Groups are currently 'G', 'D' or 'B':
//               G = Grammar pictures, D = Derivation pictures, B = Bounding
//               box pseudo pictures. 

class PictGroupSet
{
public:
	PictGroupSet (char group)                               {group_=group;}

	// Constructs a picture group set for a group number 'group'.
	// Initially, the picture group set contains no picture number
	// intervals.

	~PictGroupSet();

	int Group ()                                           {return group_;}

	// Returns the group of this picture group set.

	void NewInterval (int from, int to);

	// Creates a new interval for this group.

	Boolean IsIn (int p);  

	// Returns True if picture number 'p' is in the set of intervals. 

private:
	char group_; 	
	PictIntervalList list;
};

typedef SList<PictGroupSet *> PictGroupSetList;


// PictSet: A set of picture number intervals assigned to different groups.
//          This type of set has the special property of handling an empty
//	    set as equivalent to a set that contains all possible members.

class PictSet
{
public:
	PictSet ()                                               {current = 0;}

	// Constructs an empty picture number interval set, that is,
	// by convention, a set that contains ALL pictures.

	~PictSet ();

	void NewGroup (char group);

	// Creates new picture number interval set for that group and makes
	// the group current. 

	void NewInterval (int from, int to);

	// Creates a new interval in the current group.

	Boolean IsIn (char g, int p, Boolean isLast);

	// Returns True if the picture 'p' of group 'g' is in the
	// set of picture number intervals. True is also returned if the
	// set is empty (see note above).

	PictGroupSetList& List()                                 {return list;} 

private:
	PictGroupSetList list;
	PictGroupSet *current;
};


enum BBoxOption    // (2-D) bounding box option
{
	OptBBox,   // affects both coordinates
	OptXBBox,  // affects only x-coordinates
	OptYBBox   // affects only y-coordinates
};

enum BBoxOperation // bounding box operation
{
	BMin,      // calculate the minimum of all bounding boxes
	BMax,      // calculate the maximum of all bounding boxes
	BMinMax,   // max ( min (1,..,n-1), n )
	BMaxMin,   // min ( max (1,..,n-1), n )
	BAvg       // calculate the average of all bounding boxes
};

class OptionBBox   // bounding box option
{
public:
	BBoxOption    option;        // what coordinates are affected
	BBoxOperation operation;     // type of calculation
	PictSet       selected;      // these pictures are modified...
	PictSet       operands;      // ...by these pictures
};

typedef SList<OptionBBox *> OptionBBoxList; 
	// A list of bounding box options



// ------- Bounding box calculation -------------------------------------------

	// Bounding boxes are used in three phases:
	//
	// 1. Definition of the bounding box options. This results in a
	//    OptionBBoxList (see above).
	//
	// 2. Generation of pictures. Each picture has an original bounding
	//    box that is determined by the parts that it contains.
	//
	// 3. Calculation of the bounding boxes that are actually output
	//    based on the information of phase 1 and 2.
	//

class PictBoundingBox
{
public:
	PictBoundingBox (uint dim)           : bbOrg(dim) {group=0; picture=0;}

	// (Default copy and assignment semantics is ok)

	char        group;    // group of the picture
	int         picture;  // 0..n (NOT BBLastPic here)
 	BoundingBox bbOrg;    // original bounding box for this picture
};

typedef SList<PictBoundingBox> PictBoundingBoxList;


class BoundingBoxCalculator 
{
public:
	BoundingBoxCalculator ();

	// Constructs an (inoperable) bounding box calculator.

	void SetOptions (OptionBBoxList& options);

	// Initializes the bounding box calculator with the options information.

	void SetPicture (char group, int picture, BoundingBox& bb);

	// Sets the (seperately calculated) bounding box information for
	// the picture. 
	// Note: This should be called for _every_ created picture. 

	Boolean Calc (char group, int pic, BoundingBox *bb, Boolean *calc=0);

	// Calculates the bounding box 'bb' for the picture of the given group
	// using the internal options and bounding box information.
	// Note: In this version, the result will be either the original
	// bounding box passed in SetPicture() or a calculated bounding box
	// that is currently identical for all affected pictures.
	// Sets 'calc', if specified, to True if the box is a calculated
	// bounding box. This information may be used to optimize the 
	// 'patching' (or whatever) of existing files containing bounding 
	// box information.
	// Returns False if there's no such group/picture entry.

	PictBoundingBoxList& Boxes ()                           {return boxes;}

private:
	OptionBBoxList *options;
	PictBoundingBoxList boxes;
	OptionBBox *SelectedOption (char, int, Boolean, Boolean, Boolean);
	void MinOrMax (OptionBBox *, BoundingBox *, Boolean, Boolean);
	void Avg (OptionBBox *option, BoundingBox *result);

};

#endif /* __BOUNDBOX_H__ */

