//
// cgi_p.cc  --  CollageVR
//
// System dependent implementation class for CGI (X/XView version)
//
// Author: N. Schwabe
// Date of creation : 23.04.94
// Last modification: 09 Dec 96
//
// University of Bremen, Germany
//

// ------- Includes -----------------------------------------------------------

#include <string.h>
#include <math.h>
#include <iostream.h>
#include "cgi_p.h"
#include "vrout.h"
#ifndef __TCLCOMMANDER_H__
#include "TclCommander.h"
#endif


// ------- Module data --------------------------------------------------------

#define MAX_POLY_POINTS 40

#define WHITE 0
#define BLACK 1 

static XPoint xp[MAX_POLY_POINTS];

const panelH = 25;

// Menu constants:
static const char strMenuFile[] = "File";
static const char strMenuView[] = "View";

static const char strFileLoad[] = "Load...";
static const char strFileAgain[] = "Reload";
static const char strFileEdit[] = "Edit";
static const char strFileCommander[] = "Commander...";
static const char strFileQuit[] = "Quit";
static const int itemEdit = 4; // caution: redundant with xv_create call
static const int itemReload = 3; // dito

static const char strViewPrevPage[] =    "Previous";
static const char strViewNextPage[] =    "Next";
static const char strViewFirstPage[] =   "First";
static const char strViewLastPage[] =    "Last";
static const char strViewZoomIn[] =      "Zoom in  ";
static const char strViewZoomOut[] =     "Zoom out ";
static const char strViewScrollLeft[] =  "  <---   ";
static const char strViewScrollRight[] = "  --->   ";
static const char strViewScrollUp[] =    "   Up    ";
static const char strViewScrollDown[] =  "  Down   ";
static const char strViewReset[] =       "  Reset  ";
static const char strViewQuit[] =        "  Quit   ";
static const int itemPrevPage = 2;
static const int itemNextPage = 3; // caution: redundant with xv_create call
static const char strButPrevPage[] = "Prev";
static const char strButNextPage[] = "Next";	
static const char strButSnapshot[] = "Snapshot";


// ------- CgiPrivate ---------------------------------------------------------

// xvcpp.h macro implementations:
MAKE_MENU_STUFF (CgiPrivate)
MAKE_CANVAS_STUFF (CgiPrivate)
MAKE_BUTTON_STUFF (CgiPrivate)
MAKE_WIN_EVENT_STUFF (CgiPrivate)
MAKE_FILE_CHOOSER_STUFF (CgiPrivate)
MAKE_INTERPOSE_DESTROY_STUFF (CgiPrivate)


// ------- RGBCache (local class) ---------------------------------------------

CgiPrivate::RGBCache::RGBCache ()
{
	dpy = NULL;
	trueColor = False;	
}	

void CgiPrivate::RGBCache::Reset (Display *d)
{
	dpy = d;
	while (cache.GetAt(0)) cache.Delete ();
}

unsigned long CgiPrivate::RGBCache::GetIndex 
	(	unsigned char red,
		unsigned char green,
		unsigned char blue)
{
  if (! trueColor)
  {
	// Try to get color from cache:
	Entry *entry = cache.GetAt (0);
	while (entry)
	{
		if (entry->r == red && entry->g == green && entry->b == blue)
			return entry->colindex;
		entry = cache.GetNext ();
	}

	// Not in cache, get color from X server:
	XColor coldef;
	memset (&coldef, 0, sizeof(coldef));
	coldef.red = int(red)*256;
	coldef.green = int(green)*256;
	coldef.blue = int(blue)*256;
	if (XAllocColor (dpy, DefaultColormap(dpy,0), &coldef) == 0)
	{
		cout << "CollageSystem: Couldn't get color " 
		<< int(red) << " " << int(green) << " " << int(blue) << endl;
		cout << "               Restart system to get better color display" << endl;
	}

	// Cache new color (even if XAllocColor failed)
	Entry newEntry;
	newEntry.r = red;
	newEntry.g = green;
	newEntry.b = blue;
	newEntry.colindex = coldef.pixel;
	cache.Insert (newEntry);

	return newEntry.colindex;
  }
  else
  {
	// HACK:  (how do I get these indices correctly? why are 1,2 and 254 corrupted?)
	if (red == 1) red = 0;
	else if (red == 2) red = 3;
	else if (red == 255) red = 254;

	if (green == 1) green = 0;
	else if (green == 2) green = 3;
	else if (green == 255) green = 254;

	if (blue == 1) blue = 0;
	else if (blue == 2) blue = 3;
	else if (blue == 255) blue = 254;

	return int(red)+int(green)*256+int(blue)*65536; 
		// I simply guessed this. This may not be correct in general!
  }
}



CgiPrivate::CgiPrivate (unsigned long maxCommands,
                        unsigned long maxCoordinates)
{
  maxCmds = maxCommands;
  commands = new CgiCommand[maxCommands];
  nextCmd  =  0;

  maxCoords = maxCoordinates;
  coordHeap = new Coordinate[maxCoordinates]; 
  nextCoord = coordHeap;

  nextOut = 0;
  cntPages = 1;
  actPage = 1;
  elemsOnPage = 0;  
  xoffs = 30;
  yoffs = 30;
  xzoom = 1;
  yzoom = 1;

  frame = 0;
  canvas = 0;
  textedit = 0;
  editFrame = 0;
  editPanel = 0;
  
  vrPageShown = 0;
}

void CgiPrivate::DoProgram
	(	int argc, char *argv[],
		UserProgram theProgram,
		int width, int height)
{
  xv_init(XV_INIT_ARGC_PTR_ARGV, &argc, argv, NULL);

  // Create the frame window at the given size at an arbitrary, that is
  // default, position. Install footer (left and right) and event
  // callback routine (FrameEvent) for resize events.
  frame = (Frame) xv_create (0, FRAME, 
  	XV_X, 8, XV_Y, 8,
	XV_WIDTH, width,
	XV_HEIGHT, height+panelH,
 	FRAME_SHOW_FOOTER, TRUE,
	SET_WIN_EVENT_PROC (CgiPrivate, this, FrameEvent),
		WIN_CONSUME_EVENTS, WIN_RESIZE, NULL,
       NULL);

  // Create the 'File' menu:
  menuFile = xv_create(0, MENU,  
		MENU_GEN_PIN_WINDOW, frame, strMenuFile,
                MENU_STRINGS, strFileLoad, strFileAgain, strFileEdit,
                      strFileCommander, "", strFileQuit, NULL,
		SET_MENU_NOTIFY_PROC(CgiPrivate, this, MenuFile),
		NULL);
  // Disable menu items not available at this moment:
  xv_set (xv_get(menuFile, MENU_NTH_ITEM, 6), MENU_INACTIVE, TRUE, NULL);
  xv_set (xv_get(menuFile, MENU_NTH_ITEM, itemReload), MENU_INACTIVE, argc<=1, NULL);
  xv_set (menuFile, MENU_DEFAULT, 3, NULL);

  // Create the 'View' menu:
  menuView = xv_create(0, MENU, 
		MENU_GEN_PIN_WINDOW, frame, strMenuView,
		MENU_STRINGS, strViewPrevPage, strViewNextPage, 
		              strViewFirstPage, strViewLastPage, "",
		              strViewZoomIn, strViewZoomOut, "",
		              strViewScrollLeft, strViewScrollRight,
		              strViewScrollUp, strViewScrollDown, "",
		              strViewReset, strViewQuit, NULL,
		SET_MENU_NOTIFY_PROC(CgiPrivate, this, MenuView),
		NULL);
  // Disable menu items not available at this moment:
  xv_set (xv_get(menuView, MENU_NTH_ITEM, 6), MENU_INACTIVE, TRUE, NULL);
  xv_set (xv_get(menuView, MENU_NTH_ITEM, 9), MENU_INACTIVE, TRUE, NULL);
  xv_set (xv_get(menuView, MENU_NTH_ITEM, 14), MENU_INACTIVE, TRUE, NULL);

  // Create the panel (container) that will contain both menu buttons and
  // additional 'Prev' and 'Next' buttons for fast access.
  panel = xv_create (frame, PANEL,
       XV_X, 0, XV_Y, 0, XV_WIDTH, width, XV_HEIGHT, panelH,
       NULL);

  (void) xv_create(panel, PANEL_BUTTON, 
		PANEL_LABEL_STRING, strMenuFile,
		PANEL_ITEM_MENU, menuFile,
		NULL);

  (void) xv_create(panel, PANEL_BUTTON, 
		PANEL_LABEL_STRING, strMenuView,
		PANEL_ITEM_MENU, menuView,
		NULL);

  butPrevPage = xv_create(panel, PANEL_BUTTON, 
		PANEL_LABEL_STRING, strButPrevPage,
		SET_BUTTON_PROC (CgiPrivate, this, PrevPage),
		NULL);
  butNextPage = xv_create(panel, PANEL_BUTTON, 
		PANEL_LABEL_STRING, strButNextPage,
		SET_BUTTON_PROC (CgiPrivate, this, NextPage),
		NULL);

  butSnapshot = xv_create(panel, PANEL_BUTTON, 
		PANEL_LABEL_STRING, strButSnapshot,
		SET_BUTTON_PROC (CgiPrivate, this, Snapshot),
		NULL);

      
  // Check available Visuals:

  XVisualInfo vTemplate;
  XVisualInfo *visualList;
  int visualsMatched;
  {
	Display *display;
	
	memset (&vTemplate, 0, sizeof(vTemplate));
	vTemplate.screen = 0;
	vTemplate.depth = 24;
	display = (Display *)xv_get (frame, XV_DISPLAY);
	visualList = XGetVisualInfo (display, VisualScreenMask|VisualDepthMask,
	                             &vTemplate, &visualsMatched);
#if 0		
	if (visualsMatched > 0)
	{
		cout << "Found " << visualsMatched << " 24-bit Visuals." << endl;
		for (int i=0; i < visualsMatched; i++) 
		{
			cout << i+1 << ". Visual:" << endl;
			cout << "  visualid      = " << visualList[i].visualid << endl;
			cout << "  screen        = " << visualList[i].screen << endl;
			cout << "  depth         = " << visualList[i].depth << endl;
			cout << "  class         = " << visualList[i].c_class << endl;
			cout << "  red_mask      = " << visualList[i].red_mask << endl;
			cout << "  green_mask    = " << visualList[i].green_mask << endl;
			cout << "  blue_mask     = " << visualList[i].blue_mask << endl;
			cout << "  colormap_size = " << visualList[i].colormap_size << endl;
			cout << "  bits_per_rgb  = " << visualList[i].bits_per_rgb << endl << endl;
		}
	}
#endif		
  }


  // Create the canvas window that covers the main area of the frame
  // and is used for all graphic output.
  // Resizing of the canvas window is synchronized with the resizing
  // of the frame (see FrameEvent()), so the canvas has no scrollbars
  // attached to it.

  Boolean trueColorWanted = False;
  
#if 0  
  if (visualsMatched > 1)
  {
    char c;
    cout << "CollageSystem: This machine supports TrueColor display.\n";
    cout << "               Use TrueColor (y/n) ?";
    cin >> c;
    trueColorWanted = (c == 'y' || c == 'Y');
  }
#endif  

  if (trueColorWanted)
  {
	cout << "CollageSystem: Using TrueColor." << endl;
  	canvas = xv_create (frame, CANVAS,
		XV_VISUAL, visualList[1].visual,
		NULL);
	rgbCache.SetTrueColor (True);
  }
  else
  {
	cout << "CollageSystem: Using default colormap." << endl;
  	canvas = xv_create (frame, CANVAS,
		NULL);
	rgbCache.SetTrueColor (False);
  }

  xv_set (canvas,
	XV_HELP_DATA, "cs:canvas",  // Install link for online help
	SET_CANVAS_REPAINT_PROC (CgiPrivate, this, CanvasRepaintProc),
	XV_X, 0, 
	XV_Y, panelH, 
	CANVAS_AUTO_SHRINK, FALSE,
	CANVAS_AUTO_EXPAND, FALSE,
	CANVAS_WIDTH, width, 
	CANVAS_HEIGHT, height, 
	CANVAS_RETAINED, FALSE, // no server caching of image - too costly.
	CANVAS_FIXED_IMAGE, FALSE, // no assumptions about fixed image
	NULL);

  // Install callback function for all kinds of events occuring in the
  // canvas window:
  xv_set (canvas_paint_window(canvas),
	SET_WIN_EVENT_PROC(CgiPrivate, this, CanvasEvent),
	WIN_CONSUME_EVENTS,
	  KBD_DONE, KBD_USE, LOC_DRAG, LOC_MOVE, LOC_WINENTER,
	  LOC_WINEXIT, WIN_ASCII_EVENTS, WIN_MOUSE_BUTTONS, NULL,
	NULL);

  // Icon for closed state:
  #include "cs.bitmap"

  // Flip the bits of the image data to achieve a correct display.
  // This algorithm is (c) by Stefan and a detailed understanding of
  // how it works is not neccessary in order to understand the rest
  // of the code!
  // By the way, there has to be an official way to convert the bitmap
  // into a hardware specific format, but we didn't figure this out, yet.
  int i, j;
  unsigned int oldvalue, newvalue, tmp;
  for (i = 0; i < 512; i++) {
  	oldvalue = cs_bits[i];
  	newvalue = 0;
  	for (j=0; j<8; j++) {
  		tmp = oldvalue & 1;
  		oldvalue = oldvalue >> 1;
  		newvalue = newvalue << 1;
  		newvalue = newvalue | tmp;
  	}
  	cs_bits[i]=newvalue;
  }

  // Create a server image from the icon data:
  Server_image icon_image = (Server_image) xv_create (XV_NULL, SERVER_IMAGE,
	XV_WIDTH, cs_width,
	XV_HEIGHT, cs_height,
	SERVER_IMAGE_DEPTH, 1,
	SERVER_IMAGE_BITS, cs_bits,
	NULL);

  // Create a icon from that image:
  Icon icon = (Icon) xv_create (frame, ICON,
	XV_WIDTH, cs_width, 
	XV_HEIGHT, cs_height+20,
	XV_LABEL, argc > 1 ? argv[1] : "(no file)",
	ICON_IMAGE, icon_image,
	NULL);

  // Attach the icon to the frame window:
  xv_set (frame, FRAME_ICON, icon, NULL);

  // Init some data:
  fsel = XV_NULL;
  _paint_window = canvas_paint_window(canvas);
  _dpy = (Display *)xv_get (_paint_window, XV_DISPLAY);
  _xwin = xv_get (_paint_window, XV_XID);

#if 0
  _gc = DefaultGC(_dpy, DefaultScreen(_dpy));
#else
  CreatePaintingGC ();
#endif

  _width = (int)xv_get(_paint_window, XV_WIDTH);
  _height = (int)xv_get(_paint_window, XV_HEIGHT);
  CreateRubberGC();
  _userPrg = theProgram;
  _userArgc = argc;
  for (i=0; i < cgiMaxArgs; i++) {
  	_userArgv[i] = new char[512];
  	_userArgv[i][0] = 0;
  }
  for (i=0; i < argc; i++)
  	strcpy (_userArgv[i], argv[i]);


  // Execute:
  SetWindowTitle();
  DoRunPrg();
  xv_main_loop (frame);
}

void CgiPrivate::SetWindowTitle()
{
  char title[150];

  sprintf (title, "%s (%s) -- [", "Collage-VR", __DATE__);
  if (_userArgc > 1)
    strcat (title, _userArgv[1]);
  strcat (title, "]");
  xv_set (frame, FRAME_LABEL, title, NULL);
  xv_set (xv_get (frame, FRAME_ICON), XV_LABEL, _userArgv[1], NULL);
}


CgiCommand *CgiPrivate::NewCmd (CgiCmdType type)
{
  // Allocates a new command from the heap and initialises its type.
  // Returns address of the command or NULL in case of error. 
  if (nextCmd < maxCmds) {
    if (type == CGI_CLEAR) cntPages++;
    commands[nextCmd].command = type;
    return (commands + nextCmd++);
  }
  else
    return (NULL);
}

Coordinate *CgiPrivate::AllocCoords (unsigned int cnt)
{
  // Allocates the 'cnt' Coordinate's from the heap and returns
  // the array's start address or NULL in case of error.
  if (nextCoord + cnt < coordHeap + maxCoords)
  {
    Coordinate *start = nextCoord;
    nextCoord += cnt;
    return (start);
  }
  else
    return (NULL);
}

void CgiPrivate::DoCommand (CgiCommand *cmd)
{
  // Executes the command 'cmd'.
  
  Coordinate *a = cmd->args;

  switch (cmd->command)
  {
    case CGI_FG_COLOR:
      DoFgColor ((int)a[0]); break;
    case CGI_FG_COLRGB:
      DoFgColor ((int)a[0], (int)a[1], (int)a[2]); break;
    case CGI_BG_COLOR:
      DoBgColor ((int)a[0]); break;
    case CGI_BG_COLRGB:
      DoBgColor ((int)a[0], (int)a[1], (int)a[2]); break;
    case CGI_CLEAR:
      DoClear (); break;
    case CGI_POINT:
      DoPoint (a[0], a[1]); break;
    case CGI_LINE:
      DoLine (a[0], a[1], a[2], a[3]); break;
    case CGI_MOVE_TO:
      DoMoveTo (a[0], a[1]); break;
    case CGI_LINE_TO:
      DoLineTo (a[0], a[1]); break;
    case CGI_POLY:
      DoPoly (cmd->u.cntArg/2, a); break;
    case CGI_FILLED_POLY:
      DoFilledPoly (cmd->u.cntArg/2, a); break;
    case CGI_REAL_ELLIPSE:
      DoRealEllipse (a[0], a[1], a[2], a[3], a[4], a[5], a[6], a[7], a[8], a[9]); break;
    case CGI_ELLIPSE:
      DoEllipse (a[0], a[1], a[2], a[3], (int)a[4], False); break;
    case CGI_FILLED_ELLIPSE:
      DoEllipse (a[0], a[1], a[2], a[3], (int)a[4], True); break;
    case CGI_TEXT:
      DoText (a[0], a[1], cmd->u.string); break;
    case CGI_BOXTEXT:
      DoBoxText (a[0], a[1], (int)a[2], (int)a[3], (int)a[4], cmd->u.string); break;
    case CGI_PAGE_TITLE:
      DoPageTitle (cmd->u.string); break;
    default:
      cout << "CGI error: unknown command # " << cmd->command << ".\n";
  }
}


// ----- X implementation -----------------------------------------------------

void CgiPrivate::DoFgColor (int c)
{
  // Sets the foreground color to a color index 'c'.

  XSetForeground (_dpy, _gc, c);
  fgColor = c;
}

void CgiPrivate::DoFgColor (int r, int g, int b)
{
  DoFgColor (rgbCache.GetIndex (r,g,b));
}

void CgiPrivate::DoBgColor (int c)
{
  // Sets the background color to a color index 'c'.
  XSetBackground (_dpy, _gc, c);
  bgColor = c;
}

void CgiPrivate::DoBgColor (int r, int g, int b)
{
  DoBgColor (rgbCache.GetIndex (r,g,b));
}

void CgiPrivate::DoClear ()
{
  // Clears screen. This is here done by filling the hole window with solid 
  // white. Resets screen color to black and clears the user's pagetitle.
  XSetForeground (_dpy, _gc, WHITE);
  XFillRectangle (_dpy, _xwin, _gc, 0, 0, _width, _height);
  XSetForeground (_dpy, _gc, fgColor);
  DoPageTitle ("");
}

void CgiPrivate::DoPoint (Coordinate x, Coordinate y)
{
  // Sets a single pixel at coordinates x/y.
  XDrawPoint (_dpy, _xwin, _gc, XOUT(x), YOUT(y));
}

void CgiPrivate::DoLine (Coordinate x1, Coordinate y1, 
                        Coordinate x2, Coordinate y2) 
{
  // Draws a line from x1/y1 to x2/y2.
  XDrawLine(_dpy, _xwin, _gc, XOUT(x1), YOUT(y1), XOUT(x2), YOUT(y2));
}

void CgiPrivate::DoMoveTo (Coordinate x, Coordinate y)
{
  // Sets the actual position to x/y.
  actx = x;
  acty = y;
}

void CgiPrivate::DoLineTo (Coordinate x, Coordinate y)
{
  // Draws a line from the actual position to x/y.
  XDrawLine(_dpy, _xwin, _gc, XOUT(actx), YOUT(acty), XOUT(x), YOUT(y));
  actx = x;
  acty = y;
}

void CgiPrivate::DoPoly (int count, Coordinate p[])
{
  // Draws a polygon with 'count' points. X-coordinates have event
  // indices, Y-coordinates have odd indices in the array p.
  if (count > MAX_POLY_POINTS)
    count = MAX_POLY_POINTS;
  for (int i=0; i < count; i++) {
    xp[i].x = XOUT(p[i*2]);
    xp[i].y = YOUT(p[i*2+1]);
  }  
  XDrawLines (_dpy, _xwin, _gc, xp, count, CoordModeOrigin);
 }

void CgiPrivate::DoFilledPoly (int count, Coordinate p[])
{
  // As DoPoly() but the polygon is solid filled with the actual foreground
  // color.
  if (count > MAX_POLY_POINTS)
    count = MAX_POLY_POINTS;
  for (int i=0; i < count; i++) {
    xp[i].x = XOUT(p[i*2]);
    xp[i].y = YOUT(p[i*2+1]);
  }  
  XSetFillStyle (_dpy, _gc, FillSolid);
  XFillPolygon (_dpy, _xwin, _gc, xp, count, Complex, CoordModeOrigin);
}

void CgiPrivate::DoRealEllipse 
    (Coordinate first_x1, Coordinate first_y1, Coordinate first_x2, Coordinate first_y2,
     Coordinate secnd_x1, Coordinate secnd_y1, Coordinate secnd_x2, Coordinate secnd_y2,
     float begAngle, float endAngle)
{
  /* the simple way: */
  XDrawLine(_dpy, _xwin, _gc, XOUT(first_x1), YOUT(first_y1),
                              XOUT(first_x2), YOUT(first_y2));
  XDrawLine(_dpy, _xwin, _gc, XOUT(secnd_x1), YOUT(secnd_y1),
                              XOUT(secnd_x2), YOUT(secnd_y2));
  /* ...one could do better than that... */
}

void CgiPrivate::DoEllipse (Coordinate x, Coordinate y, 
                            Coordinate r1, Coordinate r2, 
                            Boolean scale, Boolean fill)
{
  // Draws an ellipse that is either filled or non-filled and either
  // scaled or non-scaled. A non-scaled ellipse is typically used for
  // marker positions within the graphic that should not be scaled like
  // the other objects.
  // Note that the coordinate and width/height calculations may differ
  // in a system dependent way. In this X-specific implementation the
  // x and y coordinates passed to XFillArc() / XDrawArc() specify the
  // upper left corner of a rectangle that contains the ellipse. The
  // width and height parameters specify the width and height of that
  // rectangle. For more details refer to the man pages or other XLib
  // documentation.
  if (fill) {
    if (scale)
      XFillArc (_dpy, _xwin, _gc,          // context parameters
                XOUT(x),                   // x
                YOUT(y)-(int)(r2*yzoom),   // y
                (int)(r1*xzoom),           // width 
                (int)(r2*yzoom),           // height 
                0, 64*360);                // full arc 
    else
      XFillArc (_dpy, _xwin, _gc, 
                XOUT(x)-int(r1/2), 
                YOUT(y)-int(r2/2),
                (int)r1,
                (int)r2, 
                0, 64*360);
  } else {
    if (scale)
      XDrawArc (_dpy, _xwin, _gc, 
                XOUT(x), 
                YOUT(y)-(int)(r2*yzoom),
                (int)(r1*xzoom), 
                (int)(r2*yzoom), 
                0, 64*360); 
    else
      XDrawArc (_dpy, _xwin, _gc, 
                XOUT(x)-int(r1/2), 
                YOUT(y)-int(r2/2),
                (int)r1,
                (int)r2, 
                0, 64*360);
  }
}


void CgiPrivate::DoText (Coordinate x, Coordinate y, const char *text)
{
  // Outputs the string 'text' at coordinates x/y using the font actually
  // assigned to the canvas window.

  // Inquire information of the font metrics, specifically the height
  // of a character measured in pixel:
  int d1,d2,d3;
  XCharStruct cs;
  XFontStruct fs = *(XFontStruct *)xv_get(xv_get(canvas, XV_FONT), FONT_INFO);
  XTextExtents (&fs, "W", 1, 
                &d1, &d2, &d3, &cs); 

  // The parameter of the Y-coordinate of the text needs an additional
  // adjustment because YOUT() converts (here: "flips") just a single
  // coordinate but what we need here is the conversion of the rectangle
  // that will contain the text. The reason for this is that the caller
  // of DoText() assumes x/y to be the lower left corner of (the base
  // line of) the text and this is exactly what the X-routine expects
  // also - but in a flipped coordinate system. So we have to add the
  // text height to the final output position of the text.
  XDrawString (_dpy, _xwin, _gc, 
               XOUT(x), 
               YOUT(y)+cs.ascent+cs.descent, 
               text, strlen(text));
}

void CgiPrivate::DoBoxText (Coordinate cx, Coordinate cy, uint width,
                            uint height, Boolean frame, const char *text)
{
  // Draws 'text' centered in a box of 'width' and 'height' with and optional
  // frame. cx and cy are the coordinates of the center of the box.
  // The width of the box is expanded if needed.

  int d1,d2,d3;
  XCharStruct cs;
  XFontStruct fs = *(XFontStruct *)xv_get(xv_get(canvas, XV_FONT), FONT_INFO);
  XTextExtents (&fs, "W", 1, 
                &d1, &d2, &d3, &cs); 

  int x=XOUT(cx), y=YOUT(cy); // screen coordinates of center
  int l;
  uint tw = XTextWidth (&fs, text, l=strlen(text)); // pixel width of text
  if (tw > width) {
    width = tw + 4;  // text too long, expand width.
  }
  
  // Clear background:
  XSetForeground (_dpy, _gc, bgColor); // use of background color is intended
  XFillRectangle (_dpy, _xwin, _gc, x-width/2, y-height/2, width, height);

  // Draw text and (optional) frame rectangle:
  XSetForeground (_dpy, _gc, fgColor);
  if (frame)
    XDrawRectangle (_dpy, _xwin, _gc, x-width/2, y-height/2, width, height);
  XDrawString (_dpy, _xwin, _gc,
               x-tw/2, 
               y+(cs.ascent+cs.descent)/2, text, l);
}


void CgiPrivate::DoPageTitle (const char *text)
{
  xv_set (frame, FRAME_LEFT_FOOTER, text, NULL);
}


void CgiPrivate::ShowNewPage()

// Clear page and show commands until next CGI_CLEAR or end of commands. 
// Enable/disable next/prev buttons.

{
	xv_set (xv_get(menuView, MENU_NTH_ITEM, itemPrevPage), 
        	MENU_INACTIVE, nextOut==0, NULL);
	xv_set (butPrevPage, PANEL_INACTIVE, nextOut==0, NULL);
	DoClear();
	elemsOnPage=0;
	if (nextOut < nextCmd) 
	{
	  while (nextOut < nextCmd && commands[nextOut].command != CGI_CLEAR) {
  		DoCommand (commands+nextOut++);
  		elemsOnPage++;
	  }

	  if (nextCmd >= maxCmds && nextOut >= nextCmd) // overflow occured (alwas on last page)
  		DoPageTitle ("*** OUT OF HEAP SPACE *** Increase internal settings.");
	}	
	char msg[80];
	sprintf (msg, "%i of %i primitives, page %i of %i", elemsOnPage, nextCmd, actPage, cntPages);
	xv_set (frame, FRAME_RIGHT_FOOTER, msg, NULL);
	xv_set (xv_get(menuView, MENU_NTH_ITEM, itemNextPage), 
        	MENU_INACTIVE, nextOut >= nextCmd, NULL);
	xv_set (butNextPage, PANEL_INACTIVE, nextOut >= nextCmd, NULL);
  
	//
	// Start/update VRML viewer
	//
	if (   envi.vrOutput
	    && (vrPageShown == 0 || vrPageShown != actPage))
	{
		char command[200], vrFileName[256];
		
		vrPageShown = actPage;
		theVR.GetNameOfNthFile (actPage, vrFileName);
		if (vrFileName[0])
		{
			sprintf (command, VRVIEW_COMMAND, vrFileName);
			system (command);
		}
	}		
}

void CgiPrivate::ShowThisPage ()

// Show the actual page, assuming that 'nextOut' is the CLEAR-command
// for the next page or 0 or equal to 'nextCmd'.

{
  if (nextOut > 0) nextOut--; // backstep behind CLEAR
  while (nextOut > 0 &&  commands[nextOut].command != CGI_CLEAR)
    nextOut--;
  if (nextOut > 0) /* <=> command == CLEAR */
    nextOut++;
  ShowNewPage();
}


// ----------------------------------------------------------------------------
// ------ Interactive control (XView implementation) --------------------------
// ----------------------------------------------------------------------------

// ------ Frame events --------------------------------------------------------

void CgiPrivate::FrameEvent(Xv_Window window, Event *event, Notify_arg arg)
{
  switch(event_action(event))
  {
    case WIN_RESIZE: 
      xv_set (_paint_window, 
              XV_WIDTH, xv_get(canvas, XV_WIDTH),
              XV_HEIGHT, xv_get(canvas, XV_HEIGHT),
              NULL); 
     _width = (int)xv_get(_paint_window, XV_WIDTH);
     _height = (int)xv_get(_paint_window, XV_HEIGHT);
     xv_set (panel, XV_WIDTH, _width, NULL);
     break;
  }
}

// ------ Rubber banding ------------------------------------------------------

void CgiPrivate::CreateRubberGC()
{
  Display   *dpy;
  Window    xwin;
  XGCValues values;

  dpy  = (Display*)xv_get(canvas_paint_window(canvas), XV_DISPLAY, NULL);   
  xwin = xv_get(canvas_paint_window(canvas), XV_XID, NULL);

  values.function   = GXxor;
  values.plane_mask = AllPlanes;
  values.line_width = 1;
  values.fill_style = FillSolid;
  values.foreground = 1;

  rubber_gc = XCreateGC(dpy, xwin, GCFunction | GCPlaneMask | GCFillStyle |
			GCLineWidth | GCForeground, &values);
}

void CgiPrivate::CreatePaintingGC()
{
  Display   *dpy;
  Window    xwin;
  XGCValues values;

  dpy  = (Display*)xv_get(canvas_paint_window(canvas), XV_DISPLAY, NULL);   
  xwin = xv_get(canvas_paint_window(canvas), XV_XID, NULL);

  values.function   = GXcopy;
  values.plane_mask = AllPlanes;
  values.line_width = 1;
  values.fill_style = FillSolid;
  values.foreground = 1;

  _gc = XCreateGC (dpy, xwin, GCFunction | GCPlaneMask | GCFillStyle |
			GCLineWidth | GCForeground, &values);
}

void CgiPrivate::CanvasEvent(Xv_Window window, Event *event, Notify_arg arg)
{
	static int x,y,w,h;
	static Boolean dragging=False;

	switch(event_action(event))
	{
	case LOC_DRAG:
		if (event_left_is_down(event)) {
			// zoom function
			if (dragging) {
				DrawRubber (x, y, w, h); // undraw
				w = event_x(event) - x;
				h = event_y(event) - y;
				if (h && w && !event_ctrl_is_down(event)) {
					// optional: allow only proportional scaling:
					double r = fabs ((double)w / (double)h);
					double R = fabs ((double)_width / (double)_height);
					if (r > R) 
						h = int (abs(w)/R * (h > 0 ? 1 : -1));
					else
						w = int (abs(h)*R * (w >  0 ? 1 : -1));	
				}
				DrawRubber (x, y, w, h); // draw
			} 
			else {
				x = event_x(event);
				y = event_y(event);
				w = 0; h = 0;
				dragging = True;
			}
		}
		else if (event_middle_is_down(event) && !dragging) {
			// panning function
			int dx = x - event_x(event);
			int dy = y - event_y(event);
			if (abs(dx) > 32 || abs(dy) > 32) {
				xoffs -= dx;
				yoffs += dy;
				x = event_x(event);
				y = event_y(event);
				ShowThisPage();
			}	
		}
		break;

	case ACTION_SELECT:
		// begin zoom:
		if (dragging && event_button_is_down(event)/*(!)*/) {	
			DrawRubber (x, y, w, h); // undraw
			ZoomRect (x, y, w, h);
			dragging = False;
		}
		break;

	case ACTION_MENU:
		if (dragging)
		{
			// abort zoom:
			if (event_button_is_down(event)/*(!)*/) 
			{
				// cancel rubber banding:
				DrawRubber (x, y, w, h); // undraw
				dragging = False;
			}
		}
		else
		{
			// zoom out:
			if (event_button_is_down(event)/*(!)*/) 
				ZoomOut ();
		}
		break;

	case ACTION_ADJUST:
		// begin panning:
		if (! dragging) {
			x = event_x(event);
			y = event_y(event);
		}
		break;
	}
}

void CgiPrivate::DrawRubber (int x, int y, int w, int h)
{
	if (! (w && h)) return;
	if (w < 0) {x += w; w = -w;}
	if (h < 0) {y += h; h = -h;}
	XDrawRectangle (_dpy, _xwin, rubber_gc, x, y, w, h);
}

void CgiPrivate::ZoomRect (int x, int y, int w, int h)
{
	if (! (w && h)) return;
	if (w < 0) {x += w; w = -w;}
	if (h < 0) {y += h; h = -h;}
	y = _height - y - h;
	Coordinate xN = (x-xoffs)/xzoom; 
	Coordinate yN = (y-yoffs)/yzoom;
	double fw = (double)_width / (double)w; // >= 1
	double fh = (double)_height /(double)h; // >= 1
	xzoom *= fw;
	yzoom *= fh;
	xoffs = -xN*xzoom;
	yoffs = -yN*yzoom;
	ShowThisPage();
}

void CgiPrivate::ZoomRectAnimated (int x, int y, int w, int h)
{
#if 0
	// still not reversable:
	int steps = 500/elemsOnPage;
	if (steps < 1) steps = 1;
	else if (steps > 5) steps = 5;
	double dw = double(_width-w)/double(steps);
	double dh = double(_height-h)/double(steps);
	for (int i=1; i <= steps; i++) {
		ZoomRect (int(double(x)*1/double(steps)), 
		          int(double(y)*1/double(steps)), 
		          int(_width-dw), 
		          int(_height-dh));
	}
#else
	ZoomRect (x,y,w,h);
#endif
}


void CgiPrivate::CanvasRepaintProc
	(Canvas       canvas,        
	Xv_Window     paint_window,
	Display       *dpy,
	Window        xwin,
	Xv_xrectlist  *xrects) 
{
    _width = (int)xv_get(paint_window, XV_WIDTH);
    _height = (int)xv_get(paint_window, XV_HEIGHT);
    ShowThisPage ();
}



// ------- Menus --------------------------------------------------------------

void CgiPrivate::MenuFile(Menu menu, Menu_item menu_item)
{
  char* menu_text;

  menu_text = (char*) xv_get(menu_item, MENU_STRING);

  if (!strcmp(strFileLoad, menu_text))
    Load();
  else if (!strcmp(strFileAgain, menu_text))
    Again (NULL); // again with same file
  else if (!strcmp(strFileEdit, menu_text))
    Edit();
  else if (!strcmp(strFileCommander, menu_text))
    NewCommander();  
  else if (!strcmp(strFileQuit, menu_text))
    Quit();
}

void CgiPrivate::MenuView(Menu menu, Menu_item menu_item)
{
  char* menu_text;

  menu_text = (char*) xv_get(menu_item, MENU_STRING);

  if (!strcmp(strViewPrevPage, menu_text))
    PrevPage(0,0);
  else if (!strcmp(strViewNextPage, menu_text))
    NextPage(0,0);
  else if (!strcmp(strViewFirstPage, menu_text))
    FirstPage();
  else if (!strcmp(strViewLastPage, menu_text))
    LastPage();
  else if (!strcmp(strViewZoomIn, menu_text))
    ZoomIn();
  else if (!strcmp(strViewZoomOut, menu_text))
    ZoomOut();
  else if (!strcmp(strViewScrollLeft, menu_text))
    ScrollLeft();
  else if (!strcmp(strViewScrollRight, menu_text))
    ScrollRight();
  else if (!strcmp(strViewScrollUp, menu_text))
    ScrollUp();
  else if (!strcmp(strViewScrollDown, menu_text))
    ScrollDown();
  else if (!strcmp(strViewReset, menu_text))
    Reset();
  else if (!strcmp(strViewQuit, menu_text))
    Quit();
}

void CgiPrivate::PrevPage(Panel_item item, Event *event)//unused
{
  for (int i=0; i < 2; i++)
  {
    if (nextOut > 0) nextOut--;
    while (nextOut > 0 &&  commands[nextOut].command != CGI_CLEAR)
      nextOut--;
  }
  if (nextOut > 0) /* <=> command == CLEAR */
  	nextOut++;
  actPage--; 
  ShowNewPage();
}

void CgiPrivate::NextPage (Panel_item item, Event *event)//unused
{
  nextOut++;
  actPage++;
  ShowNewPage();
}

void CgiPrivate::FirstPage ()
{
  nextOut = 0;
  actPage = 1;
  ShowNewPage();
}

void CgiPrivate::LastPage ()
{
  nextOut = nextCmd;
  actPage = cntPages;
  ShowThisPage();
}


void CgiPrivate::ScrollLeft ()
{
  xoffs -= _width/20;
  ShowThisPage();
}

void CgiPrivate::ScrollRight ()
{
  xoffs += _width/20;
  ShowThisPage();
}

void CgiPrivate::ScrollUp ()
{
  yoffs += _height/20;
  ShowThisPage();
}

void CgiPrivate::ScrollDown ()
{
  yoffs -= _height/20;
  ShowThisPage();
}

void CgiPrivate::ZoomIn ()
{
#if 1
    ZoomRectAnimated (int(0.1*_width), int(0.1*_height), int(0.8*_width), int(0.8*_height));

#else
  xzoom *= 1.1;
  yzoom *= 1.1;
  // xoffs -= (0.05*_width);
  // yoffs -= (0.05*_height);
  ShowThisPage();
#endif
}

void CgiPrivate::ZoomOut ()
{
#if 1
  ZoomRectAnimated (int(-0.125*_width), int(-0.125*_height), int(1.25*_width), int(1.25*_height));
#else
  xzoom /= 1.1;
  yzoom /= 1.1;
  // xoffs += (int)(0.05*_width);
  // yoffs += (int)(0.05*_height);
  ShowThisPage();
#endif
}

void CgiPrivate::Reset ()
{
  xzoom = 1.0;
  yzoom = 1.0;
  xoffs = 30;
  yoffs = 30;
  ShowThisPage();
}

void CgiPrivate::DoSnapshot (char *filename)
{
#if 0
	// Use XWriteBitmapFile (seems to handle B/W only):
	Pixmap pixmap;

	pixmap = XCreatePixmap (_dpy, _xwin, _width, _height, xv_get (_paint_window, XV_DEPTH));
	XCopyArea (_dpy, _xwin, pixmap, _gc, 0, 0, _width, _height, 0, 0);
	if (XWriteBitmapFile (_dpy, "snapcs", pixmap, _width, _height, -1, -1) != BitmapSuccess)
		cout << "Error writing bitmap file" << endl;
	else
		cout << "Bitmap file 'snapcs' written" << endl;
#elif 0
	// Use xgrabsc:
	char cmd[256], snapButLabel[32];
	static int snapCnt = 0;
	
	sprintf (cmd, "xgrabsc -o ColSnap_%i.dmp -id %Li -noborders -xwd", snapCnt, (long)_xwin);
	int res = system (cmd);
	cout << "result of '" << cmd << "': " << res << endl;
	snapCnt++;
	sprintf (snapButLabel, "%s (%i)", strButSnapshot, snapCnt);
	xv_set (item,
	        PANEL_LABEL_STRING, snapButLabel,
	        NULL);
	
#else
	char cmd[256], snapButLabel[32];
	static int snapCnt = 0;

	if (filename)
		sprintf (cmd, "convert X:%lu %s", 
	    	          (unsigned long)_xwin, filename);
	else	
		sprintf (cmd, "convert X:%lu PNM:ColSnap_%03i.ppm", 
	    	          (unsigned long)_xwin, snapCnt);
	    	          
	int res = system (cmd);
	cout << "result of '" << cmd << "': " << res << endl;

	if (! filename)
	{
		snapCnt++;
		sprintf (snapButLabel, "%s (%i)", strButSnapshot, snapCnt);
		xv_set (butSnapshot,
	        	PANEL_LABEL_STRING, snapButLabel,
	        	NULL);
	}
#endif
}

void CgiPrivate::Snapshot (Panel_item item, Event *event)
{
	DoSnapshot (NULL);
}

void CgiPrivate::Quit ()
{
  if (textedit) {
    delete textedit;
    textedit = NULL;
    xv_destroy_safe (editFrame);
  }
  xv_destroy_safe (frame);
}

void CgiPrivate::Again (const char *arg)
{
  if (_userArgc <= 1)
  {
    if (! arg)
      return; // no current file loaded
    _userArgc = 2;
  }
  if (! arg)
    arg = _userArgv[1];
  static char newarg[200];
  strcpy (newarg, arg);
  strcpy (_userArgv[1], newarg);
  SetWindowTitle();
  delete [] commands;
  delete [] coordHeap;
  commands = new CgiCommand[maxCmds];
  nextCmd  =  0;
  coordHeap = new Coordinate[maxCoords]; 
  nextCoord = coordHeap;
  int oldActPage = actPage;
  Boolean toEnd = (oldActPage == cntPages); // remember that is was the last page
  DoRunPrg();
  if (toEnd)
    LastPage();
  else {
    nextOut = 0;
    actPage = 1;
    while (nextOut < nextCmd) {
    	if (actPage == oldActPage)
    		break;
    	if (commands[nextOut++].command == CGI_CLEAR) 
    		actPage++;
    }
    if (actPage == oldActPage)
    	ShowNewPage(); // nextOut == 0 || command != CLEAR
    else 
   	ShowThisPage();
  }
  xv_set (xv_get(menuFile, MENU_NTH_ITEM, itemReload), MENU_INACTIVE, _userArgc<=1, NULL);
}


//----- DoRunPrg --------------------------------------------------------------

void CgiPrivate::DoRunPrg()
{
	nextOut = 0;
	cntPages = 1; 
	actPage = 1; 
	vrPageShown = 0;
	rgbCache.Reset (_dpy);
	DoBgColor (WHITE);
	DoFgColor (BLACK);
	if (envi.interp)
		cout << "Intepreter already created" << endl;
	_userPrg (_userArgc, _userArgv, envi);
#if USETCL
    if (envi.interp)
    {
    	Tcl_DeleteInterp (envi.interp);
    	envi.interp = NULL;
    }
#endif
}



// ----- File chooser and Load() ----------------------------------------------

void CgiPrivate::FileChooserNotify 
	(File_chooser f, char *pathname, char *name)
{
  _userArgc = 2;
  Again (pathname);
  if (textedit) EditSyncNewFile();
}

Notify_value CgiPrivate::FCDestroyIP (Notify_client client, Destroy_status status)
{
  return NOTIFY_DONE;
}


void CgiPrivate::Load()
{
  if (fsel) xv_destroy (fsel);

  fsel = xv_create (frame, FILE_CHOOSER,  
		FILE_CHOOSER_ABBREV_VIEW, TRUE, 
		// FILE_CHOOSER_FILTER_STRING, "cgr",
		FILE_CHOOSER_TYPE, FILE_CHOOSER_OPEN,
		SET_FILE_CHOOSER_NOTIFY_FUNC(CgiPrivate, this, FileChooserNotify),
		//FRAME_DONE_PROC, FileChooserDone,
		XV_X, xv_get (frame, XV_X)+10,
		XV_Y, xv_get (frame, XV_Y)+20,
		NULL);
  CPP_NOTIFY_INTERPOSE_DESTROY_FUNC (fsel, CgiPrivate, this, FCDestroyIP);
  xv_set (fsel, XV_SHOW, TRUE, NULL);
}

//----- Open new TclCommander -------------------------------------------------

void CgiPrivate::NewCommander ()
{
	(void) new TclCommander (*this);
}


// ----------------------------------------------------------------------------
// ----- Editor handling ------------------------------------------------------
// ----------------------------------------------------------------------------

int EditNotify (Textedit *textedit, Textsw textsw, Attr_avlist attributes, 
                void *cgiPrivate)
{
  int pass_on = TRUE; // pass the event to the text subwindow's default proc
  Attr_avlist attrs;
 
  for (attrs = attributes; *attrs; attrs = attr_next(attrs))
  {
    switch ((Textsw_action)(*attrs))
    {
      case TEXTSW_ACTION_LOADED_FILE:
        //cout << "Textedit loaded file " << (char *)attrs[1] << "\n";
        pass_on = TRUE;
        break;
      default:
        pass_on = TRUE;
        break;
    }    
  }
  return pass_on;

  return 1;
}


static Panel_setting PanelEditNotify (Panel_item item, Event *event)
{
  return PANEL_NONE;
}


void CgiPrivate::EditFrameEvent(Xv_Window window, Event *event, Notify_arg arg)
{
  switch(event_action(event))
  {
    case WIN_RESIZE: 
      xv_set (textedit->GetTextsw(), 
              XV_WIDTH, xv_get(editFrame, XV_WIDTH),
              XV_HEIGHT, xv_get(editFrame, XV_HEIGHT)-panelH,
              NULL); 
     xv_set (editPanel, XV_WIDTH, xv_get(editFrame, XV_WIDTH), NULL);
     break;
  }
}

void CgiPrivate::EditDoSave ()
{
	strcpy (_userArgv[1], (char *)xv_get (editEdit, PANEL_VALUE));
	textsw_store_file (textedit->GetTextsw(), _userArgv[1], 0, 0);
	xv_set (editFrame, FRAME_LABEL, _userArgv[1], NULL);
}

void CgiPrivate::EditRunBut (Panel_item item, Event *event)
{
  EditDoSave();
  Again (_userArgv[1]);
}

void CgiPrivate::EditSaveBut (Panel_item item, Event *event)
{
  EditDoSave();
}

void CgiPrivate::EditSaveAndCloseBut (Panel_item item, Event *event)
{
  EditDoSave();
  delete textedit;
  textedit = NULL;
  xv_destroy_safe (editFrame);
  Again (_userArgv[1]);
}

void CgiPrivate::EditAbortBut (Panel_item item, Event *event)
{
  delete textedit;
  textedit = NULL;
  xv_destroy_safe (editFrame);
}

void CgiPrivate::EditHelpBut (Panel_item item, Event *event)
{
  xv_help_show (editFrame, "cs:editor", event);
}

void CgiPrivate::EditLoadBut (Panel_item item, Event *event)
{
	strcpy (_userArgv[1], (char *)xv_get (editEdit, PANEL_VALUE));
	textedit->Load (_userArgv[1]);
	xv_set (editFrame, FRAME_LABEL, _userArgv[1], NULL);
}

void CgiPrivate::EditSyncNewFile ()
{
	textedit->Load (_userArgv[1]);
	xv_set (editFrame, FRAME_LABEL, _userArgv[1], NULL);
	xv_set (editEdit, PANEL_VALUE, _userArgv[1], NULL);
}
	
void CgiPrivate::Edit()
{
	if (textedit) {
		xv_set (editFrame, XV_SHOW, TRUE, NULL);	
		return;
	}

	editFrame = xv_create (frame, FRAME, 
		XV_X, xv_get (frame, XV_X),
		XV_Y, xv_get (frame, XV_Y)+100,
		XV_WIDTH, 800,
		XV_HEIGHT, 700,
		FRAME_LABEL, _userArgv[1],
		SET_WIN_EVENT_PROC (CgiPrivate, this, EditFrameEvent),
		NULL);

	editPanel = xv_create (editFrame, PANEL,
		XV_X, 0, XV_Y, 0, XV_WIDTH, 700, XV_HEIGHT, panelH,
		NULL);

	(void) xv_create(editPanel, PANEL_BUTTON, 
		PANEL_LABEL_STRING, "Save & run",
		SET_BUTTON_PROC (CgiPrivate, this, EditRunBut),
		NULL);

	(void) xv_create(editPanel, PANEL_BUTTON, 
		PANEL_LABEL_STRING, "Save",
		SET_BUTTON_PROC (CgiPrivate, this, EditSaveBut),
		NULL);
		
	(void) xv_create(editPanel, PANEL_BUTTON, 
		PANEL_LABEL_STRING, "Save & close",
		SET_BUTTON_PROC (CgiPrivate, this, EditSaveAndCloseBut),
		NULL);
		
	(void) xv_create(editPanel, PANEL_BUTTON, 
		PANEL_LABEL_STRING, "Abort",
		SET_BUTTON_PROC (CgiPrivate, this, EditAbortBut),
		NULL);
		
	(void) xv_create(editPanel, PANEL_BUTTON, 
		PANEL_LABEL_STRING, "Help",
		SET_BUTTON_PROC (CgiPrivate, this, EditHelpBut),
		NULL);
		
	editEdit = xv_create (editPanel, PANEL_TEXT, 
		PANEL_LABEL_STRING, " Name:",
		PANEL_VALUE, _userArgv[1],
        	PANEL_NOTIFY_PROC, PanelEditNotify,
		PANEL_VALUE_DISPLAY_LENGTH, 20,
		PANEL_VALUE_STORED_LENGTH, 200,
		NULL);
	(void) xv_create(editPanel, PANEL_BUTTON, 
		PANEL_LABEL_STRING, "Load",
		SET_BUTTON_PROC (CgiPrivate, this, EditLoadBut),
		NULL);

	textedit = new Textedit (editFrame);
	textedit->Load (_userArgv[1]);
	textedit->SetNotifyProc (EditNotify, this);
	xv_set (textedit->GetTextsw(),
		XV_X, 0,
		XV_Y, panelH,
		XV_WIDTH, 550,
		XV_HEIGHT, 300,
		TEXTSW_AUTO_INDENT, TRUE,
		TEXTSW_IGNORE_LIMIT, TEXTSW_INFINITY,
		NULL);

	xv_set (editFrame, XV_SHOW, TRUE, NULL);
}

