/*
// colgram.l  --  CollageVR
//
// LEX scanner for CollageVR Grammar
//
// Author             : Nils Schwabe
// Date of creation   : Mar 94
// Last modification  : 4 Dec 96 ns
//
// University of Bremen, Germany
*/

  #include <iostream.h>
  #include <ctype.h>
  #include "yytypes.h"
  #include "boundbox.h"
  #include "colgram.tab.h"  // this is y.tab.h if YACC is used

  char yytextbuf[256]=""; // used by scanner(write) and parser(read)
  int yylineno=0;

  #define MY_YYLMAX 200

WHITE_CHAR	[\01-\40]
NUMBER		[0-9]*\.?[0-9]+((e|E)-?[0-9]+)?

%%

[Bb]egin[Aa]ttributes	return BEGIN_ATTRIBUTES;
[Ee]nd[Aa]ttributes	return END_ATTRIBUTES;
[Gg]rammar		return GRAMMAR;
[Ss]tart		return START;
[Rr]ule			return RULE;
[Tt]able		return TABLE;
[Pp]ins			return PINS;
[Ee]dge			return EDGE;
[Ee]nd[Cc]ollage	return ENDCOLLAGE;
[Dd]ot			return DOT;
[Dd](ash)?[Ll]ine	return DLINE;
[Ll]ine			return LINE;
[Pp](oly)?[Ll]ine	return PLINE;
[Pp](oly)?[Gg]on	return PGON;
[Ff][Pp](oly)?[Gg]on	return FPGON;
[Cc]irc(le)?		return CIRC;
[Ff][Cc]irc(le)?	return FCIRC;
[Ee]ll(ipse)?		return ELL;
[Ss]phere			return SPHERE;
[Cc]ube				return CUBE;
[Cc]one				return CONE;
[Cc]ylinder			return CYLINDER;
[Nn]ative			return NATIVE;
[Aa]ll				return PART_ALL;
[Ss]ides			return PART_SIDES;
[Bb]ottom			return PART_BOTTOM;
[Tt]op				return PART_TOP;
[Pp]oint[Ll]ight	return POINTLIGHT;
[Ff][Ee]ll(ipse)?	return FELL;
[Tt]ransformation(s)?   return TRANSFORMATIONS;
[Aa]ffine               return T_AFFINE;
[Tt]ranslation(s)?      return T_TRANSLATIONS;
[Ss]cale[Tt]ranslation(s)? return T_SCALETRANSLATIONS;
[Dd]erivation		return DERIVATION;
[Uu]ntil		return UNTIL;
[Dd]o			return DO;
[Pp]rob(ab(ility)?)?	return PROB;
[Gg]lobal		return GLOBAL;
[Cc]olor		return COLOR;
[Rr][Gg][Bb]		return RGB;
[Cc]olor[Pp]roc		return COLORPROC;
[Tt]imes		return TIMES;
[Rr]epeat		return REPEAT;
\/[fF].*			{
			  char *s=yytext+2;
			  while (*s && isspace(*s)) s++;
			  strcpy (yytextbuf, s);
			  s = yytextbuf;
			  while (*s && !isspace(*s)) s++;
			  if (s) *s = 0;
			  return F_OPTION;
			}
\/[vV].*			{
			  char *s=yytext+2;
			  while (*s && isspace(*s)) s++;
			  strcpy (yytextbuf, s);
			  s = yytextbuf;
			  while (*s && !isspace(*s)) s++;
			  if (s) *s = 0;
			  return V_OPTION;
			}			
\/[aA](nnotate(d)?)?			return A_OPTION;
\/[lL](ast)?([oO](nly)?)?		return LASTONLY_OPTION;
\/[cC](olor)?[cC](hange)?[pP](art(s)?)?	return CCP_OPTION;
\/[rR](and(om)?)?([iI]nit)? 		return R_OPTION;
\/[dD](im(ension(s)?)?)? 		return D_OPTION;
\/[dD](efault)?[pP](art)?[cC](olor)? 	return DPC_OPTION;
\/[dD](efault)?[eE](dge)?[cC](olor)? 	return DEC_OPTION;
\/[Bb][Bb]ox				{yylval.i = OptBBox; 
                                         return BBOX_OPTION;}
\/[Xx][Bb][Bb]ox			{yylval.i = OptXBBox; 
                                         return BBOX_OPTION;}
\/[Yy][Bb][Bb]ox			{yylval.i = OptYBBox; 
                                         return BBOX_OPTION;}
\/[Ss]ource				return SOURCE_OPTION;
\/[Ee]val				return EVAL_OPTION;
\/[Cc](olor)?[Cc](ontext)?	return CC_OPTION;
\/[Mm](erge)?[Pp](arts)?	return MP_OPTION;
\/[Kk](reo(s)?)?[Ii](nvers(e)?)?	return KI_OPTION;
[Mm]in					{yylval.i = BMin; 
			                 return BBOX_OPERATION;}
[Mm]ax					{yylval.i = BMax; 
			                 return BBOX_OPERATION;}
[Mm]in[Mm]ax				{yylval.i = BMinMax; 
			                 return BBOX_OPERATION;}
[Mm]ax[Mm]in				{yylval.i = BMaxMin; 
			                 return BBOX_OPERATION;}
[Aa]vg					{yylval.i = BAvg; 
			                 return BBOX_OPERATION;}
[Cc]ontext				return CONTEXT;
[Nn]egative				return NEGATIVE;
[Aa]ny					return ANY;
[Oo]f					return OF;
[Dd]elete				return DELETE;
[Tt]able[Ii]nfo			return TABLEINFO;
[Ss]equential			return SEQUENTIAL;
[Ee]xhaust[Pp]rob(abilities)?	return EXHAUSTPROB;
[Oo]ne[Ff]rom[Bb]ase			return ONEFROMBASE;
[Pp]t[Pp]ush					return PTPUSH;
[Pp]t[Pp]op						return PTPOP;
[Tt]ranslate					return TRANSLATE;
[Mm]ap							return MAP;
[Tt]o							return TO;
[Cc]hain						return CHAIN;
[Nn]o[Cc]hain					return NOCHAIN;
\.\.			return UPTO;
[Ll]			return LASTPIC;
\$			return DOLLAR;


\/			return SLASH;
,			return COORDSEP;
:			return COLON;
\(			return LBRACKET;
\)			return RBRACKET;
\{			return LBRACE;
\}			return RBRACE;
\+			return PLUS;
\-			return MINUS;
\*			return MULTIPLY;

sqrt			return FKT_SQRT;
sin			return FKT_SIN;
cos			return FKT_COS;
tan			return FKT_TAN;
asin			return FKT_ASIN;
acos			return FKT_ACOS;
atan			return FKT_ATAN;
sinh			return FKT_SINH;
cosh			return FKT_COSH;
tanh			return FKT_TANH;
exp			return FKT_EXP;
log			return FKT_LOG;
log10			return FKT_LOG10;
rotate3d	return FKT_ROT3D;

[A-Za-z]	{
		  yytextbuf[0] = yytext[0];
		  yytextbuf[1] = 0;
		  yylval.s = yytextbuf;
		  return STRING;
		}

{NUMBER}	{
		  yylval.n.i = atoi (yytext);
		  yylval.n.r = atof (yytext);
		  return NUM;
		}

\"		{
		  if (ScanCString()) {
		    yylval.s = yytext;
		    return STRING;
		  }
		  else
		    return STRINGERROR;
		}

\n		{yylineno++;}

\%.*		;
{WHITE_CHAR}	;

.		return UNKNOWNSYM;

%%


int ScanCString (void) 
 
/* Scans a string literal in C-syntax. A starting " is NOT expected (must already 
   be scanned). Scanning is done until a " is read, a backslash syntax error
   occured or the end of the input stream or output buffer has been reached. 
   The latter cases result in a return value of 0.    
   Return value:
     1=ok, 0=error. 
   Results:
     The pure string (with all escape sequences resolved) is copied into yytext. 
   Extras:
     Supports string concatenation over several input lines: If a lines ends with a \
     (i.e. if a backslash-newline combination is detected), the following line will be
     concated to the string.
*/   
{
  int c=0;  /* character from input stream */
  int i=0;  /* index for yytext */

  /* memset (yytext, 0, YYLMAX); */
  yyleng = 0; 
  while (i < MY_YYLMAX && (c=yyinput()) > 0 && c != '"')
  { 
    if (c == '\\')
    {
      if ((c = yyinput()) <= 0)
      { 
        yyleng = i;
        yytext[i] = 0;
        return 0;
      }
      switch (c)
      {
        case '\n': /* a backslash-newline in the input stream -> concat */
          break;
        case 'b': /* backspace */
          yytext[i++] = 8;
          break;
        case 'f': /* formfeed */
          yytext[i++] = 12;
          break;
        case 'n':  /* newline */
          yytext[i++] = 10; 
          break;
        case 'r': /* return */
          yytext[i++] = 13;
          break;
        case 't': /* tab */
          yytext[i++] = 9;
          break;
        case '\\': /* backslash */
          yytext[i++] = '\\';
          break;
        case '\'': /* ' */
          yytext[i++] = '\'';
          break;
        case '"': /* " */
          yytext[i++] = '"';
          break;
        default:
          {
            int k=0;
            yytext[i] = 0;

            while (c > 0 && c >= '0' && c <= '7' && k < 3)  /* octal number, 3 places max. */
            {
              yytext[i] = yytext[i]*8+c-'0';    
              c=yyinput();
              k++;
            }
            if (k==0)
            { 
              yytext[i++] = '\\'; /* copy the incorrect escape sequence */
              yytext[i++] = c;
              yyleng = i;
              yytext[i] = 0;
              return (0);
            }
            else
              i++;
            if (c > 0) unput(c);
          }
      } /* switch */
    }
    else
      yytext[i++] = c;
  } /* while */

  yyleng = i;
  yytext[i] = 0;
  return (c=='"');
}
