%{
//
// colgram.y  --  CollageVR
//
// YACC parser for CollageSystem/1 Grammar
//
// Author            : N. Schwabe
// Date of creation  : 3/94
// Last modification : 18 Dec 96 
//
// University of Bremen, Germany
//

#include <limits.h>
#include <stdlib.h>
#include <string.h>
#include <stream.h>
#include "collage.h"
#include "psout.h"
#include "vrout.h"
#include "envi.h"
#include "trans.h"
#include "mathutil.h"
#include "yytypes.h"

static unsigned int curInd;       // Current index into the following arrays (0..2)
static Collage *curCollage[3];    // Currently constructed collage.
								  // [0]=right hand side, [1]=context, [2]=negative context
static PartSet *curPartSet[3]; 
static PinSequence *curPins[3];
static EdgeSet *curEdgeSet[3];
static Boolean curAnyOf[2];       // 'any of' context option. [0]=context, [1]=neg. context
static Boolean curDelContext;
static Part *curPart;
static PointSequence *curPS=0, *curiPS=0;
static Textblock *curTextblock=0;
static TransFinder *curTrans=0, *defaultTrans=0;
static OptionBBox *curOptBBox=0;
static PictSet *curPictSet=0;
static SList<TableElemEntry> *curTableElemEntries=0;
static TableInfo *curTableInfo=0;
static PointSequence *curPTS1=0, *curPTS2=0; // internal point sequences
	// Allocation of these pointer variables is always
	// done within the defining nonterminal. Deallocation,
	// if any, must be done where the result is used. This
	// allows the pointers to be used directly where it
	// is possible, which is much more efficient than always
	// copying the whole structure.  

static char curRuleLabel[256];
static ColorIndex defaultPartColor=1, defaultEdgeColor=1;
static int pseudoPicCnt=1;
static Boolean chainTrans = False; // chain parser point transformations

// Point transformation stack:
#define PT_SIZE 128
static Transformation *ptStack[PT_SIZE]; 
  // (this array is guaranteed by C++ to be completely 0 at startup)
static signed int ptInd = -1; // index into ptStack, -1 <=> stack is empty.


// Point transformation stack utility functions:

static void PtClearStack ()
{
	ptInd = -1;
	memset (ptStack, 0, sizeof(ptStack));
}

static void PtTransform (Point *p)
{
	signed int i;
	for (i=ptInd; i >= 0; i--)
		ptStack[i]->Transform (*p, p);
}

static void PtTransformFigure (Figure *f)
{
	signed int i;
	for (i=ptInd; i >= 0; i--)
		f->Transform (*ptStack[i]);
}



#define YYERROR_VERBOSE 1
#define YYDEBUG 1

%}  


%union {
  int       i;
  char      c;
  char      *s; /* read only! */
  NumberRec n;
//  PointRec  p;
  Point     *p;
}

%token <s> STRING
%token <n> NUM
%token UNKNOWNSYM STRINGERROR
       GRAMMAR START RULE TABLE PINS EDGE ENDCOLLAGE DERIVATION
       TRANSFORMATIONS T_AFFINE T_ROTATIONS T_SCALINGS T_TRANSLATIONS
       T_SCALETRANSLATIONS FKT MATRIX CONTEXT NEGATIVE ANY OF DELETE
       SLASH COLON COORDSEP LBRACKET RBRACKET
       F_OPTION A_OPTION CCP_OPTION R_OPTION D_OPTION DPC_OPTION DEC_OPTION
       LASTONLY_OPTION SOURCE_OPTION EVAL_OPTION CC_OPTION MP_OPTION V_OPTION
       KI_OPTION
       DOT DLINE LINE PLINE PGON FPGON CIRC FCIRC ELL FELL SPHERE
       CUBE POINTLIGHT CONE CYLINDER PART_ALL PART_SIDES PART_BOTTOM PART_TOP
       NATIVE UNTIL DO BEGIN_ATTRIBUTES END_ATTRIBUTES PROB GLOBAL COLOR RGB
       UPTO LASTPIC COLORPROC DOLLAR REPEAT TIMES
       PLUS MINUS MULTIPLY FKT_SQRT FKT_SIN FKT_COS FKT_TAN FKT_ASIN
       FKT_ACOS FKT_ATAN FKT_SINH FKT_COSH FKT_TANH FKT_LOG FKT_LOG10
       FKT_EXP FKT_ROT3D
       TABLEINFO SEQUENTIAL EXHAUSTPROB ONEFROMBASE
       PTPUSH PTPOP TRANSLATE MAP TO LBRACE RBRACE CHAIN NOCHAIN
 %token <i> BBOX_OPTION
%token <i> BBOX_OPERATION 

%start Description

%type <p> Point  
%type <p> internalPoint
%type <p> transPoint
%type <s> Label  
%type <i> Tablenumber
%type <i> PolygonKind
%type <i> CircleKind
%type <i> EllipseKind
%type <i> ObjectParts
%type <n> Angle
%type <n> RuleProb
%type <n> OptRuleProb
%type <n> RuleOptions 
%type <i> OptPartColor
%type <i> OptEdgeColor
%type <i> XNumber
%type <n> Num
%type <n> OldNum
%type <n> Product
%type <n> Term
%%

Description	: 
		{
			yylineno = 1;
			curInd = 0;
			curCollage[0] = 0;
			curCollage[1] = 0;
			curCollage[2] = 0;
			curPartSet[0] = 0;
			curPartSet[1] = 0;
			curPartSet[2] = 0;
			curPins[0] = 0;
			curPins[1] = 0;
			curPins[2] = 0;
			curEdgeSet[0] = 0;	
			curEdgeSet[1] = 0;	
			curEdgeSet[2] = 0;
			curPart = 0;
			curPS = 0;
			curiPS = 0;
			curTextblock = 0;
			curTrans = 0;
			curOptBBox = 0;
			curPictSet = 0;
			defaultTrans = 0;
			defaultPartColor = 1;
			defaultEdgeColor = 1;
			pseudoPicCnt = 1;
			PtClearStack ();
			chainTrans = False;
		}
		Optionspart TbGlobal Grammarpart Derivationpart
		;


/****** Options **************************************************************/

Optionspart	: Options
		;

Options		: Options Option
		| /* empty */
		;

Option	: OptionF    /* PostScript file */
		| OptionV    /* VRML file */
		| OptionA    /* annotated collage grammars */ 
		| OptionCCP  /* color change parts (generation offset) */
		| OptionDPC  /* default part color */
		| OptionDEC  /* default edge color */
		| OptionD    /* dimension */
		| OptionR    /* random-init */
		| OptionBBox /* bounding box */
		| OptionLO   /* last only */
		| OptionSource /* source Tcl script */
		| OptionEval   /* eval string as Tcl script */
		| OptionCC     /* color context option */
		| OptionMP     /* merge parts option */
		| OptionKI	  /* inverse Collage grammars */
		;


/****** Simple options *******************************************************/

OptionF		: F_OPTION
		{
			strcpy (theEnvi->filenamePattern, yytextbuf);
			theEnvi->output = True;
			thePS.SetPattern (theEnvi->filenamePattern);
		}
		;

OptionV		: V_OPTION
		{
			strcpy (theEnvi->vrFilenamePattern, yytextbuf);
			theEnvi->vrOutput = True;
			theVR.SetPattern (theEnvi->vrFilenamePattern);
		}
		;

OptionA		: A_OPTION
		{
			theEnvi->annotated = True;
		}
		;

OptionKI		: KI_OPTION
		{
			theEnvi->kInverse = True;
		}
		;


OptionCCP	: CCP_OPTION
		{
			theEnvi->colorStepIncr = True;
		}
		;

OptionDPC	: DPC_OPTION Num
		{
			defaultPartColor = $<n>2.i;	
		}
		;

OptionDEC	: DEC_OPTION Num
		{
			defaultEdgeColor = $<n>2.i;	
		}
		;


OptionD		: D_OPTION Num
		{
			theEnvi->dimensions = $<n>2.i;
		}
		;

OptionR		: R_OPTION Num
		{
			theEnvi->randInit = $<n>2.i;
		}
		;

OptionLO	: LASTONLY_OPTION
		{
			theEnvi->lastOnly = True;
		}
		;

OptionSource	: SOURCE_OPTION STRING
		{
		#if USETCL
			int result;
			if (! theEnvi->interp)
			{
				cout << "/source -> Creating interpreter" << endl;
				theEnvi->interp = Tcl_CreateInterp ();
				Tcl_Init (theEnvi->interp);
			}
			result = Tcl_EvalFile (theEnvi->interp, $<s>2);
			if (result != TCL_OK)
			{
				char msg[1024];
				sprintf (msg, "Error in script %s: %s", $<s>2, theEnvi->interp->result);
				yyerror (msg);
			}
		#else
			cerr << "CollageSystem: Warning: /source option without activated scripting support is ignored." << endl;
		#endif
		}
		;

OptionEval	: EVAL_OPTION STRING
		{
		#if USETCL
			int result;
			if (! theEnvi->interp)
			{
				cout << "/eval -> Creating interpreter" << endl;
				theEnvi->interp = Tcl_CreateInterp ();
				Tcl_Init (theEnvi->interp);
			}
			result = Tcl_Eval (theEnvi->interp, $<s>2);
			if (result != TCL_OK)
			{
				char msg[1024];
				sprintf (msg, "Error in /eval: %s", theEnvi->interp->result);
				yyerror (msg);
			}
		#else
			cerr << "CollageSystem: Warning: /eval option without activated scripting support is ignored." << endl;
		#endif
		}
		;

OptionCC	: CC_OPTION
		{
			theEnvi->colourContext = True;
		}
		;
		
OptionMP	: MP_OPTION
		{
			theEnvi->mergeParts = True;
		}
		;
		

/****** Bounding box options *************************************************/ 

OptionBBox	: BBOX_OPTION 
		{
			if (! theEnvi->bboxCalc)
			{
				theEnvi->bboxCalc = new BoundingBoxCalculator;
				theEnvi->bboxCalc->SetOptions (theEnvi->bboxList);
			}
			curOptBBox = new OptionBBox;
			curOptBBox->option = BBoxOption ($<i>1);
		}
		  DoOptionBBox
		{
			theEnvi->bboxList.Append (curOptBBox);
		}
		;

DoOptionBBox	: 
		{
			curPictSet = &curOptBBox->selected;
				// (assign new intervals to this picture set.)
		}
		  SelectedPics BBOX_OPERATION 
		
		{	
			curPictSet = &curOptBBox->operands;
				// (assign new intervals to this picture set.)
		} 
		  OperandPics
		{
			curOptBBox->operation = (BBoxOperation)$3; 
		}
		;

SelectedPics	: SelectedPics PictListItem
		| /* empty */
		;

OperandPics	: OperandPics XPictListItem
		| /* empty */
		;

PictListItem	: STRING
		{
			curPictSet->NewGroup (yytextbuf[0]);
		}
		  XNumberList
		;

XPictListItem	: PictListItem
		| LBRACKET Num COORDSEP Num COORDSEP Num COORDSEP Num RBRACKET
		{
			curPictSet->NewGroup ('B');
			curPictSet->NewInterval (pseudoPicCnt, pseudoPicCnt);
			
			BoundingBox bb(2);
			bb.GetAt(0)->min = $<n>2.r;
			bb.GetAt(0)->max = $<n>6.r;
			bb.GetAt(1)->min = $<n>4.r;
			bb.GetAt(1)->max = $<n>8.r;
			theEnvi->bboxCalc->SetPicture ('B', pseudoPicCnt++, bb);
		}


XNumberList	: XNumberList COORDSEP XNumListItem
		| XNumListItem
		; 

XNumListItem	: XNumber
		{
			curPictSet->NewInterval ($<i>1, $<i>1);
		}
		| XNumber UPTO XNumber
		{
			curPictSet->NewInterval ($<i>1, $<i>3);
		}
		;

XNumber		: Num
		{
			$$ = $<n>1.i;
		}
		| LASTPIC
		{
			$$ = BBLastPic;
		}
		;




/****** Grammar part *********************************************************/


TbGlobal	: 
		{	curTextblock = NULL; }
		  Textblock
		{	theEnvi->CG->psbGlobal = curTextblock; 
		}
		;



Grammarpart	: PtOperations GRAMMAR OptTransforms
		{
			if (curTrans)
				defaultTrans = curTrans;
			else
				// default-default is affine transf.:
				defaultTrans = new AffineFinder (theEnvi->dimensions);
		}
		  TbGrammar StartCollage Rules
		;

TbGrammar	: 
		{	curTextblock = NULL; }
		  Textblock
		{	theEnvi->CG->psbGrammar = curTextblock; }
		;

StartCollage	: START TbStart Collage
		{	
			theEnvi->CG->SetStart (curCollage[curInd]); // curInd == 0 due to initialization
			// don't deallocate curCollage -- pointer is stored.	
		}
		;

TbStart		: 
		{	curTextblock = NULL; }
		  Textblock
		{	theEnvi->CG->psbStart = curTextblock; }
		;


Rules		: Rules Rule
		| /* empty */
		;

Rule		: RULE Label 
		{
		  strcpy (curRuleLabel, $<s>2);
		  curInd = 0;
		}
		  RuleOptions TbRule Collage Context NContext Tablenumber
		{
			if (! curTrans)
				curTrans = defaultTrans->Copy();	
			CollageRule *rule = new CollageRule (curRuleLabel, curCollage[0], 
			                                     curCollage[1], curCollage[2],
			                                     curAnyOf[0], curAnyOf[1],
			                                     curDelContext,
			                                     curTrans, $<n>9.i, $<n>4.r);
			// don't deallocate curCollage[] -- pointers are stored.	
			rule->psbRule = curTextblock;
			theEnvi->CG->InsertRule (rule); 
		}
		;


RuleOptions	: Transformations OptRuleProb
		{	$$ = $2; }
		| RuleProb OptTransforms
		{	$$ = $1; }
		| /* empty */
		{	curTrans = 0;
			$<n>$.r = 1.0; }
		;

OptRuleProb	: /* empty */
		{	$<n>$.r = 1.0; }
		| RuleProb
		;

RuleProb	: PROB Num
		{
			$$ = $2;
		}
		;

TbRule		: 
		{	curTextblock = NULL; }
		  Textblock
		  /* pointer is copied in nonterminal Rule (see above) */
		;



Tablenumber	: TABLE Num
		{
			$$ = $<n>2.i;
			if ($$ == 0)
				yyerror ("Table numbers from 1..n or 'global' instead of 'table <n>'");
		}
		| GLOBAL
		{
			$$ = 0;
		}
		;
		
Context		: CONTEXT
		{	curInd = 1;
			curCollage[1] = 0;
		}
			OptAnyOf OptDelContext Collage
		|	/* empty */
		{
			curCollage[1] = 0;
			curAnyOf[0] = False; // but this is ignored anyway
		}
		;
		
NContext	: NEGATIVE CONTEXT
		{	curInd = 2;
			curCollage[2] = 0;
		}
			OptAnyOf Collage
		|	/* empty */
		{
			curCollage[2] = 0;
			curAnyOf[1] = False; // but this is ignored anyway
		}
		;
	
OptAnyOf	: ANY OF
		{
			curAnyOf[curInd-1] = True;
		}
			| /* empty */
		{
			curAnyOf[curInd-1] = False;
		}
		;
		
OptDelContext	: DELETE
		{
			curDelContext = True;
		}
				| /* empty */
		{
			curDelContext = False;
		}
		;			

Collage		: PtOperations
		{
			curPins[curInd] = 0;
			curPartSet[curInd] = 0;
			curEdgeSet[curInd] = 0;
		}
		CollageContent 
		{
			if (! curPins[curInd]) 
				curPins[curInd] = new PinSequence;
			if (! curPartSet[curInd]) 
				curPartSet[curInd] = new PartSet;
			if (! curEdgeSet[curInd])
				curEdgeSet[curInd] = new EdgeSet;
			
			curCollage[curInd] = new Collage (curPins[curInd], curPartSet[curInd],
			                                  curEdgeSet[curInd]);
			// don't deallocate curPins, curPartSet, curEdgeSet.
		}
		;

CollageContent	: Pins PtOperations CollageContent 
		| OneEdge PtOperations CollageContent 
		| OnePart PtOperations CollageContent 
		| PtOperations ENDCOLLAGE 
		; 

Pins		: PINS Points
		{
			if (curPins[curInd])
				yyerror ("Pin-points defined twice");
			else {
				curPins[curInd] = new PinSequence;
				*curPins[curInd] = *curPS; // elementwise copy
				delete curPS;
			}
		}
		;


OneEdge		: EDGE Label Points OptEdgeColor
		{
			if (! curEdgeSet[curInd])
				curEdgeSet[curInd] = new EdgeSet;
			Hyperedge *edge = new Hyperedge ($<s>2);
			edge->SetColor ((ColorIndex)$<n>4.i);
			*edge = *curPS; // elementwise copy
			delete curPS; 
			curEdgeSet[curInd]->Include (edge); 
		}
		;

OptEdgeColor	: COLOR Num
		{
			$<n>$.i = $<n>2.i;
		}
		| /* empty */
		{
			$<n>$.i = defaultEdgeColor;
		}
		;


OnePart		: Part OptPartColor OptColorProc
		{
			if (! curPartSet[curInd])
				curPartSet[curInd] = new PartSet;
			curPartSet[curInd]->Include (*curPart);
		}
		;

OptPartColor	: COLOR Num
		{
			curPart->SetColor ($<n>2.i);
		}
		| RGB LBRACKET Num COORDSEP Num COORDSEP Num RBRACKET
		{
			curPart->SetRGB (RGBValue($<n>3.i, $<n>5.i, $<n>7.i));
		}
		| /* empty */
		{
			curPart->SetColor (defaultPartColor);
		}
		;

OptColorProc	: COLORPROC STRING
		{
			SString<char> procname ($<s>2);
			curPart->SetColorProc (procname);
		}
		| /* empty */
		;

Part		: Dot
		| Line
		| DashLine
		| Polyline
		| Polygon
		| Circle
		| Ellipse
		| Sphere
		| Cube
		| Cone
		| Cylinder
		| PointLight
		| Native
		;


Dot		: DOT Point
		{
			Dot *dot = new Dot (*$<p>2);
			delete $<p>2;
			curPart = dot;
		}
		;

Line		: LINE Point Point
		{
			Line *line = new Line (*$<p>2, *$<p>3);
			delete $<p>2; 
			delete $<p>3;
			curPart = line;
		}
		;

DashLine	: DLINE Point Point
		{
			DashLine *dline = new DashLine (*$<p>2, *$<p>3);
			delete $<p>2;
			delete $<p>3;
			curPart = dline;
		}
		;

Polyline	: PLINE Point Point Points
		{
			PolyLine *pline = new PolyLine (*curPS);
			pline->Insert (*$<p>3);
			pline->Insert (*$<p>2);
			delete $<p>2;
			delete $<p>3;
			curPart = pline;
		}
		;

Polygon		: PolygonKind Point Point Points
		{
			Polygon *pgon = new Polygon (*curPS, (Boolean)$<i>1);
			pgon->Insert (*$<p>3);
			pgon->Insert (*$<p>2);
			delete $<p>2;
			delete $<p>3;
			curPart = pgon;
		}
		;

PolygonKind	: PGON
		{	$$ = 0; }
		| FPGON
		{	$$ = 1; }
		;

Circle		: CircleKind Point Point
		{
			Ellipse *circle = new Ellipse (*$<p>2, *$<p>3, (Boolean)$<i>1);
			delete $<p>2;
			delete $<p>3;
			curPart = circle;
		}
		;

CircleKind	: CIRC
		{	$$ = 0; }
		| FCIRC
		{	$$ = 1; }
		;

Ellipse		: EllipseKind Point Point Point Point Angle COORDSEP Angle
		{
			Ellipse *ellipse = new Ellipse (
			  Line (*$<p>2, *$<p>3),
			  Line (*$<p>4, *$<p>5),
			  (Boolean)$<i>1, $<n>6.r, $<n>7.r);
			delete $<p>2;
			delete $<p>3;
			delete $<p>4;
			delete $<p>5;
			curPart = ellipse;
		}
		;

EllipseKind	: ELL
		{	$$ = 0; }
		| FELL
		{	$$ = 1; }
		;
		
Sphere	: SPHERE Num
		{
			Sphere* sphere = new Sphere ($<n>2.r);
			PtTransformFigure (sphere);
			curPart = sphere;
		}
		;
		
Cube	: CUBE Num Num Num
		{
			Cube* cube = new Cube ($<n>2.r, $<n>3.r, $<n>4.r);
			PtTransformFigure (cube);
			curPart = cube;
		}
		;
		
Cone	: CONE ObjectParts Num Num 
		{
			Cone* cone = new Cone ($<n>3.r, $<n>4.r, (Cone::Parts)$<n>2.i);
			PtTransformFigure (cone);
			curPart = cone;
		}
		;	

Cylinder	: CYLINDER ObjectParts Num Num 
		{
			Cylinder* cylinder = new Cylinder ($<n>3.r, $<n>4.r, (Cylinder::Parts)$<n>2.i);
			PtTransformFigure (cylinder);
			curPart = cylinder;
		}
		;	
				
ObjectParts	: PART_ALL {$$ = 0;}
			| PART_SIDES {$$ = 1;}
			| PART_BOTTOM {$$ = 2;}
			| PART_TOP {$$ = 3;}
			;
		
Native	: NATIVE 
		{
			curTextblock = NULL;
		}
		  Textblock
		{
			Native* native = new Native (curTextblock);
			PtTransformFigure (native);
			curPart = native;
		}
		;		
		
PointLight : POINTLIGHT Point
		{
			PointLight* light = new PointLight (*$<p>2);
			delete $<p>2;
			curPart = light;
		}
		;			

Label		: STRING {$$ = $<s>1;}
		;


Points		: 
		{
			curPS = new PointSequence;
		}
		  DoPoints 
		;

DoPoints	: Point 
		{
			curPS->Append (*$<p>1);
			delete $<p>1;
		}
		  DoPoints

		| PtOperations
		;

Point		:	PtOperations transPoint 
			{
				$$ = $2;
			}
			;
			

iPoints		: 
		{
			curiPS = new PointSequence; 
		}
		  DoiPoints
		  
		;

DoiPoints	: internalPoint 
		{
			curiPS->Append (*$<p>1);
			delete $<p>1;
		}
		  DoiPoints
		| /* empty */
		;
		
internalPoint 	:  LBRACKET Num COORDSEP Num RBRACKET
		{
			$<p>$ = new Point (2/*theEnvi->dimensions*/);	
			$<p>$->Set (0, $<n>2.r);
			$<p>$->Set (1, $<n>4.r);
		#if 0
			if (theEnvi->dimensions > 2) 
			{
				cout << "Grammar: 2-D points conversion to " << 
				theEnvi->dimensions << "-D, padding with 0's." << endl;
				for (int d=2; d < theEnvi->dimensions; d++)
					$<p>$->Set (d, 0);
			}
		#endif
		}
		|  LBRACKET Num COORDSEP Num COORDSEP Num RBRACKET
		{
			$<p>$ = new Point (3/*theEnvi->dimensions*/);	
			$<p>$->Set (0, $<n>2.r);
			$<p>$->Set (1, $<n>4.r);
			$<p>$->Set (2, $<n>6.r);
			if (theEnvi->dimensions < 3)
				yyerror ("3-D points only allowed in 3-D mode");
		}
		 
		| FKT_ROT3D LBRACKET internalPoint internalPoint Num RBRACKET
		{
			Point null (3);
			null.Set(0,0); null.Set(1,0); null.Set(2,0);
			$<p>$ = new Point (3);
			Rotate3d ($<p>3, &null, $<p>4, $<n>5.r, $<p>$);
		}		
		;
			
transPoint	:  LBRACKET Num COORDSEP Num RBRACKET
		{
			$<p>$ = new Point (2/*theEnvi->dimensions*/);	
			$<p>$->Set (0, $<n>2.r);
			$<p>$->Set (1, $<n>4.r);
		#if 0	
			if (theEnvi->dimensions > 2) 
			{
				cout << "Grammar: 2-D points conversion to " << 
				theEnvi->dimensions << "-D, padding with 0's." << endl;
				for (int d=2; d < theEnvi->dimensions; d++)
					$<p>$->Set (d, 0);
			}
		#endif
			PtTransform ($<p>$);
		}
		| OldNum COORDSEP OldNum  /* backward compatibility syntax, same as above */
		{
			$<p>$ = new Point (2/*theEnvi->dimensions*/);	
			$<p>$->Set (0, $<n>1.r);
			$<p>$->Set (1, $<n>3.r);
		#if 0	
			if (theEnvi->dimensions > 2) 
			{
				cout << "Grammar: 2-D points conversion to " << 
				theEnvi->dimensions << "-D, padding with 0's." << endl;
				for (int d=2; d < theEnvi->dimensions; d++)
					$<p>$->Set (d, 0);
			}
		#endif
			PtTransform ($<p>$);
		}
		|  LBRACKET Num COORDSEP Num COORDSEP Num RBRACKET
		{
			$<p>$ = new Point (3/*theEnvi->dimensions*/);	
			$<p>$->Set (0, $<n>2.r);
			$<p>$->Set (1, $<n>4.r);
			$<p>$->Set (2, $<n>6.r);
			if (theEnvi->dimensions < 3)
				yyerror ("3-D points only allowed in 3-D mode");
			PtTransform ($<p>$);
		}
		|  OldNum COORDSEP OldNum COORDSEP OldNum /* see above */
		{
			$<p>$ = new Point (3/*theEnvi->dimensions*/);	
			$<p>$->Set (0, $<n>1.r);
			$<p>$->Set (1, $<n>3.r);
			$<p>$->Set (2, $<n>5.r);
			if (theEnvi->dimensions < 3)
				yyerror ("3-D points only allowed in 3-D mode");
			PtTransform ($<p>$);
		}
		;

Angle		: Num	
		;

OptTransforms	: Transformations
		| /* empty */
		{	curTrans = 0;}
		;

Transformations : TRANSFORMATIONS Transformation
                ;
                               
Transformation  : GeneralTrans
		;

GeneralTrans	: AffineTrans
                | Translations
                | ScaleTranslations
		;

AffineTrans     : T_AFFINE Num
                {
			curTrans = new AffineFinder ($<n>2.i);
                }                
                ;
              
Translations    : T_TRANSLATIONS Num
                {
			curTrans = new TranslationFinder ($<n>2.i);
                }
                ;

ScaleTranslations : T_SCALETRANSLATIONS Num
                {
			curTrans = new ScaleTranslationFinder ($<n>2.i);
                }
                ;
                
                
/*** Point transformations ***/    

PtOperations	: /* empty */
				| LBRACE SomePtOperations RBRACE PtOperations
				;
				
SomePtOperations	: /* empty */
				| PtOperation SomePtOperations
				;

PtOperation		: PTPUSH 
				{
					ptInd++;
					if (ptInd >= PT_SIZE)
						yyerror ("Point transformation stack size exceeded (PT_SIZE in colgram.y).");	
				}
				  NewPointTrans 
					
				| PTPOP 
				{
					if (ptInd < 0)
						yyerror ("POP on empty point transformation stack.");
					delete ptStack[ptInd];
					ptStack[ptInd] = 0; // just to uncover illegal access.
					ptInd--;
				}
				;
				
NewPointTrans	:	MAP T_AFFINE iPoints
			{
				if (curPTS1)
					delete curPTS1;
				curPTS1 = curiPS;
			}
			TO iPoints
			{
				if (curPTS2)
					delete curPTS2;
				curPTS2 = curiPS;			
				Point *ptest = curPTS1->GetAt (0);
				if (! ptest)
					yyerror ("You must provide at least one point in each internal point list.");
				AffineFinder af (ptest->GetDim());
				Transformation *t = af.FindTrans (*curPTS1, *curPTS2);
				if (! t)
					yyerror ("ptrans map affine: transformation not found.");	
				else
				{
					ptStack[ptInd] = t;
				}	
			}

			;

                
Derivationpart	: DERIVATION TbDerivation TbDerivStart DerivEntries TableInfos
		;

TbDerivation	: 
		{	curTextblock = NULL; }
		  Textblock
		{	theEnvi->derivationTable->psbDerivation = curTextblock; }
		;

TbDerivStart	: /* empty */
		| START 
		{	curTextblock = NULL; }
		  Textblock
		{ 	theEnvi->derivationTable->psbDerivStart = curTextblock; }
		

DerivEntries	: DerivEntries DerivEntry
		| /* empty */
		;

DerivEntry	: UNTIL Num DO Num TbTableEntry 
		{
			TableElement *tabElem = new TableElement ($<n>2.i, curTextblock);
			TableElemEntry entry (1, $<n>4.i);
			tabElem->entries.Append (entry);
			if (! theEnvi->derivationTable->Insert (tabElem))
			{
				delete tabElem;
				yyerror ("Duplicate derivation entry");
			}
		}
		| UNTIL Num REPEAT TableElemEntries TbTableEntry
		{
			TableElement *tabElem = new TableElement ($<n>2.i, curTextblock);
			tabElem->entries = *curTableElemEntries;

			if (! theEnvi->derivationTable->Insert (tabElem))
			{
				delete tabElem;
				yyerror ("Duplicate derivation entry");
			}
			delete curTableElemEntries;
			curTableElemEntries = NULL;
		}
		;

TableElemEntries: TabElemEntry
		| TableElemEntries COORDSEP TabElemEntry
		;

TabElemEntry	: Num TIMES Num
		{
			if (! curTableElemEntries)
				curTableElemEntries = new SList<TableElemEntry>;
			if ($<n>1.i > 0)
			{
				TableElemEntry entry ($<n>1.i, $<n>3.i);
				curTableElemEntries->Append (entry);
			}
		}
		;

TbTableEntry	: 
		{	curTextblock = NULL; }
		  Textblock
		  /* pointer is copied in nonterminal DerivEntry (see above) */
		;


TableInfos	: /* empty */
		   | TableInfo TableInfos
		   ;
	
TableInfo	: TABLEINFO Num
		{
			curTableInfo = new TableInfo ($<n>2.i);
		}
			TableInfoEntries
		{
			theEnvi->tableInfoList.Append (curTableInfo); // pointer stored
		}
			;
			
TableInfoEntries : /* empty */
			| TableInfoEntry TableInfoEntries
			;
			
TableInfoEntry	: SEQUENTIAL Num
			{
				curTableInfo->SetSequential ((Boolean)$<n>2.i);
			}
			|	EXHAUSTPROB Num
			{
				curTableInfo->SetExhaustProb ((Boolean)$<n>2.i);
			}
			|	ONEFROMBASE Num
			{
				curTableInfo->SetOneFromBase ((Boolean)$<n>2.i);
			}
			;
			

Textblock	: /* empty */
		| BEGIN_ATTRIBUTES 
		{
			curTextblock = new Textblock;
		}
		  Strings END_ATTRIBUTES
		;

Strings		: /* empty */
		| STRING 
		{
			char *str = $<s>1, *copy;
			copy = (char *)malloc (strlen(str)+1);
			// note: this is currently never deallocated!			
			strcpy (copy, str);
			curTextblock->Append (copy);
		}
		  Strings
	 	;


OldNum		: NUM 
		{
			$<n>$.i = $<n>1.i;
			$<n>$.r = $<n>1.r;
		}
		| MINUS NUM
		{
			$<n>$.i = -$<n>2.i;
			$<n>$.r = -$<n>2.r;
		}
		;

Num		: Product 
		{
			$<n>$.i = $<n>1.i;
			$<n>$.r = $<n>1.r;
		}
		| Product PLUS Num
		{
			$<n>$.i = $<n>1.i + $<n>3.i;
			$<n>$.r = $<n>1.r + $<n>3.r;
		}
		| Product MINUS Num 
		{
			$<n>$.i = $<n>1.i - $<n>3.i;
			$<n>$.r = $<n>1.r - $<n>3.r;
		}
		;

Product		: Term
		| Term MULTIPLY Product
		{
			$<n>$.i = $<n>1.i * $<n>3.i;
			$<n>$.r = $<n>1.r * $<n>3.r;
		}
		| Term SLASH Product
		{
			if ($<n>3.i != 0) 
				$<n>$.i = $<n>1.i / $<n>3.i;
			else {
				cerr << "CollageSystem warning: integer division by zero, fixing..." << endl;
				if ($<n>1.i > 0) 
					$<n>$.i = LONG_MAX;
				else if ($<n>1.i < 0)
					$<n>$.i = LONG_MIN;
				else
					$<n>$.i = 0;
			}

			if ($<n>3.r != 0.0) 
				$<n>$.r = $<n>1.r / $<n>3.r;
			else {
				cerr << "CollageSystem warning: floating point division by zero, fixing..." << endl;
				if ($<n>1.r > 0.0) 
					$<n>$.r = DBL_MAX;
				else if ($<n>1.r < 0)
					$<n>$.r = DBL_MIN;
				else
					$<n>$.r = 0.0;
			}
		}
		;

Term		: NUM 
		{
			$<n>$ = $<n>1;
		}
		| MINUS Term /* unary minus */
		{	
			$<n>$.i = -$<n>2.i;
			$<n>$.r = -$<n>2.r;
		}

		| DOLLAR STRING
		{
		#if USETCL
			if (! theEnvi->interp) 
			{
				yyerror ("Attempt to substitute variable without any script /source'd");
			}
			else	
			{
				char *varname = $<s>2, *content;
				content = Tcl_GetVar (theEnvi->interp, varname, TCL_GLOBAL_ONLY);
				if (! content)
					yyerror ("Variable has not been defined by any script.");
				$<n>$.i = atoi (content);
				$<n>$.r = atof (content);
			}
		#else
			cout << "CollageSystem: WARNING: no scripting support, Substituting 0 for expression $" << $<s>2 << endl;
			$<n>$.i = 0;
			$<n>$.r = 0.0;
		#endif
		}
		| LBRACKET Num RBRACKET
			{$<n>$ = $<n>2;}
		| FKT_SQRT LBRACKET Num RBRACKET
		{	//$<n>$.i = (int)sqrt ($<n>3.i);
			$<n>$.r = sqrt ($<n>3.r);
		}
		| FKT_SIN LBRACKET Num RBRACKET
		{	//$<n>$.i = (int)sin ($<n>3.i);
			$<n>$.r = sin ($<n>3.r);
		}
		| FKT_COS LBRACKET Num RBRACKET
		{	//$<n>$.i = (int)cos ($<n>3.i);
			$<n>$.r = cos ($<n>3.r);
		}
		| FKT_TAN LBRACKET Num RBRACKET
		{	//$<n>$.i = (int)tan ($<n>3.i);
			$<n>$.r = tan ($<n>3.r);
		}
		| FKT_ACOS LBRACKET Num RBRACKET
		{	//$<n>$.i = (int)acos ($<n>3.i);
			$<n>$.r = acos ($<n>3.r);
		}
		| FKT_ASIN LBRACKET Num RBRACKET
		{	//$<n>$.i = (int)asin ($<n>3.i);
			$<n>$.r = asin ($<n>3.r);
		}
		| FKT_ATAN LBRACKET Num RBRACKET
		{	//$<n>$.i = (int)atan ($<n>3.i);
			cerr << "atan(" << $<n>3.r << ") = ";
			$<n>$.r = atan ($<n>3.r);
			cerr << $<n>$.r << endl;
		}
		| FKT_COSH LBRACKET Num RBRACKET
		{	//$<n>$.i = (int)cosh ($<n>3.i);
			$<n>$.r = cosh ($<n>3.r);
		}
		| FKT_SINH LBRACKET Num RBRACKET
		{	//$<n>$.i = (int)sinh ($<n>3.i);
			$<n>$.r = sinh ($<n>3.r);
		}
		| FKT_TANH LBRACKET Num RBRACKET
		{	//$<n>$.i = (int)tanh ($<n>3.i);
			$<n>$.r = tanh ($<n>3.r);
		}
		| FKT_EXP LBRACKET Num RBRACKET
		{	//$<n>$.i = (int)exp ($<n>3.i);
			$<n>$.r = exp ($<n>3.r);
		}
		| FKT_LOG LBRACKET Num RBRACKET
		{	//$<n>$.i = (int)log ($<n>3.i);
			$<n>$.r = log ($<n>3.r);
		}
		| FKT_LOG10 LBRACKET Num RBRACKET
		{	//$<n>$.i = (int)log10 ($<n>3.i);
			$<n>$.r = log10 ($<n>3.r);
		}
		;
%%

/* There seems to be no standard version of yyerror, so I
   implement one here: */

void yyerror (char *s)
{
	cout << "Grammar: " << s << " -- near line " << yylineno << endl;
	sprintf (yytextbuf, "%s -- near line %i.", s, yylineno);
	longjmp (csjmpbuf, -1);
}


