//
// collage.cc  --  CollageTwo
//
// Classes for representing a collage
//
// Author            : Nils Schwabe, Stefan Taubenberger
// Date of creation  : 19 Apr 94
// Last modification : 10 Jun 96
//
// University of Bremen, Germany
//

// ------- Includes -----------------------------------------------------------

#include <stdio.h>
#include <time.h>
#include <stdlib.h>
#include <iostream.h> 
#include "psout.h"
#include "vrout.h"
#include "collage.h"
#include "trans.h"
#include "cgi.h"


// ------- PinSequence --------------------------------------------------------

void PinSequence::Draw() const
{
	Point *cur = GetAt(0);
	int i=0;
	Coordinate mx=0, my=0;

	while (cur) {
		i++;
		mx += cur->Get(0);
		if (cur->GetDim() >= 2)
			my += cur->Get(1);
		else
			my += DrawDim1Y;
		cur = GetNext();
	}
	if (i == 0) return;
	mx /= i;
	my /= i;	
		
	i = 0;	
	cur = GetAt (0);	
	while (cur) {
		i++;
		Coordinate x,y,dx,dy,b;
		x = cur->Get(0);
		if (cur->GetDim() >= 2)
			y = cur->Get(1);
		else
			y = DrawDim1Y;
		theScreen->Ellipse (x, y, 4, 4, False);

		dx = x-mx;
		dy = y-my;
		if ((b = sqrt (dx*dx+dy*dy)) > 0) {
			dx /= b;
			dy /= b;
		}	
		x += dx*6;
		y += dy*6;
		char pinnr[5];
		sprintf (pinnr, "%i", i);
		theScreen->Text (x, y, pinnr);
		cur = GetNext();
	}
}

void PinSequence::Transform (const Transformation &T)
{
	Point *cur = GetAt(0);
	while (cur) {
		T.Transform (*cur, cur);
		cur = GetNext();
	}
}



// ------- Collage ------------------------------------------------------------

Collage::Collage (PinSequence *initPins,
	          PartSet *initParts,
	          EdgeSet *initEdges)
{
	bb = NULL;
	pins = initPins;
	parts = initParts;
	edges = initEdges;
}

Collage::Collage ()
{
	bb = NULL;
	pins = new PinSequence;
	parts = new PartSet;
	edges = new EdgeSet;
}

Collage::Collage (const Collage &theCollage) 
{
	bb = NULL;
	pins = new PinSequence;
	parts = new PartSet;
	edges = new EdgeSet;
	(*pins) = *(theCollage.pins);
	(*parts) = *(theCollage.parts);
	(*edges) = *(theCollage.edges);
}

Collage::~Collage () 
{
	delete edges;
	delete parts;
	delete pins;
	if (bb) delete bb;
}

Collage &Collage::operator= (const Collage &theCollage) 
{
	if (this != &theCollage) {
		if (bb) delete bb;
		bb = NULL;
		(*pins) = *(theCollage.pins);
		(*parts) = *(theCollage.parts);
		(*edges) = *(theCollage.edges);
	};
	return *this;
}


void Collage::AppendPin (const Point &thePoint) 
{
	pins->Append ((Point &)thePoint);
}

void Collage::InsertPart (const Part &thePart) 
{
	parts->Include (thePart);
}

void Collage::InsertEdge (Edge *theEdge) 
{
	edges->Include (theEdge);
}


void Collage::Transform (const Transformation &T) 
{
	edges->Transform (T);
	pins->Transform (T);
	parts->Transform (T);  
};

void Collage::ReplEdge (Edge *edge, 
                        const Collage &repl,
                        Transformation *t,
                        ColorIndex colorOffset,
                        Boolean kInverse,
                        Collage* delContext
#if USETCL
	                ,Tcl_Interp *interp
#endif
                        )
{
#if USETCL
	char script[300], curCol[10];
#endif

	//
	// Take the right-hand-side out of the box
	//
	Collage c (repl);  
		// makes a deep copy; this should be no serious performance
		// problem since 'repl' is only a rule-collage. 
	
	//
	// Normal or Kreo's inverse replacement
	//
	if (kInverse)
		this->Transform(*t);		
	else
		c.Transform (*t);

	//
	// Delete context (Type-0 grammar) ?
	//	
	if (delContext) {
		Collage transDelContext(*delContext); // deep copy
		transDelContext.Transform(*t);
		RemoveSubCollage(transDelContext);
	}
	
	//
	// Replace the edge
	//
	edge->FreeTrans (t);	
	edges->Exclude (edge);
			
	// color stuff
	{
		Part *cur = c.parts->GetAt(0);
		while (cur) 
		{
			// cout << cur->GetColorProc().Get() << endl;
			if (cur->GetColorProc().Len())
			{
#if USETCL
				// Calculate color with Tcl proc:
				sprintf (curCol, "%i", (int)cur->GetRGB().Red());
				Tcl_SetVar (interp, "_cs_red", curCol, TCL_GLOBAL_ONLY);
				sprintf (curCol, "%i", (int)cur->GetRGB().Green());
				Tcl_SetVar (interp, "_cs_green", curCol, TCL_GLOBAL_ONLY);
				sprintf (curCol, "%i", (int)cur->GetRGB().Blue());
				Tcl_SetVar (interp, "_cs_blue", curCol, TCL_GLOBAL_ONLY);
				sprintf (script, "%s %i _cs_red _cs_green _cs_blue",
				                 cur->GetColorProc().Get(),
				                 (int)colorOffset);
				int result = Tcl_Eval (interp, script);
				if (result != TCL_OK)
					cerr << interp->result << endl;
				else
				{
					char *newVal;
					RGBValue rgb;
					newVal = Tcl_GetVar (interp, "_cs_red", TCL_GLOBAL_ONLY);
					rgb.SetRed (atoi(newVal));
					newVal = Tcl_GetVar (interp, "_cs_green", TCL_GLOBAL_ONLY);
					rgb.SetGreen (atoi(newVal));
					newVal = Tcl_GetVar (interp, "_cs_blue", TCL_GLOBAL_ONLY);
					rgb.SetBlue (atoi(newVal));
					cur->SetRGB (rgb);
				}
#endif
			}
			else
			{
				if (! cur->UsesRGB())
				{
					// Old-fashioned index increment:

					cur->SetColor (cur->GetColor()+colorOffset);
				}
			}
			cur = c.parts->GetNext();
		}
	}
	
	// in either case, add 'c' to the collage
	
	Add (c); // again, makes deep copy of the parts and edges of 'c'.
	
	// c is local, so it will be destroyed now	
	
	//
	// Check number of objects treshold
	//
	if (parts->Length() + edges->Length() > csMaxObjects) {
		cout << "CollageSystem: Maximum number of generated objects exceeded." << endl;
		cout << "               The current maximum is " << csMaxObjects << "." << endl;
		cout << "               Either change this value and rebuild the system or change" << endl;
		cout << "               the current grammar." << endl;
		cout << "               THE SYSTEM IS NOW TERMINATING..." << endl;
		
		//
		// Generate error file
		//
		ofstream e("cserr.txt");
		e << "The CollageSystem stopped the derivation of this grammar because internal resource limits were exceeded." << endl;
		e.close();

		exit(-1000);
	}
	
}

void Collage::Add (const Collage &theCollage) 
{
	parts->Add (*theCollage.parts);
	edges->Add (*theCollage.edges);   
}


void Collage::DeriveWithBase 
	(	const Base &theBase,
		ColorIndex colorOffset,
		Boolean kInverse
#if USETCL
	        ,Tcl_Interp *interp
#endif
	)
{
	int count = 0;
	BaseElement* cur = theBase.GetAt(0);
	//cout << endl << "DeriveWithBase:" << endl;
	while (cur) {
		//cout << "  " << count << ". Edge: " << cur->target->GetLabel() << "(" << (long)cur->target << ")" << endl;
		//cout << "     Collage: " << (long)cur->source << endl;

		count ++;

		ReplEdge (cur->target, *cur->source, cur->trans, colorOffset, kInverse, cur->delContext
#if USETCL
		,interp
#endif
		);
		cur = theBase.GetNext();
	}
}

void Collage::Draw () const
{
	parts -> Draw();
	pins -> Draw();
	edges -> Draw();
}

void Collage::MakeBoundingBox (uint dim)
{
	if (bb) delete bb;
	bb = new BoundingBox (dim);
	parts->CalcBoundingBox (*bb);
	edges->CalcBoundingBox (*bb);
	pins->CalcBoundingBox (*bb);
}

void Collage::PSOut (Textblock *t1, Textblock *t2, Textblock *t3) const
{
	// WARNING: Code depending on this is placed in PSOut::PatchBox 
	thePS.Out ("\nBeginCollage");
	if (bb) {
		thePS.Out ("\n");
		thePS.Out (bb->GetAt(0)->min); // llx = min. of 1st dimension
		thePS.Out (bb->GetAt(1)->min); // lly = min. of 2nd dimension
		thePS.Out (bb->GetAt(0)->max); // urx = max. of 1st dimension
		thePS.Out (bb->GetAt(1)->max); // ury = max. of 2nd dimension
		thePS.Out ("_BBox\n");
		thePS.Out ("                                             \n");
	}
	else
		cerr << "Collage::PSOut: No bounding box present." << endl;
	// END WARNING

	thePS.Out (t1);
	thePS.Out (t2);
	thePS.Out (t3);
	
	thePS.Out ("\nBeginParts\n");
	parts->PSOut();
	thePS.Out ("\nEndParts\n");

	thePS.Out ("\nBeginEdges\n");
	edges->PSOut();
	thePS.Out ("\nEndEdges\n");

	thePS.Out ("\nBeginPins\n");
	pins->PSOut();
	thePS.Out ("P");
	thePS.Out ("\nEndPins");

	thePS.Out ("\nEndCollage\n");
}


void Collage::VROut () const
{	
	parts->VROut();
	edges->VROut();
	pins->VROut();
}


//----- Find context within a collage -----------------------------------------

Boolean Collage::FindParts (Collage &context, Boolean anyOf)
{
	Figure *contextPart = context.parts->GetAt (0);
	Boolean contains;
	
	if (! contextPart)
		return True; // empty context always matches
	
	while (contextPart)
	{
		contains = (parts->Contains (*contextPart) != 0);
		if ((! anyOf) && (! contains))
		{
#if 0		
			cout << "Non-matching part: kind = " << contextPart->Kind() << endl;
			theScreen->Clear ();
			contextPart->Draw ();	
			theScreen->PageTitle ("Non-matching part");
			theScreen->Clear ();
#endif			
			return False;
		}			
		if (anyOf && contains)
			return True;
		contextPart = context.parts->GetNext ();
	}
	return (! anyOf);
}

Boolean Collage::FindEdges (Collage &context, Boolean anyOf)
{
	int contextIndex = context.edges->GetIndex (),
	    selfIndex = edges->GetIndex ();
	Edge *contextEdge = context.edges->GetAt (0);
	Boolean contains;
	
	if (! contextEdge)
	{
		(void) edges->GetAt (selfIndex);
		(void) context.edges->GetAt (contextIndex);
		return True; // empty context always matches
	}
		
	int i=0;
	while (contextEdge)
	{
		contains = (edges->Contains (*contextEdge) != 0);
		if ((! anyOf) && (! contains))
		{
#if 0		
			theScreen->Clear ();
			contextEdge->Draw ();	
			theScreen->PageTitle ("Non-matching edge");
			theScreen->Clear ();
#endif		
			(void) edges->GetAt (selfIndex);
			(void) context.edges->GetAt (contextIndex);
			return False;
		}			
		if (anyOf && contains)
		{
			(void) edges->GetAt (selfIndex);
			(void) context.edges->GetAt (contextIndex);
			return True;		
		}
		contextEdge = context.edges->GetAt (++i);
	}
	(void) edges->GetAt (selfIndex);
	(void) context.edges->GetAt (contextIndex);
	return (! anyOf);
}

Boolean Collage::FindContext (Collage &context, Boolean anyOf)
{
	return FindParts (context, anyOf) && FindEdges (context, anyOf);
}


//-----	Remove a sub-collage from a collage -----------------------------------

void Collage::RemoveSubCollage(Collage& subCollage)
{
	//
	// Remove the parts
	//
	int cntSubParts = subCollage.parts->Length();
	Figure* subPart;
	Figure* thisPart;
	for (int subPartIdx = 0; subPartIdx < cntSubParts; subPartIdx++) {
		subPart = subCollage.parts->GetAt(subPartIdx);
		thisPart = parts->Contains(*subPart);
		if (thisPart)
			parts->Exclude(thisPart);
	}
	
	//
	// Remove the edges
	//
	int oldSubEdgeIdx = subCollage.edges->GetIndex();
	int oldThisEdgeIdx = edges->GetIndex();
	int cntSubEdges = subCollage.edges->Length();
	Edge* subEdge;
	Edge* thisEdge;
	for (int subEdgeIdx = 0; subEdgeIdx < cntSubEdges; subEdgeIdx++) {
		subEdge = subCollage.edges->GetAt(subEdgeIdx);
		thisEdge = edges->Contains(*subEdge);
		if (thisEdge)
			edges->Exclude(thisEdge);
	}
	(void) edges->GetAt(oldThisEdgeIdx);
	(void) subCollage.edges->GetAt(oldSubEdgeIdx);
}



// ----------------------------------------------------------------------------
// ------- CollageRule --------------------------------------------------------
// ----------------------------------------------------------------------------

CollageRule::CollageRule 
	(	const EdgeText& left, 
		Collage *right,
		Collage *cont,
		Collage *ncont,
		Boolean cAny,
		Boolean ncAny,
		Boolean cDel,
		TransFinder *tf, 
		int table,
		double p)
{
	leftHandText = left;
	leftHandClass = 0;
	rightHandSide = right;
	context = cont;
	ncontext = ncont;
	cAnyOf = cAny;
	ncAnyOf = ncAny;
	cDelete = cDel;
	tableNumber = table;
	psbRule = NULL;
	prob = p;
	transFinder = tf;
}

CollageRule::~CollageRule ()
{
	if (rightHandSide)
		delete rightHandSide;
	if (transFinder)
		delete transFinder;
	if (context)
		delete context;
	if (ncontext)
		delete ncontext;
}

void CollageRule::Draw () const
{
	char label[100], title[150];
	const BoundingBox *bb;
	MinMaxPair *minmax;
	Coordinate rhsMaxX,     // right-hand-side maximum x value
	           rhsXShift,   // right-hand-side x offset 
	           ymid,        // right-hand-side y-center
	           ctMinX,      // positive context minimum x value
	           ctMaxX,      // positive context maximum x value
	           ctXShift,    // positive context x offset
	           nctMinX,     // negative context minimum x value
	           nctXShift;   // negative context x offset
	           
	Collage drawRHS (*rightHandSide); // copy of right-hand-side used for drawing
	TranslationTrans translation (2); // a translation to display collages at the right place
	
	// Calculate bounding box of right-hand-side:
	rightHandSide->MakeBoundingBox (2);
	bb      = rightHandSide->GetBoundingBox ();
	minmax  = bb->GetAt (0); // extent in "x" direction	
	rhsMaxX = minmax->max;
	minmax  = bb->GetAt (1); // extent in "y" direction
	ymid    = (minmax->max - minmax->min) / 2;
	
	// Draw label:
	sprintf (label, "%s ::=", leftHandText.Get());
	theScreen->FgColor (255, 90, 90);
	theScreen->Text (0, ymid, label);
	
	// Draw right-hand-side collage:
	rhsXShift = 50;
	translation.offset.Set (0, rhsXShift);
	drawRHS.Transform (translation);
	drawRHS.Draw();
	
	// Draw coordinate axes for right-hand-side:
	theScreen->FgColor (255, 255, 255);
	theScreen->Line (rhsXShift,   -10, rhsXShift,    10);
	theScreen->Line (rhsXShift-10,  0, rhsXShift+10,  0);

	// Positive context ?	                 
	if (context)
	{
		Collage drawContext (*context); // copy of context collage used for drawing
				
		// Calculate bounding box of positive context collage:
		context->MakeBoundingBox (2);
		bb      = context->GetBoundingBox ();
		minmax  = bb->GetAt (0); // extent in "x" direction	
		ctMinX  = minmax->min;
		ctMaxX  = minmax->max;
		
		// Draw context collage:				
		ctXShift = rhsXShift + rhsMaxX - ctMinX + 100;
		translation.offset.Set (0, ctXShift);
		drawContext.Transform (translation);
		drawContext.Draw ();
		
		// Draw 'if' text:
		theScreen->FgColor (0, 255, 0);
		theScreen->Text (rhsXShift + rhsMaxX + 50, ymid, "if");
		if (cAnyOf)
			theScreen->Text (rhsXShift + rhsMaxX + 50, ymid-20, "ANY OF");

		// Draw coordinate axes for positive context collage:
		theScreen->Line (ctXShift,   -10, ctXShift,    10);
		theScreen->Line (ctXShift-10,  0, ctXShift+10,  0);
	}	 
	else
	{
		ctMaxX = rhsMaxX;
		ctXShift = rhsXShift;                
	}
	
	// Negative context ?	                 
	if (ncontext)
	{
		Collage drawContext (*ncontext); // copy of context collage used for drawing
		char *text;
				
		// Calculate bounding box of negative context collage:
		ncontext->MakeBoundingBox (2);
		bb       = ncontext->GetBoundingBox ();
		minmax   = bb->GetAt (0); // extent in "x" direction	
		nctMinX  = minmax->min;

		// Draw context collage:				
		nctXShift = ctXShift + ctMaxX - nctMinX + 100;
		translation.offset.Set (0, nctXShift);
		drawContext.Transform (translation);
		drawContext.Draw ();
		
		// Draw 'not' text:
		theScreen->FgColor (255, 0, 0);
		if (context)
			text = "and not";
		else
			text = "if not";
		theScreen->Text (ctXShift + ctMaxX + 50, ymid, text);
		if (ncAnyOf)
			theScreen->Text (ctXShift + ctMaxX + 50, ymid-20, "ANY OF");

		// Draw coordinate axes for negative context collage:
		theScreen->Line (nctXShift,   -10, nctXShift,    10);
		theScreen->Line (nctXShift-10,  0, nctXShift+10,  0);
	}	                 
		
	                 
	// Page labeling:
	if (tableNumber > 0)
		sprintf (title, "%s ::= ... class %i, table %i, prob %f, %s",
	                 leftHandText.Get(), leftHandClass, tableNumber, prob,
	                 transFinder->GetName());
	else
		sprintf (title, "%s ::= ... class %i, global, prob %f, %s",
	                 leftHandText.Get(), leftHandClass, prob,
	                 transFinder->GetName());

	theScreen->PageTitle (title);
	
	theScreen->Clear ();
}

void CollageRule::PSOut (Textblock *first, Textblock *second) const
{
	rightHandSide->MakeBoundingBox(2);
	const BoundingBox *bb = rightHandSide->GetBoundingBox();
	thePS.NewFile ("G", bb);
	rightHandSide->PSOut (first, second, psbRule);
	char tmp[10];
	sprintf (tmp,"(%s) LHS",leftHandText.Get());
	thePS.Out (tmp);
	thePS.Out ("\n");
	
	if (context)
	{
		context->MakeBoundingBox(2);
		bb = context->GetBoundingBox();
		thePS.NewFile ("G", bb, "_C");
		context->PSOut (first, second, psbRule);
	}
}

void CollageRule::VROut () const
{
	theVR.NewFile ("G");
	rightHandSide->VROut ();

	if (context)
	{
		theVR.NewFile ("G", "_C");
		context->VROut ();
	}
}


ostream& operator<< (ostream& s, const CollageRule &cr)
{
	return s << "RULE(" << cr.GetText() << "[" << cr.GetClass() << "] tab " 
	  << cr.GetTable() << " prob " << cr.GetProb() << ")";
}


// ------- CollageRuleSet -----------------------------------------------------

void CollageRuleSet::Insert (CollageRule *newElement)
{
	CollageRulePList::Append (newElement);
}


void CollageRuleSet::Draw () const
{
	CollageRule **cur = GetAt(0);
	while (cur) {
		(*cur)->Draw ();
		cur = GetNext();
	}
}

void CollageRuleSet::PSOut (Textblock *first, Textblock *second) const
{
	CollageRule **cur = GetAt(0);
	while (cur) {
		(*cur)->PSOut (first, second);
		cur = GetNext();
	}
}


void CollageRuleSet::VROut () const
{
	CollageRule **cur = GetAt(0);
	while (cur) {
		(*cur)->VROut ();
		cur = GetNext();
	}
}



// ----------------------------------------------------------------------------
// ------- Base ---------------------------------------------------------------
// ----------------------------------------------------------------------------

	
// ------- RuleProbSet (implementation class) ---------------------------------

class RuleProb
{
public:
	RuleProb (CollageRule *r, double p) {rule = r; prob = p;}
	CollageRule *rule;
	double prob;
};

class RuleProbSet : public SList<RuleProb>
{
public:
	void BuildSet (Edge *curEdge, CollageRuleSet *theRules, int table);

		// Builds a list of all rules together with their initial
		// probability, that match the class of 'curEdge' and belong
		// to 'table'.

	Boolean RemoveRule (CollageRule *rule);

		// Removes 'rule' and adjust all remaining probabilities.
		// Returns False if 'rule' was the last rule in the set.

};


void RuleProbSet::BuildSet (Edge *curEdge, CollageRuleSet *theRules, int table)
{
	CollageRule **curRule = theRules->GetAt(0);
		
	while (curRule) 
	{
		//--- same equivalence class ? ---
		if ( (*curRule)->GetClass() != curEdge->GetClass() )
		{
			curRule = theRules->GetNext();
			continue;
		}
				
		//--- correct table ? ---
		if ( (*curRule)->GetTable() != 0 && (*curRule)->GetTable() != table )
		{
			curRule = theRules->GetNext ();
			continue;
		}

		RuleProb rp (*curRule, (*curRule)->GetProb());
		Append (rp);
		curRule = theRules->GetNext ();
	}
}


Boolean RuleProbSet::RemoveRule (CollageRule *rule)
{
	// Delete rule from the list:
	RuleProb *rp = GetAt (0);
	while (rp && rp->rule != rule)
		rp = GetNext ();
	if (! rp)
	{
		cout << "CollageSystem: WARNING: Unexpected end of list in RuleProbSet::RemoveRule" << endl;
		return False;
	}
	double factor;
	if (rp->prob < 1.0)
		factor = 1.0 / (1.0 - rp->prob);
	else
		factor = 1.0; // don't modify incorrect probabilities
	Delete ();
	
	// Distribute the probability equally between all remaining rules:
	rp = GetAt (0);
	if (! rp)
		return False;
	while (rp)
	{
		rp->prob *= factor;
		rp = GetNext ();
	}		
	return True;
}

// ------- Base ---------------------------------------------------------------

double Base::GetNormRandom ()
{
	const double randmax = RAND_MAX;
	long r;
	double normrand;

#if LINUX
	r = random(); /* 0..RAND_MAX */
#else
	r = rand();
#endif
	normrand = double(r) / randmax;
	if (normrand >= 1.0)
	{
		cout << "CollageSystem: WARNING: normalized random value >= 1. Check basic.h !" << endl;
		normrand = 1.0;
	}
	return normrand;
}


Boolean Base::ContextOk  (Collage *theCollage, CollageRule *curRule, 
                          Transformation *trans)

	// Returns True if 'curRule' satisfies all context conditions.
{
	Boolean pcontextOk, ncontextOk;

	if (! curRule->GetContext())
		pcontextOk = True;
	else
	{
		Collage transContext (*(curRule->GetContext()));
		transContext.Transform (*trans);
		pcontextOk = ( theCollage->FindContext (transContext, curRule->GetCAnyOf()));
	}

	if (! curRule->GetNContext())
		ncontextOk = True;
	else
	{
		Collage transContext (*(curRule->GetNContext()));
		transContext.Transform (*trans);
		ncontextOk = (! theCollage->FindContext (transContext, curRule->GetNCAnyOf()));
	}

	return (pcontextOk && ncontextOk);
}

Boolean Base::GetEdgeRuleNormal (Collage *theCollage, CollageRuleSet *theRules,
                                 TableInfo *tableInfo, Edge *curEdge,
                                 Boolean kInverse,
                      /* out: */ CollageRule *&rule, Transformation *&trans)
{
	double normrand, totalProb;
	CollageRule **curRule;

	// Throw the dice only once per edge:
	normrand = GetNormRandom ();
	totalProb = 0.0;
	curRule = theRules->GetAt(0);
	trans = 0;
		
	while (curRule) 
	{
		//--- same equivalence class ? ---
		if ( (*curRule)->GetClass() != curEdge->GetClass() )
		{
			curRule = theRules->GetNext();
			continue;
		}
				
		//--- correct table ? ---
		if ( (*curRule)->GetTable() != 0 && (*curRule)->GetTable() != tableInfo->Table() )
		{
			curRule = theRules->GetNext ();
			continue;
		}
		
		//--- probability ok ? ---
		totalProb += (*curRule)->GetProb();
		if (totalProb > 1.0)
			cout << "Warning: Probability > 1 in table " << int(tableInfo->Table()) 
			     << " for edge " << curEdge->GetLabel().Get() << endl;
		if (totalProb < normrand)
		{
			curRule = theRules->GetNext ();
			continue;
		}				
		
		// After this point, the edge will be replaced by the current rule or not at all,
		// i.e. no more attempts are made to find another rule if this rule fails after it
		// has been chosen by the random generator.
		
		//--- transformation found ? ---
		if ( (trans = curEdge->FindTrans ((*curRule)->GetRight()->GetPins(),
		                                  (*curRule)->GetTrans(), kInverse)) == 0 )
			return False;
			                                            
		if (ContextOk (theCollage, *curRule, trans))
		{
			rule = *curRule;
			return True;
		}
		return False;
	}
	return False;	
}

Boolean Base::GetEdgeRuleExhaust (Collage *theCollage, CollageRuleSet *theRules,
                                 TableInfo *tableInfo, Edge *curEdge,
                                 Boolean kInverse,
                      /* out: */ CollageRule *&rule, Transformation *&trans)
{
	double normrand, totalProb;
	RuleProb *rp;
	RuleProbSet ruleset;
	
	ruleset.BuildSet (curEdge, theRules, tableInfo->Table());
	
	while (ruleset.Length() > 0)
	{
		// Throw the dice:
		normrand = GetNormRandom ();
		totalProb = 0.0;
		trans = 0;

		rp = ruleset.GetAt (0);
		while (rp)
		{
			totalProb += rp->prob;
			if (totalProb < normrand)
			{
				rp = ruleset.GetNext ();
				continue;  // try next rule
			}	
			
			//--- transformation found ? ---
			if ( (trans = curEdge->FindTrans (rp->rule->GetRight()->GetPins(),
		                                      rp->rule->GetTrans(), kInverse)) == 0 )
			{
				ruleset.RemoveRule (rp->rule);
				break; // start another round
			}
			
			//--- context ok ? ---
			if (! ContextOk (theCollage, rp->rule, trans))
			{
				ruleset.RemoveRule (rp->rule);
				break; // start another round
			}
							
			// --- rule found: ---
			rule = rp->rule;
			return True;			
		}
	}
	return False;
}

	
Boolean Base::Build (Collage* theCollage, CollageRuleSet* theRules, 
	                 TableInfo *tableInfo, Boolean kInverse)
{
	// Version with two alternate probabilitiy modes.
	// A return value of 'False' indicates that there are no more edges
	// to replace in the collage. 

	Edge *curEdge = theCollage->GetEdgeSet()->GetAt(0);
	CollageRule *curRule;
	Transformation *trans;
	Boolean gotRule;
	int edgeListIndex;
		
	if (! curEdge) 
		return False; 
	
	while (curEdge)     // for all egdes of the collage try to find a rule
	{ 
		edgeListIndex = theCollage->GetEdgeSet()->GetIndex();
	
		if (tableInfo->ExhaustProb ())
		{
			gotRule = GetEdgeRuleExhaust (theCollage, theRules, tableInfo,
			                          curEdge, kInverse, curRule, trans);
		}
		else
		{
			gotRule = GetEdgeRuleNormal (theCollage, theRules, tableInfo,
				                         curEdge, kInverse, curRule, trans);
		}

		if (gotRule)
		{
			Collage* delContext = 0;
			if (curRule->GetContext() && curRule->GetCDelete())
				delContext = curRule->GetContext();				
			Insert (curEdge, curRule->GetRight(), trans, delContext);
			if (tableInfo->Sequential() && !tableInfo->OneFromBase())
			{
				return True; 
				// only one edge should be replaced, and we can take the
				// first one in the order
			}
		}
			
		curEdge = theCollage->GetEdgeSet()->GetAt (++edgeListIndex);
	}
	
	int numElems = BaseElementList::Length();
	if (numElems > 0 && tableInfo->OneFromBase())
	{
		//
		// Select one of the base elements randomly and discard all others.
		//
		double normrand = GetNormRandom();		
		int elemIndex = (int)(normrand * (double)numElems);
		BaseElement* pSurvivor = BaseElementList::GetAt(elemIndex);
		BaseElement newSurvivor (*pSurvivor);
		while (BaseElementList::GetAt(0)) 
			BaseElementList::Delete();
		BaseElementList::Append (newSurvivor); // is copied again		
	}
	
	return True;
}



// ------- CollageGrammar -----------------------------------------------------

void CollageGrammar::PSOut () const
{
	// output start collage in the grammar part:
	start->MakeBoundingBox(2);
	const BoundingBox *bb = start->GetBoundingBox();
	thePS.NewFile ("G", bb);
	start->PSOut (psbGlobal, psbGrammar, psbStart);

	// output rules in the grammar part:
	rules->PSOut (psbGlobal, psbGrammar);
}

void CollageGrammar::VROut () const
{
	// output start collage in the grammar part:
	theVR.NewFile ("G");
	start->VROut ();

	// output rules in the grammar part:
	rules->VROut ();
}


void CollageGrammar::Normalize (Boolean transRightSide)
{
	rules->Normalize (transRightSide);
	rules->ClassifyEdges (start);
}


void CollageGrammar::Annotate ()
{
	CollageRule **rulep, *rule;

	rules->AnnotateEdges (start);
	rulep = rules->GetAt(0);
	while (rulep)
	{
		rule = *rulep;
		rules->AnnotateEdges (rule->GetRight());
		if (rule->GetContext())
			rules->AnnotateEdges (rule->GetContext());
		if (rule->GetNContext())
			rules->AnnotateEdges (rule->GetNContext());
		rulep = rules->GetNext();
	} // while (rulep)
}




// ------- Tables -------------------------------------------------------------

unsigned int TableElement::NextTable () 
{
	TableElemEntry *entry = entries.GetAt (actEntry);
	if (! entry)
	{
		// wrapped around, start again at beginning:
		ResetCounters ();
		entry = entries.GetAt (actEntry);
	}
	if (! entry) 
	{
		cerr << "CollageSystem: Internal error in derivation table" << endl;
		return 0;
	}
	actTimes++; // initialized with 0 -> gets 1 for first application
	if (entry->times >= actTimes)
	{
		// Once again the actual table:
		return entry->table;
	}
	else
	{
		// Next table (or wrap around):
		actEntry++;
		actTimes = 0;
		return NextTable ();
	}
}

Table::~Table ()
{
	TableElement **cur = GetAt (0);
	while (cur)
	{
		Delete ();
		delete *cur;
		cur = GetAt (0);
	}
}

Boolean Table::Insert (TableElement *te)
{
	// Insert sorted:

	TableElement **cur = GetAt (0);
	uint pos=0;
	while (cur) 
	{
		if ((*cur)->until > te->until) 
		{
			InsertAt (pos, te);
			return True;
		}
		else if ((*cur)->until == te->until) 
		{
			cerr << "CollageSystem: Duplicate 'until " << te->until << "' in derivation table" << endl;
			return False;
		}
		cur = GetAt (++pos);
	}	
	// first entry or larger than all other keys, append to end of list:
	Append (te);
	return True;
}

TableElement *Table::NextTable (int until, int *table)
{
	TableElement **cur = GetAt(0);
	while (cur && (*cur)->until != until) cur = GetNext();
	if (cur) 
	{
		*table = (*cur)->NextTable();
		return *cur;
	}
	return NULL;
}

Boolean Table::GetMaxUntil (int *maxuntil) const
{
	TableElement **last = GetLast(); // list is sorted
	if (last) *maxuntil = (*last)->until;
	return last!=NULL;
}


//----- TableInfoList -----------------------------------------------------

TableInfoList::~TableInfoList ()
{
	TableInfo **cur;

	while ((cur = GetAt(0)) != NULL)
	{
		delete *cur;
		Delete ();
	}
}

TableInfo *TableInfoList::Find (int table)
{
	TableInfo **cur = GetAt (0);
	while (cur && (*cur)->Table() != table)
		cur = GetNext ();
	if (cur)
		return *cur;
	return NULL;
}

