#ifndef __COLLAGE_H__
#define __COLLAGE_H__
//
// collage.h  --  CollageTwo
//
// Classes for collages, rules and derivation bases
//
// Authors            : Nils Schwabe, Stefan Taubenberger
// Date of creation   : 19 Apr 94
// Last modification  : 12 Feb 96 ns
//
// University of Bremen, Germany
//

#include "figures.h"
#include "hyper.h"
#include "trans.h"


// ------- PinSequence --------------------------------------------------------

// A PinSequence is a PointSequence that has additional Draw() and Transform()
// facilties and is used to represent the pin-points of a collage. 

class PinSequence : public PointSequence  // (PointSequence defined in point.h)
{
public:
	PinSequence &operator= (const PinSequence &copy) {
		PointSequence::operator= (copy); return *this;}
		
	PinSequence &operator= (const PointSequence &copy) {
		PointSequence::operator= (copy); return *this;}
		
	void Draw() const;
	
	void Transform (const Transformation &T);
};


// ------- Collage ------------------------------------------------------------

// Objects of this class represent both decorated and terminal collages.

class Base; // forward

class Collage 
{
public:

	Collage (PinSequence *initPins,
	         PartSet *initParts,
	         EdgeSet *initEdges);         
	Collage ();
	Collage (const Collage &theCollage);
	~Collage ();
	
	Collage &operator= (const Collage &theCollage);

	void AppendPin (const Point &thePoint);
	
		// Appends 'thePoint' to the list of pin-points.
	
	const PinSequence *GetPins() {return pins;}
	
		// Returns a pointer to the pin-points sequence.
	
	void InsertPart (const Part &thePart);
	
		// Insert 'thePart' into the collage. The part is copied.
	
	void InsertEdge (Edge *theEdge); 
	
		// Insert 'theEdge' into the collage. The pointer is stored!
		
	EdgeSet *GetEdgeSet() {return edges;}
	
		// Return a pointer to the edge sequence.
		
	void SetEdgeSet (EdgeSet *set) {edges = set;} 
	
		// Set the edge sequence. The pointer 'set' is stored!

	void Draw () const;
	
		// Draw the collage into the current screen window.
		
	void PSOut (Textblock *t1, Textblock *t2, Textblock *t3) const;
	
		// Output the collage to the current PostScript context.
		
	void VROut () const;
		
	void Transform (const Transformation &T);
	
		// Apply a transformation to the collage.

	void ReplEdge (Edge *edge,
	               const Collage &repl,
	               Transformation *t,
	               ColorIndex colorOffset,
	               Boolean kInverse,
	               Collage* delContext
#if USETCL
	               ,Tcl_Interp *interp
#endif
	               );
	               
	// Replaces 'edge' with 'repl' which will be transformed using 't'
	// which must be a transformation obtained by edge->FindTrans().
	// 'edge' and 't' are destroyed, 'repl' remains untouched.
	// The color indices of all new parts (and hyperedges, if any)
	// are set to the value of their source (from 'repl) plus the
	// value of 'colorOffset'. 

	void DeriveWithBase (const Base &theBase,
	                     ColorIndex colorOffset,
	                     Boolean kInverse
#if USETCL
	                     ,Tcl_Interp *interp
#endif
	                     );
	                     
	// Carries out all edge replacements described in 'theBase'.
	
	void MakeBoundingBox (uint dim);
	const BoundingBox *GetBoundingBox() const {return bb;}
	
	
	Boolean FindContext (Collage &context, Boolean anyOf);
	
	// Returns True if 'context' is contained within this collage.
	// If 'anyOf' is True then True is returned if at least one element
	// (part or edge) is contained within this collage.

	void RemoveSubCollage(Collage& subCollage);

private:

	void Add (const Collage &theCollage);
	// Add the parts and edges of theCollage to this collage.
	// The pin points are ignored.
	
	Boolean FindParts (Collage &context, Boolean anyOf);
	Boolean FindEdges (Collage &context, Boolean anyOf);
	

	PartSet *parts;
	PinSequence *pins;
	EdgeSet *edges;
	BoundingBox *bb;
};



// ------- CollageRule(Set) ---------------------------------------------------

// Classes representing rules and sets of rules of a collage grammar.

class CollageRule 
{
public:
	CollageRule (const EdgeText& left, 
	             Collage *       right,
	             Collage *       context,
	             Collage *       ncontext, 
	             Boolean         cAnyOf,
	             Boolean         ncAnyOf,
	             Boolean         cDelete,
	             TransFinder *   transFinder,
	             int             table = 0,
	             double          prob = 1);
	~CollageRule ();

	void SetText (const EdgeText& left) {leftHandText = left;}
	
		// Sets the text label of the edge.
		
	void SetClass (EdgeClass ec) {leftHandClass = ec;}
	
		// Sets the edge class.
		
	void SetRight (Collage* right) {rightHandSide = right;}
	
		// Sets the right-hand-side collage of the rule.
		
	void SetTable (int table) {tableNumber = table;}
	
		// Sets the table to which this rule belongs. 0 = global rule.

	const EdgeText& GetText() const {return leftHandText;}
	
		// Returns the text label of the edge.
		
	EdgeClass GetClass() const {return leftHandClass;}
	
		// Returns the edge class.
		
	Collage *GetRight() const {return rightHandSide;}
	
		// Returns a pointer to the right-hand-side collage.	
		
	Collage *GetContext() const {return context;}
	
		// Returns a pointer to the context collage. May be 0.
		
	Collage *GetNContext() const {return ncontext;}
		
		// Returns a pointer to the negative context collage. May be 0.
		
	Boolean GetCAnyOf () const {return cAnyOf;}
	
		// Returns the 'any of' setting for the context collage.
		
	Boolean GetNCAnyOf () const {return ncAnyOf;}
	
		// Returns the 'any of' setting for the negative context collage.
		
	Boolean GetCDelete () const {return cDelete;}
	
		// Returns the 'delete context' setting foe the context collage.
		
	int GetTable() const {return tableNumber;}
	
		// Returns the table to which this rule belongs. 0 = global rule.
		
	double GetProb() const {return prob;}
	
		// Returns the probability assigned to this rule.

	void Draw () const;		
	void PSOut (Textblock *first, Textblock *second) const;	
	void VROut () const;

	void SetTrans (TransFinder *tf) {transFinder = tf;}
	
	TransFinder *GetTrans () {return transFinder;}
	
	Transformation *FindTrans (const PointSequence *source,
	                           const PointSequence *target)
		{return transFinder->FindTrans (*source, *target);}

	Textblock *psbRule; // third

private:
	EdgeText    leftHandText;   // external label
	EdgeClass   leftHandClass;  // internal label (equivalence class)
	Collage *   rightHandSide;  // right hand side collage
	Collage *   ncontext;       // negative context (dto.) 
	Collage *   context;        // context collage (optional, may be NULL)
	Boolean     cAnyOf, ncAnyOf;// "any of" flag for both contexts
	Boolean     cDelete;        // "delete context" flag for positive context
	int         tableNumber;    // table number
	double      prob;           // probability, 0.0 <= prob <= 1.0
	TransFinder *transFinder;   // allowed transformations
};

ostream& operator<< (ostream& s, const CollageRule &cr);


typedef SList<CollageRule *> CollageRulePList;

class CollageRuleSet : public CollageRulePList 
{
public:
	void Insert (CollageRule *newElement);
	
	// Inserts a new rule into the collage rule set.

	void Draw () const;
	void PSOut (Textblock *first, Textblock *second) const;
	void VROut () const;

	void Normalize (Boolean transRightSide);
	
		// Normalizes the rules, i.e. assigns equivalence classes to
		// every rule and every edge in the right-sides.
		// If 'transRightSide'==True then transforms all right sides
		// of one equivalence class to identical pin-points.

	void ClassifyEdges (Collage *collage);
	
		// Assigns equivalence classes to all edges of 'collage'.
		// This is a part of Normalize() for all right-sides but must
		// be called explicitly for the start collage (see below) and
		// the context collages.

	void AnnotateEdges (Collage *collage);
	
		// Assuming that all edges in 'collage' are 'Hyperedge's,
		// replaces them by corresponding 'AnnotatedEdge's. The
		// rules and edges have to be normalized and classified
		// for this to work correctly.

private:
	void ClassifyEdge (Hyperedge *edge);

};


//----- TableInfo ---------------------------------------------------------

// Contains settings information for a derivation table.
// 'Table'       : the number of the table
// 'Sequential'  : use sequential derivations in this table (edge by edge)
// 'ExhaustProb' : Exhausting probabilties mode (see documentation)
// 'OneFromBase' : In sequential mode, build a complete base and then select
//                 one base element for application.

class TableInfo
{
public:
	TableInfo (unsigned int t, Boolean s=False, Boolean e=False,
	           Boolean ofb = false)
	{
		table = t; sequential = s; exhaustProb = e; oneFromBase = ofb;
	}
	
	void SetTable (int t)             {table = t;}
	void SetSequential (Boolean s)    {sequential = s;}
	void SetExhaustProb (Boolean e)   {exhaustProb = e;}
	void SetOneFromBase (Boolean e)   {oneFromBase = e;}
	
	unsigned int Table () const       {return table;}
	Boolean      Sequential () const  {return sequential;}
	Boolean      ExhaustProb () const {return exhaustProb;}
	Boolean      OneFromBase () const {return oneFromBase;}
	
private:
	unsigned int table;          // number of the table
	Boolean      sequential;     // sequential dervation
	Boolean      exhaustProb;    // 'exhausted probabilities' mode
	Boolean      oneFromBase;
};

class TableInfoList : public SList<TableInfo *>
{
public:
	~TableInfoList ();

	TableInfo *Find (int table);
};



// ------- Base ---------------------------------------------------------------

// Derivation base.

class BaseElement
{
public:
	BaseElement (Edge *e, Collage *s, Transformation *t, Collage* delCon) 
		{target=e; source=s; trans=t; delContext=delCon;}
		
	Edge* target;          // this edge should be replaced...
	Collage* source;       // ...by this collage...
	Transformation* trans; // ...using this transformation.
	Collage* delContext;   // != 0 => also delete this from the collage.
	                       // 'delContext' is used READ-ONLY by the BaseElement
};

typedef SList<BaseElement> BaseElementList;


class Base : public BaseElementList
{
public:
	Boolean Build (Collage* theCollage, CollageRuleSet* theRules, 
	               TableInfo *tableInfo, Boolean kInverse);
private:
	void Insert (Edge *edge, Collage *repl, Transformation *t, Collage* delContext) 
	{
		BaseElement elem (edge, repl, t, delContext);
		BaseElementList::Append (elem); // is copied
	}

	double GetNormRandom ();
	
		// Returns a normalized pseudo-random number ranging from 
		// 0.0 to 1.0.
	
	Boolean ContextOk  (Collage *theCollage, CollageRule *curRule, 
                        Transformation *trans);
                        
		// Checks the context conditions.

	Boolean GetEdgeRuleNormal (Collage *theCollage, CollageRuleSet *theRules,
                               TableInfo *tableInfo, Edge *curEdge,
                               Boolean kInverse,
                    /* out: */ CollageRule *&curRule, Transformation *&trans);
                      
		// Selects a rule using normal probabilities mode.

	Boolean GetEdgeRuleExhaust (Collage *theCollage, CollageRuleSet *theRules,
                                TableInfo *tableInfo, Edge *curEdge,
                                Boolean kInverse,
                     /* out: */ CollageRule *&curRule, Transformation *&trans);

		// Selects a rule using exhausting probabilities mode.

};


// ------- CollageGrammar -----------------------------------------------------

class CollageGrammar 
{
public:
	CollageGrammar () {start=0; rules = new CollageRuleSet; 
	                   psbGlobal=0; psbGrammar=0; psbStart=0;}
	                   
	~CollageGrammar () {delete start; delete rules;}
	
	void InsertRule (CollageRule* rule) {rules->Insert (rule); }
		
	CollageRuleSet *GetRules() const {return rules;}
	
	void SetStart (Collage* s) {start = s;}
	Collage *GetStart () const {return start;}
	
	void PSOut () const;
	void VROut () const;
	Textblock *psbGlobal;
	Textblock *psbGrammar;
	Textblock *psbStart;

	void Normalize (Boolean transRightSide);
	
		// Normalizes the collage grammar.

	void Annotate ();
	
		// Annotates the collage grammar (assuming an already normalized 
		// grammar).	

private:
	Collage* start;
	CollageRuleSet* rules;
};


// ------- Table --------------------------------------------------------------

// The following classes are used to store the derivation specification, i.e.
// which table to use in which derivation step.

class TableElemEntry
{
public:
	TableElemEntry (unsigned int tim, unsigned int tab) {
		times = tim; table = tab;
	}
	unsigned int times, table;
};

class TableElement 
{
public:
	TableElement (unsigned int u=0, Textblock *tb=0) {
		until=u; psbTableElem=tb; ResetCounters ();}
	unsigned int until;
	SList<TableElemEntry> entries;
	
	Textblock *psbTableElem;

	unsigned int NextTable ();
	void ResetCounters () {
		actEntry = 0; actTimes = 0;
	}

	unsigned int actEntry; // index in 'entries'
	unsigned int actTimes; // 0 <= actTimes <= 'times' of actual entry
};

typedef SList<TableElement *> TableElementList;


class Table : public TableElementList
{
public:
	Table () {
		psbDerivation=0; 
		psbDerivStart=0;
	}
	~Table ();

	Boolean Insert (TableElement *te);
	
		// Inserts a new table element, if not already defined.
		// Answers True if successful (element did not exist).

	TableElement *NextTable (int until, int *table);
	
		// Gets the next table number and associated table element
		// with exact matched 'until' value. Returns NULL if no
		// such table element exists.

	Boolean GetMaxUntil (int *maxuntil) const;
	Textblock *psbDerivation;
	Textblock *psbDerivStart;
};

#endif /* !defined(__COLLAGE_H__) */

