//
// convert.cc  --  CollageTwo
//
// Collage grammar conversions 
//
// Author            : Nils Schwabe
// Date of creation  : 19 Jul 94
// Last modification : Feb 96
//
// University of Bremen, Germany
//

#include <iostream.h>
#include "collage.h"

//------ CollageRuleSet::Normalize --------------------------------------------

void CollageRuleSet::Normalize
	(Boolean transRightSide)
{
	CollageRule **rulep, *rule;
	EdgeClass curClass = 0; // first valid equivalence class is 1

	// Step 1: Iterate once through all rules and assign equivalence classes.

	rulep = GetAt(0);
	while (rulep) 
	{
		rule = *rulep;
		if (rule->GetClass() != 0) 
		{
			rulep = GetNext();
			continue; // this rule as already been classified
		}
		
		// This rule becomes the representant of a new equivalence class:
		curClass++;
		//cout << *rule << " creates class " << curClass << endl;
		rule->SetClass (curClass);


		// Check all remaining rules if they belong to this class:
		int lastIndex = GetIndex(); // remember position of 'rule' in list	
		CollageRule **remainp = GetNext(), *remain;
		while (remainp) 
		{
			remain = *remainp;
			//cout << "  remain = " << *remain << endl;
			if (   remain->GetClass() == 0               // not yet classified
			    && remain->GetText().operator== (rule->GetText())) // external labels match
			{
				// Can the pin-points of 'remain' be mapped to 'rule' (using
				// 'remain's private transformation)? If so, then both rules
				// belong to the same equivalence class.

				const PinSequence *source, *target;
				Transformation *trans;

				source = remain->GetRight()->GetPins();
				target = rule->GetRight()->GetPins();
				//cout << "  Trying to map " << *source << " to " << *target << endl;

				if ((trans = remain->FindTrans (source, target)) != 0)
				{
					//cout << "    ---> " << curClass << endl;
					remain->SetClass (curClass);
					if (transRightSide)
						remain->GetRight()->Transform (*trans);
					delete trans;	
				}
			}
			remainp = GetNext();

		} // while (remainp)
		
		// re-position to 'rule':
		if ((rulep = GetAt(lastIndex)) == 0)
			cerr << "CollageRuleSet::Normalize: Internal list error (1)" << endl;
		else
			rulep = GetNext();

	} // while (rulep)	
	

	// Step 2: Iterate through all rules and all hyperedges and assign 
	//         the new equivalence classes to the hyperedges.

	rulep = GetAt(0);
	while (rulep)
	{
		rule = *rulep;
		//cout << "Classify edges of rule " << *rule << endl;
		int lastIndex = GetIndex(); // remember position of 'rule' in list	

		ClassifyEdges (rule->GetRight());
		if (rule->GetContext())
			ClassifyEdges (rule->GetContext());
		if (rule->GetNContext())
			ClassifyEdges (rule->GetNContext());
		
		//cout << "End Edges of rule" << endl;

		// re-position to 'rule':
		if ((rulep = GetAt(lastIndex)) == 0)
			cerr << "CollageRuleSet::Normalize: Internal list error (2)" << endl;
		else
			rulep = GetNext();
		
	} // while (rulep)
}


void CollageRuleSet::ClassifyEdges (Collage *collage)
{
	EdgeSet *edges = collage->GetEdgeSet(); 
	Hyperedge *edge = (Hyperedge *)edges->GetAt(0); 
	while (edge)
	{
		ClassifyEdge (edge);
		edge = (Hyperedge *)edges->GetNext(); 
	}
}


void CollageRuleSet::ClassifyEdge (Hyperedge *edge)
{
	CollageRule **rulep, *rule;
	
	// Iterate through the rules to find an applicable
	// equivalence class, which, by definition, is unique.

	rulep = GetAt(0);
	while (rulep)
	{
		rule = *rulep;
		//cout << "    Test rule " << *rule << " ... ";

		const PinSequence *source;
		Transformation *trans = 0;
		source = rule->GetRight()->GetPins();
		if (   edge->GetLabel() == rule->GetText()
                    && (trans = rule->FindTrans (source, edge)) != 0)
		{
			//cout << "positive" << endl;
			edge->SetClass (rule->GetClass());
			delete trans;
			break;
		}
		else
			//cout << "negative" << endl;
		rulep = GetNext();
	} // while (rulep)

	if (edge->GetClass() == 0)
		cout << "WARNING: Edge " << *edge << " couldn't be classified." << endl;
}


void CollageRuleSet::AnnotateEdges (Collage *collage)
{
	EdgeSet *edges = collage->GetEdgeSet();
	Hyperedge *edge = (Hyperedge *)edges->GetAt(0);
	EdgeSet *newedges = new EdgeSet();

	int lastIndex = GetIndex();

	while (edge)
	{
		// Find a rule of the same equivalence class:
		CollageRule **rulep, *rule;
		rulep = GetAt(0);

		while (rulep)
		{
			rule = *rulep;
			if (rule->GetClass() == edge->GetClass())
			{
				Transformation *trans;
				if ((trans = rule->FindTrans (rule->GetRight()->GetPins(), edge)) == 0) 
					cerr << "CollageRuleSet::Annotate: Can't find transformation" << endl;
				else 
				{
					//cout << "Annotating edge " << *(Hyperedge *)edge << endl;
					AnnotatedEdge *anno = new AnnotatedEdge (edge->GetLabel(), trans);
					anno->SetClass (edge->GetClass());
					//cout << "anno = " << anno->GetLabel() << "[" << anno->GetClass() << "]\n";
					newedges->Include (anno);
					break;
				}
			}
			rulep = GetNext();
		} // while (rulep)

		edge = (Hyperedge *)edges->GetNext();

	} // while (edge)

	if (lastIndex >= 0 && GetAt(lastIndex) == 0)
		cerr << "CollageRuleSet::AnnotateEdges: Voodoo list error" << endl;		

	if (edges->Length() != newedges->Length())
		cerr << "CollageRuleSet::AnnotateEdges: Voodoo length error" << endl;

	delete edges;
	collage->SetEdgeSet (newedges);	
}

