//
// figures.cc  --  CollageSystem
//
// Classes for representing the PARTS of a collage
//
// Author: N. Schwabe
// Date of creation : 11.03.94
// Last modification: 09 Dec 96 ns
//
// University of Bremen, Germany
//

#include <iostream.h>
#include "psout.h"
#include "vrout.h"
#include "figures.h"
#include "cgi.h"
#include "envi.h"  // needed for some global options (see 'theEnvi->')


// ------ Figure -------------------------------------------------------------- 

int Figure::operator==(const Figure &compare) const
{
	if (theEnvi && ! theEnvi->colourContext)
		return 1; // colours should not affect tests for equalitity
	if (useRGB)
		return rgb == compare.rgb;
	return color == compare.color;
}

void Figure::PSOut () const
{
	if (useRGB)
	{
		thePS.Out (((double)rgb.Red()) / 255.0 );
		thePS.Out (((double)rgb.Green()) / 255.0);
		thePS.Out (((double)rgb.Blue()) / 255.0);
	}
	else
	{
		thePS.Out ("0.0 0.0 0.0"); // black is default
	}
}

void Figure::Draw () const
{
	if (useRGB)
		theScreen->FgColor (rgb.Red(), rgb.Green(), rgb.Blue());
	else
		theScreen->FgColor (color);
}

void Figure::VROut () const
{
	if (useRGB)
	{
        theVR.Out ("  Material {\n");
        theVR.Out ("    diffuseColor ");
        theVR.Out (((double)rgb.Red()) / 255.0);
        theVR.Out (((double)rgb.Green()) / 255.0);
        theVR.Out (((double)rgb.Blue()) / 255.0);
        theVR.Out ("\n    ambientColor .5 .5 .5\n  } #Material\n");
	}
	else
	{
		cerr << "CollageSystem: Warning: 3-D part without RGB color - using default (light grey)\n";
        theVR.Out ("Material {\n");
        theVR.Out ("    diffuseColor .8 .8 .8\n  } #Material\n");	
	}	
}




// ------ Dot ----------------------------------------------------------------- 

void Dot::Draw () const
{
  Figure::Draw();
  if (thePoint.GetDim() >= 2)
    theScreen->Point (thePoint.Get(0), thePoint.Get(1));
  else
    theScreen->Point (thePoint.Get(0), DrawDim1Y);
};

void Dot::PSOut () const
{
  thePoint.PSOut();
  Figure::PSOut ();
  thePS.Out("d");
}

void Dot::VROut () const
{
	theVR.Out ("\nSeparator {\n");
	
	Figure::VROut (); 
	
	theVR.Out ("    Coordinate3 {point ");
	theVR.Out (thePoint.Get(0));
	theVR.Out (thePoint.Get(1));
	theVR.Out (thePoint.Get(2));
	theVR.Out ("} PointSet {numPoints 1}\n");
	theVR.Out ("} #Separator\n");
}


void Dot::Transform (const Transformation &T)
{
  T.Transform (thePoint, &thePoint);
}

int Dot::operator==(const Figure &compare) const
{
  if (Kind() != compare.Kind()) 
    return 0;
  if (! Figure::operator==(compare))
    return 0;
  return (thePoint.operator==(((Dot *)&compare)->thePoint));
}

void Dot::CalcBoundingBox (BoundingBox &bb) const
{
  bb.Calc (thePoint);
}


// ------ Line ---------------------------------------------------------------- 

Line &Line::operator=(const Line &ass)
{
  begin = ass.begin;
  end = ass.end;
  Figure::operator=((Figure &)ass);
  return *this;
}

void Line::Draw () const
{
  Figure::Draw();
  if (begin.GetDim() >= 2)
    theScreen->Line (begin.Get(0), begin.Get(1), end.Get(0), end.Get(1));
  else
    theScreen->Line (begin.Get(0), DrawDim1Y, end.Get(0), DrawDim1Y);
}

void Line::PSOut () const
{
  begin.PSOut();
  end.PSOut();
  Figure::PSOut ();
  thePS.Out ("l");
};


void Line::VROut () const
{
	theVR.Out ("\nSeparator {\n");
	
	Figure::VROut (); 
	
	theVR.Out ("    Coordinate3 {point [");
	theVR.Out (begin.Get(0));
	theVR.Out (begin.Get(1));
	theVR.Out (begin.Get(2));
	theVR.Out (",");
	theVR.Out (end.Get(0));
	theVR.Out (end.Get(1));
	theVR.Out (end.Get(2));	
	theVR.Out ("]} IndexedLineSet {coordIndex [0,1]}\n");
	theVR.Out ("}\n");
}


void Line::Transform (const Transformation &T)
{
  T.Transform (begin, &begin);
  T.Transform (end, &end);
}

int Line::operator==(const Figure &compare) const
{
  if (Kind() != compare.Kind()) 
    return 0;
  if (! Figure::operator==(compare))
    return 0;
  return (   (   begin.operator==(((Line *)&compare)->begin)
              && end.operator==(((Line *)&compare)->end))
          || (   begin.operator==(((Line *)&compare)->end)
              && end.operator==(((Line *)&compare)->begin)));
}

void Line::CalcBoundingBox (BoundingBox &bb) const
{
  bb.Calc (begin);
  bb.Calc (end);
}


// ------ DashLine ------------------------------------------------------------ 

void DashLine::Draw () const
{
  Figure::Draw();
  if (begin.GetDim() >= 2) {
    Coordinate x, y, dx, dy;
    int i;
    dx = 1.0 * (end.Get(0) - begin.Get(0)) / 5;
    dy = 1.0 * (end.Get(1) - begin.Get(1)) / 5;
    x = begin.Get(0);
    y = begin.Get(1);
    for (i=1; i<=3; i++) {
       theScreen->Line (x, y, x + dx, y + dy);
       x = x + dx + dx;
       y = y + dy + dy;
    }
  }
  else 
    theScreen->Line (begin.Get(0), DrawDim1Y, end.Get(0), DrawDim1Y);
}

void DashLine::PSOut () const
{
  begin.PSOut();
  end.PSOut();
  Figure::PSOut ();
  thePS.Out ("m");
};


// ------ PolyLine ------------------------------------------------------------ 

void PolyLine::Draw() const
{
  Figure::Draw();
  Point *p;
  if (! (p=GetAt(0))) return; /* empty */
  if (p->GetDim() < 2) return;              /* NO IMPL. OF 1-D POLYLINE, YET */
  theScreen->MoveTo (p->Get(0), p->Get(1));
  p = GetNext();
  while (p) {
    theScreen->LineTo (p->Get(0), p->Get(1));
    p = GetNext();
  }
}

void PolyLine::PSOut() const
{
  PointSequence::PSOut();
  Figure::PSOut ();
  thePS.Out ("p");
}


void PolyLine::VROut () const
{
	theVR.Out ("\nSeparator {\n");
	
	Figure::VROut (); 
	
	theVR.Out ("    Coordinate3 { point [");
	Point *p = GetAt(0);
	while (p) 
	{
		theVR.Out (p->Get(0));
		theVR.Out (p->Get(1));
		theVR.Out (p->Get(2));
		p = GetNext ();
		if (p)
			theVR.Out (",");
		
	}
	theVR.Out ("]}\n");
	theVR.Out ("    IndexedLineSet {coordIndex [");
	p = GetAt(0);
	int i=0;
	while (p) 
	{
		theVR.Out (i);
		p = GetNext ();
		if (p)
			theVR.Out (",");
		i++;
	}	
	theVR.Out ("]}\n");
	theVR.Out ("} #Separator\n");
}



void PolyLine::Transform (const Transformation &T)
{
  Point *cur = GetAt(0);
  while (cur) {
    T.Transform (*cur, cur);
    cur = GetNext();
  }
}


int PolyLine::operator==(const Figure &compare) const
{
  if (Kind() != compare.Kind()) 
    return 0;
  if (! Figure::operator==(compare))
    return 0;
  PolyLine *c = (PolyLine *)&compare;
#if 0
  Point *pself = GetAt(0), *pcmp = c->GetAt(0);
  while (pself && pcmp) {
    if (*pself != *pcmp) return 0;
    pself = GetNext();
    pcmp = c->GetNext();
  }
  return (pself == pcmp) /*<=> both==0*/; 
#else
  return CyclicIdentical (*c);
#endif
}

void PolyLine::CalcBoundingBox (BoundingBox &bb) const
{
  Point *cur = GetAt(0);
  while (cur) {
    bb.Calc (*cur);
    cur = GetNext();
  }
}


// ------ Polygon ------------------------------------------------------------- 

void Polygon::Draw() const
{
  Point *p = GetAt(0);
  uint len = 0, i;

  Figure::Draw();
  while (p) {
    len++;
    p = GetNext();
  }
  if (len == 0) return;

  Coordinate *points = new Coordinate[(len+1)*2];  
  p = GetAt(0);
  i = 0;
  points[len*2] = p->Get(0);
  points[len*2+1] = p->Get(1);
  while (p) {
    points[i*2] = p->Get(0);
    points[i*2+1] = p->Get(1);
    i++;
    p = GetNext();
  }
  if (fill)
    theScreen->FilledPoly (len+1, points);
  else
    theScreen->Poly (len+1, points); 
}

void Polygon::PSOut() const
{
  PointSequence::PSOut();
  Figure::PSOut ();
  if (fill) {
    thePS.Out ("r");
  }
  else {
    thePS.Out ("q");
  }
}

void Polygon::VROut () const
{
	theVR.Out ("\nSeparator {\n");
	
	Figure::VROut (); 
	
	theVR.Out ("    Coordinate3 {point [");
	Point *p = GetAt(0);
	while (p) 
	{
		theVR.Out (p->Get(0));
		theVR.Out (p->Get(1));
		theVR.Out (p->Get(2));
		p = GetNext ();
		if (p)
			theVR.Out (",");
		
	}
	if (! fill) // closed polyline, repeat first point
	{
		Point *p = GetAt(0);
		if (p)
		{
			theVR.Out (",");
			theVR.Out (p->Get(0));
			theVR.Out (p->Get(1));
			theVR.Out (p->Get(2));
		}
	}
	theVR.Out ("]}\n");
	
	if (fill)
		theVR.Out ("    IndexedFaceSet {coordIndex [");
	else
		theVR.Out ("    IndexedLineSet {coordIndex [");
		
	p = GetAt(0);
	int i=0;
	while (p) 
	{
		theVR.Out (i);
		p = GetNext ();
		if (p)
			theVR.Out (",");
		i++;
	}
	if ((! fill) && i > 0)
	{
		theVR.Out (",0");
	}
	theVR.Out ("]}\n");
	theVR.Out ("} #Separator\n");
}


int Polygon::operator==(const Figure &compare) const
{
  if (Kind() != compare.Kind()) 
    return 0;
  if (! Figure::operator==(compare))
    return 0;
  // same as for Polyline, but calling this is not possible because 
  // the Kind()-Test would fail.
  Polygon *c = (Polygon *)&compare;
#if 0
  Point *pself = GetAt(0), *pcmp = c->GetAt(0);
  while (pself && pcmp) {
    if (*pself != *pcmp) return 0;
    pself = GetNext();
    pcmp = c->GetNext();
  }
  return (pself == pcmp) /*<=> both==0*/; 
#else
  return CyclicIdentical (*c);
#endif
}


// ------ Ellipse ------------------------------------------------------------- 


Ellipse::Ellipse () : firstAxis(2), secondAxis(2)
{
  beginAngle = 0;
  endAngle = 360;
  fill = False;
}

Ellipse::Ellipse (const Point& initCenter, 
	         const Point& initOuter, 
	         Boolean initFill,
	         float initBeginAngle, 
	         float initEndAngle) : firstAxis(2), secondAxis(2)
{
/* Berechnung zweier Achsen QR und ST, die orthogonal sind und
   durch den Ursprung InitCenter gehen. Die vier Punkte Q,R,S,T 
   liegen auf dem Kreis. 
                                                                NOT OPTIMZED !
*/
   Point Q(2),R(2),S(2),T(2);
   float deltax, deltay;
   Coordinate px,py;

   px = initCenter.Get(0); 
   py = initCenter.Get(1);
   Q = initOuter;
   deltax = Q.Get(0) - px;
   deltay = Q.Get(1) - py;
   R = Point (px - deltax, py - deltay);
   S = Point (px - deltay, py + deltax);
   T = Point (px + deltay, py - deltax);
   firstAxis = Line (Q,R);
   secondAxis = Line (S,T);
//	Center = InitCenter;
//	Outer  = InitOuter;
   fill = initFill;
   beginAngle = initBeginAngle;
   endAngle = initEndAngle;
}

Ellipse::Ellipse (const Point& initCenter, 
	         float initRadius, 
	         Boolean initFill,
	         float initBeginAngle, 
	         float initEndAngle) : firstAxis(2), secondAxis(2)

/* Berechnung zweier Achsen QR und ST, die orthogonal sind und
   durch den Ursprung InitCenter gehen. Die vier Punkte Q,R,S,T
   liegen auf dem Kreis. 
                                                                NOT OPTIMIZED !
*/
{
   Point Q(2),R(2),S(2),T(2);
   Coordinate px,py;
   px = initCenter.Get(0); 
   py = initCenter.Get(1);
   Q = Point (px - initRadius, py);
   R = Point (px + initRadius, py);
   S = Point (px, py + initRadius);
   T = Point (px, py - initRadius);
   firstAxis = Line (Q,R);
   secondAxis = Line (S,T);
//	Center = InitCenter;
//	Outer.SetX (Center.GetX() + InitRadius);
//	Outer.SetY (Center.GetY());
   fill = initFill;
   beginAngle = initBeginAngle;
   endAngle = initEndAngle;
}

Ellipse::Ellipse (const Line& initFirstAxis, 
	         const Line& initSecondAxis,
 	         Boolean initFill,
	         float initBeginAngle, 
	         float initEndAngle) : firstAxis(2), secondAxis(2)
/*
                                                                NOT OPTIMIZED !
*/
{
  firstAxis = initFirstAxis;
  secondAxis = initSecondAxis;
  fill = initFill;
  beginAngle = initBeginAngle;
  endAngle = initEndAngle;
}

Ellipse::Ellipse (const Ellipse &copy) 
    : Figure ((Figure &)copy), firstAxis(copy.firstAxis), secondAxis(copy.secondAxis)
{
  beginAngle = copy.beginAngle;
  endAngle = copy.endAngle;
  fill = copy.fill;
  Figure::operator=((Figure &)copy);
}

Ellipse &Ellipse::operator=(const Ellipse &ass)
{
  firstAxis = ass.firstAxis;
  secondAxis = ass.secondAxis;
  beginAngle = ass.beginAngle;
  endAngle = ass.endAngle;
  fill = ass.fill;
  Figure::operator=((Figure &)ass);
  return *this;
}

void Ellipse::Draw() const
{
  firstAxis.Draw();
  secondAxis.Draw();
}


void Ellipse::PSOut() const
{
  ((Ellipse *)this)->firstAxis.GetBegin().PSOut();
  ((Ellipse *)this)->firstAxis.GetEnd().PSOut();
  ((Ellipse *)this)->secondAxis.GetBegin().PSOut();
  ((Ellipse *)this)->secondAxis.GetEnd().PSOut();
  thePS.Out (beginAngle);
  thePS.Out (endAngle);
  Figure::PSOut ();
  if (fill)
    thePS.Out ("fl");
  else
    thePS.Out ("el");
}

void Ellipse::Transform (const Transformation &trans)
{
  firstAxis.Transform (trans);
  secondAxis.Transform (trans);
}

int Ellipse::operator==(const Figure &compare) const
{
  if (Kind() != compare.Kind()) 
    return 0;
  if (! Figure::operator==(compare))
    return 0;
  Ellipse *cmp = (Ellipse *)&compare;
  return (beginAngle == cmp->beginAngle
       && endAngle == cmp->endAngle
       && fill == cmp->fill
       && firstAxis.operator==(cmp->firstAxis)
       && secondAxis.operator==(cmp->secondAxis));
}

void Ellipse::CalcBoundingBox (BoundingBox &bb) const
{
  bb.Calc (((Ellipse *)this)->firstAxis.GetBegin());
  bb.Calc (((Ellipse *)this)->firstAxis.GetEnd());
  bb.Calc (((Ellipse *)this)->secondAxis.GetBegin());
  bb.Calc (((Ellipse *)this)->secondAxis.GetEnd());
  // Don't tell me that it's much more complicated than this,
  // I DON'T WANNA HEAR IT !!
}


// ------ Trans3DFigure -------------------------------------------------------


Trans3DFigure::Trans3DFigure ()
{
	trans = NULL;
	InitTransformation ();
}


Trans3DFigure::Trans3DFigure (const Trans3DFigure& copy)

	: Figure (copy),
	  trans (NULL)
{
	if (copy.trans)
		trans = copy.trans->Copy ();
	else
		InitTransformation ();
}


Trans3DFigure& Trans3DFigure::operator= (const Trans3DFigure& ass)
{
	Figure::operator= (ass);
	if (ass.trans)
	{
		if (trans)
			delete trans;
		trans = ass.trans->Copy ();
	}
	else
		InitTransformation ();	
	return *this;
}


void Trans3DFigure::Draw () const
{
	// not supported
}


void Trans3DFigure::PSOut () const
{
	// not supported
}


void Trans3DFigure::CalcBoundingBox (BoundingBox &bb) const
{
	// not supported
}


void Trans3DFigure::InitTransformation ()
{
	MatrixTrans* mtrans = new MatrixTrans (4,4);
	
	mtrans->Set(0,0,1); mtrans->Set(0,1,0); mtrans->Set(0,2,0);
	mtrans->Set(0,3,0);
		
	mtrans->Set(1,0,0); mtrans->Set(1,1,1); mtrans->Set(1,2,0);
	mtrans->Set(1,3,0);
		
	mtrans->Set(2,0,0); mtrans->Set(2,1,0); mtrans->Set(2,2,1);
	mtrans->Set(2,3,0);
		
	mtrans->Set(3,0,0); mtrans->Set(3,1,0); mtrans->Set(3,2,0);
	mtrans->Set(3,3,1);
		
	trans = mtrans;
}

void Trans3DFigure::Transform (const Transformation &t)
{
	if (trans) 
	{
		Transformation *n = t.Chain (trans);
		delete trans;
		trans = n;
	}
	else
		trans = t.Copy ();
}

void Trans3DFigure::VROut () const
{
	if (! trans)
	{
		cerr << "CollageSystem: Trans3DFigure::VROut: No transformation!" << endl;
		return;
	}
	
	if (trans->InternalType() != Transformation::transMatrix)
	{
		cerr << "CollageSystem: Trans3DFigure::VROut: Can currently only handle matrix-based transformations.\n"
		        "               Use affine transformations only with 3D objects. Have to exit now, sorry...\n";
		exit (-1);
	}			
	
	theVR.Out ("  MatrixTransform { matrix ");

	MatrixTrans *mtrans = (MatrixTrans *)trans;
		
	theVR.Out (mtrans->Get(0,0));
	theVR.Out (mtrans->Get(1,0));
	theVR.Out (mtrans->Get(2,0));
	theVR.Out (mtrans->Get(3,0));

	theVR.Out (mtrans->Get(0,1));
	theVR.Out (mtrans->Get(1,1));
	theVR.Out (mtrans->Get(2,1));
	theVR.Out (mtrans->Get(3,1));
	
	theVR.Out (mtrans->Get(0,2));
	theVR.Out (mtrans->Get(1,2));
	theVR.Out (mtrans->Get(2,2));
	theVR.Out (mtrans->Get(3,2));

	theVR.Out (mtrans->Get(0,3));
	theVR.Out (mtrans->Get(1,3));
	theVR.Out (mtrans->Get(2,3));
	theVR.Out (mtrans->Get(3,3));

	theVR.Out ("}\n");	
}

int Trans3DFigure::operator==(const Figure &compare) const
{
	// <M> Test transformation results with 3 linear independent points
	return False;
}



// ------ Sphere --------------------------------------------------------------

Sphere::Sphere (double aRadius)

	: radius (aRadius)
	  
{
}

	  
Sphere::Sphere (const Sphere& copy)

	: Trans3DFigure (copy),
	  radius (copy.radius)
{
}

	  
Sphere& Sphere::operator= (const Sphere& ass)
{
	Trans3DFigure::operator= (ass);
	this->radius = ass.radius;
	return *this;
}


void Sphere::VROut () const
{
	theVR.Out ("\nSeparator {\n");
	
	Figure::VROut (); 
	Trans3DFigure::VROut ();

	theVR.Out ("  Sphere { radius ");
	theVR.Out (radius);
	theVR.Out ("}\n");

	theVR.Out ("} #Separator\n");
}


int Sphere::operator==(const Figure &compare) const
{
  if (Kind() != compare.Kind()) 
    return 0;
  return Trans3DFigure::operator==(compare);
}




// ------ Cube --------------------------------------------------------------


Cube::Cube (double w, double h, double d)

	: width (w),
	  height (h),
	  depth (d)
{
}

	  
Cube::Cube (const Cube& copy)

	: Trans3DFigure (copy),
	  width (copy.width),
	  height (copy.height),
	  depth (copy.depth)

{

}

	  
Cube& Cube::operator= (const Cube& ass)
{
	Trans3DFigure::operator= (ass);
	width = ass.width;
	height = ass.height;
	depth = ass.depth;
	return *this;
}

	


void Cube::VROut () const
{
	theVR.Out ("\nSeparator {\n");
	
	Figure::VROut (); 
	Trans3DFigure::VROut ();

	theVR.Out ("  Cube { width ");
	theVR.Out (width);
	theVR.Out (" height ");
	theVR.Out (height);
	theVR.Out (" depth ");
	theVR.Out (depth);
	theVR.Out ("}\n");

	theVR.Out ("} #Separator\n");
}


int Cube::operator==(const Figure &compare) const
{
  if (Kind() != compare.Kind()) 
    return 0;
  return Trans3DFigure::operator==(compare);
}



// ------ Cone --------------------------------------------------------------


Cone::Cone (double br, double h, Parts p)

	: bottomRadius (br),
	  height (h),
	  parts (p)
{
}

	  
Cone::Cone (const Cone& copy)

	: Trans3DFigure (copy),
	  bottomRadius (copy.bottomRadius),
	  height (copy.height),
	  parts (copy.parts)

{

}

	  
Cone& Cone::operator= (const Cone& ass)
{
	Trans3DFigure::operator= (ass);
	bottomRadius = ass.bottomRadius;
	height = ass.height;
	parts = ass.parts;
	return *this;
}

	


void Cone::VROut () const
{
	theVR.Out ("\nSeparator {\n");
	
	Figure::VROut (); 
	Trans3DFigure::VROut ();

	theVR.Out ("  Cone { parts ");
	if (parts == all)
		theVR.Out ("ALL");
	else if (parts == bottom)
		theVR.Out ("BOTTOM");
	else
		theVR.Out ("SIDES");
	theVR.Out (" bottomRadius ");
	theVR.Out (bottomRadius);
	theVR.Out (" height ");
	theVR.Out (height);
	theVR.Out ("}\n");

	theVR.Out ("} #Separator\n");
}


int Cone::operator==(const Figure &compare) const
{
  if (Kind() != compare.Kind()) 
    return 0;
  return Trans3DFigure::operator==(compare);
}



// ------ Cylinder -----------------------------------------------------------


Cylinder::Cylinder (double r, double h, Parts p)

	: radius (r),
	  height (h),
	  parts (p)
{
}

	  
Cylinder::Cylinder (const Cylinder& copy)

	: Trans3DFigure (copy),
	  radius (copy.radius),
	  height (copy.height),
	  parts (copy.parts)

{

}

	  
Cylinder& Cylinder::operator= (const Cylinder& ass)
{
	Trans3DFigure::operator= (ass);
	radius = ass.radius;
	height = ass.height;
	parts = ass.parts;
	return *this;
}

	


void Cylinder::VROut () const
{
	theVR.Out ("\nSeparator {\n");
	
	Figure::VROut (); 
	Trans3DFigure::VROut ();

	theVR.Out ("  Cylinder { parts ");
	if (parts == all)
		theVR.Out ("ALL");
	else if (parts == bottom)
		theVR.Out ("BOTTOM");
	else if (parts == sides)
		theVR.Out ("SIDES");
	else
		theVR.Out ("TOP");
	theVR.Out (" radius ");
	theVR.Out (radius);
	theVR.Out (" height ");
	theVR.Out (height);
	theVR.Out ("}\n");

	theVR.Out ("} #Separator\n");
}


int Cylinder::operator==(const Figure &compare) const
{
  if (Kind() != compare.Kind()) 
    return 0;
  return Trans3DFigure::operator==(compare);
}


// ------ Native -----------------------------------------------------------


Native::Native (Textblock* def)

	: definition (def)	// <C> Pointer is stored 
{
}

	  
Native::Native (const Native& copy)

	: Trans3DFigure (copy),
	  definition (copy.definition) // <C> Pointer is copied, and never freed!
{
}

	  
Native& Native::operator= (const Native& ass)
{
	Trans3DFigure::operator= (ass);
	definition = ass.definition; // <C> see above
	return *this;
}

	


void Native::VROut () const
{
	theVR.Out ("\nSeparator {\n");
	
	Figure::VROut ();
	Trans3DFigure::VROut ();
	
	theVR.Out (definition);

	theVR.Out ("\n} #Separator\n");
}


int Native::operator==(const Figure &compare) const
{
  if (Kind() != compare.Kind()) 
    return 0;
  return Trans3DFigure::operator==(compare);
}



// ------ PointLight -----------------------------------------------------------


void PointLight::Transform (const Transformation &t)
{
	t.Transform (position, &position);
}


int PointLight::operator==(const Figure &compare) const
{
	if (Kind() != compare.Kind()) 
		return 0;
	if (! Figure::operator==(compare))
		return 0;
	return (position.operator==(((PointLight *)&compare)->position));
}
	

void PointLight::VROut () const
{
	theVR.Out ("\nPointLight {");

	if (UsesRGB())
	{
		theVR.Out (" color ");
		theVR.Out (((double)GetRGB().Red()) / 255.0 );
		theVR.Out (((double)GetRGB().Green()) / 255.0);
		theVR.Out (((double)GetRGB().Blue()) / 255.0);
	}
	
	theVR.Out (" location ");
	theVR.Out (position.Get(0));
	theVR.Out (position.Get(1));
	theVR.Out (position.Get(2));
	
	theVR.Out (" }\n\n");	
}




// ------ FigureSet ----------------------------------------------------------- 

void FigureSet::AppendPhysical (const FigureSet &theSet)
{
  Figure **cur = theSet.FigureList::GetAt(0);
  while (cur) {
    Include (**cur); // is making a copy, see below
    cur = theSet.FigureList::GetNext();
  }
}

FigureSet::FigureSet (const FigureSet &theSet)
    : FigureList ()
{
  AppendPhysical (theSet);
}

FigureSet& FigureSet::operator= (const FigureSet &theSet)
{
  Figure **cur = FigureList::GetAt(0);
  while (cur) {
    FigureList::Delete();
    cur = FigureList::GetAt(0);
  }
  AppendPhysical (theSet);
  return *this;
}

void FigureSet::Include (const Figure &element)
{
  if (   (! theEnvi->mergeParts)  
      || (! Contains (element)) )
  {
    Figure *incl = element.Clone(); // make physical copy of figure
    Append (incl);
  }
}

void FigureSet::Add (const FigureSet &theSet)
{
  Append (theSet);
}

Figure* FigureSet::Contains (const Figure &element) const
{
  Figure **cur = FigureList::GetAt(0);

  while (cur) {
    if (**cur == element) 
    	return *cur;
    cur = FigureList::GetNext();
  } 
  return 0;
}

void FigureSet::Exclude(Figure* element) 
{
  Figure **cur = FigureList::GetAt(0);

  while (cur) {
    if (*cur == element) {
    	FigureList::Delete();
    	return;
    }
    cur = FigureList::GetNext();
  } 
}



void FigureSet::Draw() const
{
  Figure **cur = FigureList::GetAt(0);

  while (cur) {
    (*cur)->Draw();
    cur = FigureList::GetNext();
  }
}

void FigureSet::PSOut() const
{
  Figure **cur = FigureList::GetAt(0);

  while (cur) {
    (*cur)->PSOut();
    cur = FigureList::GetNext();
  }
}

void FigureSet::VROut() const
{
  Figure **cur = FigureList::GetAt(0);

  while (cur) {
    (*cur)->VROut();
    cur = FigureList::GetNext();
  }
}

void FigureSet::Transform (const Transformation &trans)
{
  Figure **cur = FigureList::GetAt(0);

  while (cur) {
    (*cur)->Transform (trans);
    cur = FigureList::GetNext();
  }
}

void FigureSet::CalcBoundingBox (BoundingBox &bb) const
{
  Figure **cur = FigureList::GetAt(0);

  while (cur) {
    (*cur)->CalcBoundingBox (bb);
    cur = FigureList::GetNext();
  }
}


#if 0    /* SOME TEST CODE: */

ScreenWin theScreen->

void Main (int argc, char *argv[]) 
{
  FigureSet set;

  Dot a(Point(100,100)), b(Point(100,100)), c(Point(100,100));

  set.Insert (&a);
  if (set.Contains (&b))
    cout << "b schon drin\n";
  else
    set.Insert (&b);

  if (set.Contains(&c))
    cout << "c schon drin\n";
  else
    set.Insert (&c);
  set.Draw();
}

int main (int argc, char *argv[])
{
  theScreen->DoProgram (argc, argv, Main, 640, 480);
  return 0;
}



#endif /* TEST CODE */



