#ifndef __FIGURES_H__
#define __FIGURES_H__
//
// figures.h  --  CollageTwo
//
// Classes for representing the PARTS of a collage
//
// Authors           : Nils Schwabe, Stefan Taubenberger
// Date of creation  : 11 Mar 94 ns
// Last modification : 09 Dec 96 ns
//
// University of Bremen, Germany
//

#ifndef __TRANS_H__
#include "trans.h"
#endif

// ------ Figure -----------------------------------------------------abstract-

enum FigureKind
{
	FigureDot,
	FigureLine,
	FigureDashLine,
	FigurePolyLine,
	FigurePolygon,
	FigureEllipse,
	
	FigureSphere,
	FigureCube,
	FigureCone,
	FigureCylinder,
	FigurePointLight,
	FigureNative
};

class Figure 
{
public:
	Figure ()                                     {color=0; useRGB=False;}

	virtual Figure *Clone() const = 0; 

		// Virtual copy constructor.

	virtual void Draw () const; // sets color

	virtual void PSOut () const; // outputs color
	
	virtual void VROut () const; // outputs color

	virtual void Transform (const Transformation &T) = 0;

	virtual int operator==(const Figure &compare) const;
	
		// compares the colour.

	virtual void CalcBoundingBox (BoundingBox &bb) const = 0;

	void SetColor (ColorIndex c)                    
		                                  {useRGB=False; color = c;}
	ColorIndex GetColor() const                          {return color;}

	void SetRGB (RGBValue value)                  
		                                 {useRGB=True; rgb = value;}
	RGBValue GetRGB () const                               {return rgb;}

	Boolean UsesRGB () const                            {return useRGB;}

	void SetColorProc (const SString<char> &proc)  
	                                                 {colorProc = proc;}
	SString<char> &GetColorProc ()                 
	                                                 {return colorProc;}

	virtual FigureKind Kind() const = 0;
	// THIS SHOULD BE REPLACED BY A RTTI AS SOON AS THESE BECOME
	// AVAILABLE.
	
private:
	ColorIndex color;
	RGBValue rgb;
	Boolean useRGB;
	SString<char> colorProc;
};    

 
// ------ Dot -----------------------------------------------------------------

class Dot : public Figure
{
public:
	Dot (ushort dim)	: thePoint(dim)                             {}

	Dot (const Point &p)	: thePoint(p)                           {}

	Dot (const Dot &copy)	: Figure ((Figure &)copy), 
				  thePoint(copy.thePoint)                           {}

	Dot &operator= (const Dot &ass) 
	{
		thePoint.operator=(ass.thePoint); 
		Figure::operator=((Figure &)ass);
		return *this;
	}

	Figure *Clone() const                         {return new Dot(*this);}

	void Draw () const;
	void PSOut () const;
	void VROut () const;

	void Transform (const Transformation &T);
	int operator==(const Figure &compare) const;
	void CalcBoundingBox (BoundingBox &bb) const;

        Point& Get ()                                       {return thePoint;}

protected:
	FigureKind Kind() const {return FigureDot;}

private:
	Point thePoint;
};


// ------ Line ---------------------------------------------------------------- 

class Line : public Figure 
{
public:
	Line (ushort dim)		: begin(dim), end(dim)              {}
 
	Line (const Point &initBegin, 
	      const Point &initEnd)	: begin(initBegin), end (initEnd)   {}

	Line (const Line &copy) 
	     				: Figure ((Figure &)copy), 
					  begin(copy.begin), end(copy.end)  {}

	Line &operator=(const Line &ass);

	Figure *Clone() const                        {return new Line(*this);}

	void Draw () const;
	void PSOut () const;
	void VROut () const;
	
	void Transform (const Transformation &T);
	int operator==(const Figure &compare) const;
	void CalcBoundingBox (BoundingBox &bb) const;

	Point& GetBegin ()                                     {return begin;}
	Point& GetEnd ()                                         {return end;}

protected:
	FigureKind Kind() const {return FigureLine;}
	Point begin, end;
};


// ------ DashLine ----------------------------------------------------------- 

class DashLine : public Line 
{
public:
  	DashLine (ushort dim)		: Line(dim)                         {} 

  	DashLine (const Point &initBegin, 
	          const Point &initEnd)	: Line (initBegin, initEnd)     {}

  	DashLine (const DashLine &copy) : Line ((Line &)copy)           {}

  	DashLine &operator=(const DashLine &ass) 
	{
		Line::operator=(ass); return *this;
	}

  	Figure *Clone() const                    {return new DashLine(*this);}

	void Draw () const;
	void PSOut () const;

protected:
	FigureKind Kind() const {return FigureDashLine;}
};


// ------ PolyLine ------------------------------------------------------------ 

class PolyLine : public PointSequence, public Figure 
{
public:
	PolyLine () 			: PointSequence()                   {}

	PolyLine (const PolyLine &copy) : Figure ((Figure &)copy), 
					  PointSequence(copy)               {}

	PolyLine (const PointSequence &copy)	: PointSequence(copy)       {}

	PolyLine &operator=(const PolyLine &ass) 
	{
		PointSequence::operator=(ass); 
		Figure::operator=((Figure &)ass);
		return *this; 
	}

	Figure *Clone() const {return new PolyLine(*this);}

	void Draw () const;
	void PSOut () const;
	void VROut () const;
	
	void Transform (const Transformation &T);
	int operator==(const Figure &compare) const;
	void CalcBoundingBox (BoundingBox &bb) const;

protected:
	FigureKind Kind() const {return FigurePolyLine;}
};


// ------ Polygon -------------------------------------------------------------

class Polygon : public PolyLine 
{
public:
	Polygon (Boolean initFill = False) 	: PolyLine()  {fill=initFill;}

	Polygon (const Polygon &copy) 		: PolyLine(copy) 
						             {fill=copy.fill;}

	Polygon (const PointSequence &copy, 
		Boolean initFill)		: PolyLine(copy) 
						              {fill=initFill;}

	Polygon &operator=(const Polygon &ass) 
	{
		PolyLine::operator=(ass); 
		return *this; 
	}

	Figure *Clone() const                     {return new Polygon(*this);}

	void Draw () const;
	void PSOut () const;
	void VROut () const;
	
	int operator==(const Figure &compare) const;

	void SetFill (Boolean fillValue)                   {fill = fillValue;}

protected:
	FigureKind Kind() const                        {return FigurePolygon;}

private:
	Boolean fill;
};


// ------ Ellipse ------------------------------------------------------------- 

class Ellipse : public Figure 
{
public:
	Ellipse ();
	Ellipse (const Point&	initCenter, 
	         const Point&	initOuter, 
	         Boolean 	initFill = False,
	         float 		initBeginAngle = 0, 
	         float 		initEndAngle = 360);

	Ellipse (const Point& 	initCenter, 
	         float 		initRadius, 
	         Boolean 	initFill = False,
	         float 		initBeginAngle = 0, 
	         float 		initEndAngle = 360);

	Ellipse (const Line& 	initFirstAxis, 
	         const Line& 	initSecondAxis,
 	         Boolean 	initFill = False,
	         float 		initBeginAngle = 0, 
	         float 		initEndAngle = 360);

	Ellipse (const Ellipse &copy);

	Ellipse &operator=(const Ellipse &ass);

	Figure *Clone() const {return new Ellipse(*this);}

	void Draw () const;
	void PSOut () const;
	void Transform (const Transformation &T);
	int operator==(const Figure &compare) const;
	void CalcBoundingBox (BoundingBox &bb) const;

	void SetFill (Boolean fillValue)                   {fill = fillValue;}

	Line& GetFirstAxis()                               {return firstAxis;}
	Line& GetSecondAxis()                             {return secondAxis;}
	float GetBeginAngle()                             {return beginAngle;}
	float GetEndAngle()                                 {return endAngle;}
	Boolean GetFill()                                       {return fill;}
	
protected:
	FigureKind Kind() const                        {return FigureEllipse;}

private:
	Line firstAxis, secondAxis;
	float beginAngle, endAngle;
//	Point  center, outer;
	Boolean fill;

};


// ------ Trans3DFigure -------------------------------------------------------

class Trans3DFigure : public Figure
{
public:
	Trans3DFigure ();
	Trans3DFigure (const Trans3DFigure& copy);
	Trans3DFigure& operator= (const Trans3DFigure& ass);
	
	void Draw () const;		// empty
	void PSOut () const;	// empty
	
	void VROut () const; 	
		// Outputs transformation matrix. Does NOT call Figure::VROut().
	
	void Transform (const Transformation &T);
	int operator==(const Figure &compare) const;
	void CalcBoundingBox (BoundingBox &bb) const; // empty
		
protected:

private:	
	void InitTransformation ();
	Transformation *trans;
};



// ------ Sphere --------------------------------------------------------------

class Sphere : public Trans3DFigure
{
public:
	Sphere (double radius);
	Sphere (const Sphere& copy);
	Sphere& operator= (const Sphere& ass);
	
	Figure *Clone() const {return new Sphere(*this);}

	void VROut () const; 
	
	int operator==(const Figure &compare) const;
	
protected:
	FigureKind Kind() const                        {return FigureSphere;}

private:
	double radius;
};
	
	
// ------ Cube --------------------------------------------------------------

class Cube : public Trans3DFigure
{
public:
	Cube (double width, double height, double depth);
	Cube (const Cube& copy);
	Cube& operator= (const Cube& ass);
	
	Figure *Clone() const {return new Cube(*this);}

	void VROut () const; 
	
	int operator==(const Figure &compare) const;
	
protected:
	FigureKind Kind() const                        {return FigureCube;}

private:
	double width, height, depth;
};
	
	
	
// ------ Cone --------------------------------------------------------------

class Cone : public Trans3DFigure
{
public:
	enum Parts {all, sides, bottom};
	Cone (double bottomradius, double height, Parts parts=all);
	Cone (const Cone& copy);
	Cone& operator= (const Cone& ass);
	
	Figure *Clone() const {return new Cone(*this);}

	void VROut () const; 
	
	int operator==(const Figure &compare) const;
	
protected:
	FigureKind Kind() const                        {return FigureCone;}

private:
	double bottomRadius, height;
	Parts parts;
};
	

// ------ Cylinder -----------------------------------------------------------

class Cylinder : public Trans3DFigure
{
public:
	enum Parts {all, sides, bottom, top};
	Cylinder (double radius, double height, Parts parts=all);
	Cylinder (const Cylinder& copy);
	Cylinder& operator= (const Cylinder& ass);
	
	Figure *Clone() const {return new Cylinder(*this);}

	void VROut () const; 
	
	int operator==(const Figure &compare) const;
	
protected:
	FigureKind Kind() const                        {return FigureCylinder;}

private:
	double radius, height;
	Parts parts;
};
	
	
// ------ Native -------------------------------------------------------------

class Native : public Trans3DFigure
{
public:
	Native (Textblock* definition);
	Native (const Native& copy);
	Native& operator= (const Native& ass);
	
	Figure *Clone() const {return new Native(*this);}

	void VROut () const; 
	
	int operator==(const Figure &compare) const;
	
protected:
	FigureKind Kind() const                        {return FigureNative;}

private:
	Textblock* definition;
};
	
	
// ------ PointLight ----------------------------------------------------------

class PointLight : public Figure
{
public:
	PointLight (const Point &p)	: position(p)                           {}

	PointLight (const PointLight &copy)	: Figure ((Figure &)copy), 
				  position(copy.position)                           {}

	PointLight &operator= (const PointLight &ass) 
	{
		position.operator=(ass.position); 
		Figure::operator=((Figure &)ass);
		return *this;
	}
	
	Figure *Clone() const {return new PointLight(*this);}

	virtual void Transform (const Transformation &T);

	void Draw () const  {}
	void PSOut () const {}
	
	void VROut () const; 
	
	int operator==(const Figure &compare) const;
	void CalcBoundingBox (BoundingBox &bb) const {}
	
protected:
	FigureKind Kind() const                        {return FigurePointLight;}

private:
	Point position;
};
	


// ------ FigureSet ----------------------------------------------------------- 

	// A FigureSet is a container for Figures with restricted access. 
	// Note that each figure in the list has its own identity and is 
	// _not_ just a pointer to another object.

typedef SList<Figure *> FigureList;

class FigureSet : private FigureList
{
public:
	FigureSet () 	: FigureList()                                      {}

	FigureSet (const FigureSet &theSet);

	FigureSet &operator=(const FigureSet &theSet);

	void Include (const Figure &element);

	void Add (const FigureSet &theSet);

	Figure* Contains (const Figure &element) const;
	
		// Returns a pointer to an element in the figure set that is
		// equal to 'element'. Returns 0 if no such element exists.
	
	void Exclude (Figure* element); 
	
		// Deletes the given 'element', provided that exactly this
		// element, e.g. this pointer, is stored within the figure set.

	Figure *GetAt (uint at) const 
	{
		Figure **res = FigureList::GetAt(at);
		if (res) return *res; else return 0;
	}

	Figure *GetNext () const 
	{
		Figure **res = FigureList::GetNext();
		if (res) return *res; else return 0;
	}
	
	int Length () {return FigureList::Length();}	

	void Draw () const;
	void PSOut () const;
	void VROut () const;
	
	void Transform (const Transformation &T);
	void CalcBoundingBox (BoundingBox &bb) const;

private:
	void AppendPhysical (const FigureSet &theSet);
};

typedef FigureSet PartSet;
typedef Figure    Part;

#endif /* !defined (__FIGURES_H__) */

