//
// hyper.cc  --  CollageVR
//
// Classes for representing the hyperedges of a collage
//
// Author: N. Schwabe, S.Taubenberger
// Date of creation : 06.04.94
// Last modification: 11 Dec 96 ns
//
// University of Bremen, Germany
//

#include <stdio.h>
#include <string.h>
#include <iostream.h> // test
#include "psout.h"
#include "vrout.h"
#include "hyper.h"
#include "cgi.h"


// ------- Edge ---------------------------------------------------------------

Edge::Edge (const EdgeText& initLabel) : label (initLabel)
{
  eqclass = 0;
}

Edge::Edge (const Edge& copy) : label (copy.label)
{
  eqclass = copy.eqclass;
}

Edge &Edge::operator= (const Edge& theEdge) 
{
  if (this != &theEdge) {
    label = theEdge.label;
    eqclass = theEdge.eqclass;
  };
  return *this;
}

int Edge::operator== (const Edge &compare) const
{
	return (eqclass == compare.eqclass);
}
		

void Edge::SetLabel (const EdgeText& theLabel) 
{
  label = theLabel;
}


// ------- Hyperedge ----------------------------------------------------------

int Hyperedge::operator== (const Edge &edge) const
{
	if (! Edge::operator== (edge))
		return 0;
		
	Hyperedge *hyper = (Hyperedge *)&edge; // <C> down-cast
//	cout << "Hyperedge::operator==:" << endl;
//	cout << "    self = " << *(const PointSequence *)this << endl;
//	cout << "    cmp  = " << *(const PointSequence *)hyper << endl;

#if 0
	return CyclicIdentical (*hyper);
	// this is not really appropriate for hyperedges since rotations or
	// mirroring does significantly change the result of further replacements.
#else
	// Test for exact identity:
	Point *p = GetAt(0), *q = hyper->GetAt (0);
	while (p && q)
	{
		if (*p != *q)
			return False;
		p = GetNext ();
		q = hyper->GetNext ();
	}
	return q == p; // both 0
#endif
}	

void Hyperedge::Transform (const Transformation &T) 
{
  Point *cur = GetAt(0);
  while (cur) {
    T.Transform (*cur, cur);
    cur = GetNext();
  }
}

Transformation *Hyperedge::FindTrans 
	(const PointSequence *s, const TransFinder *finder, Boolean inverse) 
{
	if (! s) {
		cerr << "Hyperedge::FindTrans: Invalid (NULL) PointSequence." << endl;
		return 0;
	} 
	if (inverse)
		return finder->FindTrans(*this, *s);
	else
		return finder->FindTrans (*s, *this);
}

void Hyperedge::FreeTrans (Transformation *t)
{
	delete t;
}


void Hyperedge::Draw () const
{
  uint length=0, i;
  Point m(0,0); // 2D point
  Point *cur = GetAt(0);


  while (cur) {
    for (i=0; i < 2; i++)
      m.Set (i, m.Get(i) + cur->Get(i));
    length++;
    cur = GetNext();
  }
  if (length==0) return;

  for (i=0; i < 2; i++)
    m.Set (i, m.Get(i) / length);

  Coordinate mx=m.Get(0), my=m.Get(1);

  for (i=0; i < length; i++)
  {
    cur = GetAt(i);
    Coordinate tx=cur->Get(0), ty=cur->Get(1);
    /* tentacle line: */
    //theScreen->FgColor (color);
    theScreen->FgColor (255, 0, 0); // <T>
    theScreen->Line (mx, my, tx, ty);
    theScreen->FilledEllipse (tx/*-3*/, ty/*-3*/, 6, 6, False);
    /* number of tentacle: */
    char str[10];
    sprintf (str, "%i", i+1);
#if False
    // scales version:
    Coordinate dx;
    if (tx < mx)
      dx = -8;
    else
      dx = 0;
    theScreen->Text (tx+(mx-tx)/2+dx, ty+(my-ty)/2+4, str); 
#else
    // new version:
    theScreen->BoxText (tx+(mx-tx)/2, ty+(my-ty)/2, 10, 10, False, str);
#endif
  }

  /* label: */
  // theScreen->FgColor (color);
  theScreen->FgColor (255, 0, 0); // <T>
  theScreen->BoxText (mx, my, 18, 18, True, GetLabel().Get());

}

void Hyperedge::PSOut () const
{
	AttachSequence::PSOut ();
	int i;
	Coordinate minx = GetAt(0)->Get(0);
	Coordinate miny = GetAt(0)->Get(1);
	Coordinate maxx = minx;
	Coordinate maxy = miny;
	Coordinate tmpx;
	Coordinate tmpy;
	int length = Length();
	for (i = 1; i < length; i++) {
		tmpx = GetAt(i)->Get(0);
		tmpy = GetAt(i)->Get(1);
		if (tmpx < minx)
			minx = tmpx;
		else
			if (tmpx > maxx)
				maxx = tmpx;
		if (tmpy < miny)
			miny = tmpy;
		else
			if (tmpy > maxy)
				maxy = tmpy;
	};
	Coordinate midx = (maxx + minx) / 2;
	Coordinate midy = (maxy + miny) / 2;
	// Ausgabe des Labels als PS-String
	thePS.Out (midx);
	thePS.Out (midy);
	char tmp [10];
	sprintf (tmp,"(%s) H", GetLabel().Get());
	thePS.Out (tmp);
}


void Hyperedge::VROut () const
{
	int i;
	Coordinate minx = GetAt(0)->Get(0);
	Coordinate miny = GetAt(0)->Get(1);
	Coordinate minz = GetAt(0)->Get(2);
	Coordinate maxx = minx;
	Coordinate maxy = miny;
	Coordinate maxz = minz;
	Coordinate tmpx;
	Coordinate tmpy;
	Coordinate tmpz;
	int length = Length();
	for (i = 1; i < length; i++) 
	{
		tmpx = GetAt(i)->Get(0);
		tmpy = GetAt(i)->Get(1);
		tmpz = GetAt(i)->Get(2);
		
		if (tmpx < minx)
			minx = tmpx;
		else
			if (tmpx > maxx)
				maxx = tmpx;
				
		if (tmpy < miny)
			miny = tmpy;
		else
			if (tmpy > maxy)
				maxy = tmpy;
				
		if (tmpz < minz)
			minz = tmpz;
		else
			if (tmpz > maxz)
				maxz = tmpz;				
	};
	Coordinate midx = (maxx + minx) / 2;
	Coordinate midy = (maxy + miny) / 2;
	Coordinate midz = (maxz + minz) / 2;

		
	//
	// Label
	//
	theVR.Out ("\nSeparator {\n");
	theVR.Out
    ("    Material {\n"
     "        ambientColor 1 0.8 0.8\n"
     "        diffuseColor 1 0.8 0.8\n"
     "        specularColor 1 0 0\n"
     "        emissiveColor 1 0 0\n"
     "        shininess 0.8\n"
     "        transparency 0\n"
     "    }\n"
     "    Translation { translation ");
    theVR.Out (midx); 
    theVR.Out (midy);
    theVR.Out (midz);
    theVR.Out 
    ("}\n"
     "    AsciiText {\n"
     "        string \"");
    theVR.Out (GetLabel()); 
    theVR.Out
    ("\"\n"
     "        spacing 1\n"
     "        justification CENTER\n"
     "    }\n");
	theVR.Out ("} #Separator\n");
     
	//
	// Attach points
	//
	theVR.Out ("\nSeparator {\n");
	theVR.Out ("    Coordinate3 { point [ ");	
	for (i = 0; i < length; i++) 
	{
		tmpx = GetAt(i)->Get(0);
		tmpy = GetAt(i)->Get(1);
		tmpz = GetAt(i)->Get(2);	
		
		theVR.Out (tmpx);
		theVR.Out (tmpy);
		theVR.Out (tmpz);
		if (i < length-1)
			theVR.Out (",");

	}
	theVR.Out ("] }\n");
	theVR.Out ("    PointSet { numPoints ");
	theVR.Out (length);
	theVR.Out ("}\n");	
	theVR.Out ("} #Separator\n");

	//
	// Tentacles
	//     	
	theVR.Out ("\nSeparator {\n");
	theVR.Out ("    Material {diffuseColor 0 1 1}\n    Coordinate3 { point [ ");
	for (i = 0; i < length; i++) 
	{
		tmpx = GetAt(i)->Get(0);
		tmpy = GetAt(i)->Get(1);
		tmpz = GetAt(i)->Get(2);	
		
		theVR.Out (tmpx);
		theVR.Out (tmpy);
		theVR.Out (tmpz);
		
		theVR.Out (", ");
	}
	theVR.Out (midx);
	theVR.Out (midy);
	theVR.Out (midz);
	theVR.Out (" ] }\n    IndexedLineSet { coordIndex [ ");
	for (i = 0; i < length; i++)
	{
		theVR.Out (i);
		theVR.Out (",");
		theVR.Out (length);
		theVR.Out (",");
		theVR.Out (-1);
		if (i < length-1)
			theVR.Out (",");
	}
	theVR.Out (" ] }\n");	
	theVR.Out ("} #Separator\n");	
}



ostream& operator<< (ostream& s, const Hyperedge& h)
{
	return s << h.GetLabel() << "[" << h.GetClass() << "]:" << AttachSequence(h);
}



// ------- AnnotatedEdge ------------------------------------------------------------

int AnnotatedEdge::operator== (const Edge &edge) const
{
	if (! Edge::operator== (edge))
		return 0;
				
	AnnotatedEdge *anno = (AnnotatedEdge *)&edge; // <C> down-cast
	
	// Test if the transformations maps linear independent 2-d points to the    
	// same location.   <T>
	
	Point a0(2), a1(2), a2(2);  // C++ restricts use of an array here
	Point r1(2), r2(2);
	
	a0.Set(0, 1.0);  a0.Set(1, 0.0);
	a1.Set(0, 0.0);  a1.Set(1, 1.0);
	a2.Set(0, 1.0);  a2.Set(1, 1.0);
	
	this->trans->Transform (a0, &r1);
	anno->trans->Transform (a0, &r2);
	if (r1 != r2)
		return False;

	this->trans->Transform (a1, &r1);
	anno->trans->Transform (a1, &r2);
	if (r1 != r2)
		return False;

	this->trans->Transform (a2, &r1);
	anno->trans->Transform (a2, &r2);
	return (r1 == r2);
}

void AnnotatedEdge::Transform (const Transformation &t)
{
	if (trans) {
		Transformation *n = t.Chain (trans);
		delete trans;
		trans = n;
	}
	else
		cerr << "AnnotatedEdge::Transform: Chain broken." << endl;
}


// ------- EdgeSet ------------------------------------------------------------------

EdgeSet::EdgeSet () : EdgePList() 
{
}

EdgeSet::EdgeSet (const EdgeSet &copy) 
{
	Edge *cur = copy.GetAt(0);
	while (cur) {
		Include (cur->Copy());	
		cur = copy.GetNext();
	}
}


EdgeSet& EdgeSet::operator= (const EdgeSet &theSet)
{
	if (this != &theSet) {
		// delete all own edges:
		Edge **curp = EdgePList::GetAt(0);
		while (curp) {
			EdgePList::Delete();
			delete *curp;
			curp = EdgePList::GetAt(0);
		}
		// insert new edges:
		Edge *cur = theSet.GetAt(0);
		while (cur) {
			Include (cur->Copy());	
			cur = theSet.GetNext();
		}
	};
	return *this;
}


void EdgeSet::Include (Edge *newEdge)
{
  EdgePList::Append (newEdge);
} 

void EdgeSet::Exclude (Edge* theEdge) 
{
  Edge *cur = GetAt(0);
  while (cur && cur != theEdge) cur = GetNext();
  if (cur) {
    EdgePList::Delete();
    delete cur;
  }
  else
    cerr << "EdgeSet::Exclude: Edge not found." << endl;
}

void EdgeSet::Add (const EdgeSet &theSet) 
{
	Edge *cur = theSet.GetAt(0);
	while (cur) {
		Include (cur->Copy());	
		cur = theSet.GetNext();
	}
}

Edge *EdgeSet::GetAt (uint at) const
{
  Edge **cur = EdgePList::GetAt(at);
  if (cur)
    return *cur;
  return 0;
}

Edge *EdgeSet::GetNext () const
{
  Edge **cur = EdgePList::GetNext();
  if (cur)
    return *cur;
  return 0;
}

void EdgeSet::Transform (const Transformation &T) 
{
  Edge *cur = GetAt(0); 
  while (cur) {
    cur->Transform(T);
    cur = GetNext();
  }
}


void EdgeSet::Draw () const
{
  Edge *cur = GetAt(0);
  while (cur) {
    cur->Draw();
    cur = GetNext();
  }
}

void EdgeSet::PSOut () const
{
  Edge *cur = GetAt(0);
  while (cur) {
    cur->PSOut();
    cur = GetNext();
  }
}

void EdgeSet::VROut () const
{
  Edge *cur = GetAt(0);
  while (cur) {
    cur->VROut();
    cur = GetNext();
  }
}

void EdgeSet::CalcBoundingBox (BoundingBox &bb) const
{
  Edge* cur = GetAt(0);
  while (cur) {
    cur->CalcBoundingBox (bb);
    cur = GetNext();
  }
}

Edge* EdgeSet::Contains (const Edge &test) const
{	
	Edge* cur = this->GetAt(0);
	
	while (cur) 
	{
		if (cur->operator==(test))
			return cur;
		cur = GetNext ();
	}
	return 0;
}


