#ifndef __HYPER_H__
#define __HYPER_H__
//
// hyper.h  --  CollageVR
//
// Classes for representing the hyperedges of a collage
//
// Authors            : Nils Schwabe
// Date of creation   : 06 Apr 94 ns
// Last modification  : 04 Dec 96 ns
//
// University of Bremen, Germany
//

#ifndef __TRANS_H__
#include "trans.h"
#endif


// ------- Edge ---------------------------------------------------------------

// 'Edge' is the abstract base class for the different implementations of 
// 'hyperedge concepts' in CollageTwo.
//
// An edge in this sense is a labeled object that 
// - can be transformed 
//   similar to the transformation of the 'parts' of a collage, and that
// - can evaluate a transformation
//   that maps a given pin-point sequence on itself using a specified
//   transformation class.
//
// Notes:
// From the OOP viewpoint, an 'Edge' is free to decide what a 'transformation'
// of itself means and what kind of transformations to evaluate. However,
// from the Collage-Grammar viewpoint, there exist some assumptions about
// the behaviour of concrete 'Edge' implementations, such as Hyperedges
// or Annotated Edges. An 'AnnotatedEdge', for example, completely ignores
// the parameters passed to its transformation evaluation routine and always
// returns its stored transformation. This behaviour is always correct 
// from the OOP viewpoint (the edge has its own will), but may be incorrect
// from the Collage-Grammar viewpoint in a general context. In this example,
// ignoring the actual parameters is correct only if the corresponding
// 'Hyperedge' (from which the 'AnnotatedEdge' was created) would have
// returned the same transformation given the same actual parameters.


typedef SString<char> EdgeText;
	// External label (input from and visible to the user)

typedef int EdgeClass; 
	// Internal label (= equivalence class) that is used for assigning
	// rules to edges.

class Edge
{
public:
	Edge (const EdgeText& initLabel);

		// Construct edge with a initial label.

	Edge (const Edge& theEdge);

		// Construct edge from another edge.

	virtual ~Edge ()                                                       {}

		// Provides a virtual destructor.

	virtual Edge* Copy() = 0; 

		// Virtual copy constructor.

	Edge &operator= (const Edge& theEdge);
	
		// Assignment.
		
	virtual int operator== (const Edge& compare) const;
	
		// Comparison. Returns nonzero value if both edges are equal.
	
	void SetLabel (const EdgeText& theLabel); 
	const EdgeText& GetLabel() const                         {return label;}

		// Set or get the label of the edge.

	void SetClass (EdgeClass ec)                             {eqclass = ec;}
	EdgeClass GetClass () const                            {return eqclass;}

		// Set or get the class of the edge.
	
	virtual void Transform (const Transformation &T) = 0; 

		// Transform the edge with transformation 'T'.

	virtual Transformation *FindTrans 
		(const PointSequence *s, const TransFinder *finder,
		 Boolean inverse) = 0;

		// Return a pointer to a transformation that maps 's' to the 
		// edge using the 'finder' class or NULL if no such 
		// transformation exists.

	virtual void FreeTrans (Transformation *t) = 0;

		// Assuming that 't' is a transformation obtained by GetTrans(),
		// free this transformation if it had to be allocated.

	// Optional features that may or may not be supported by an edge:

	virtual void Draw () const                                            {}
	virtual void PSOut () const                                           {}
	virtual void VROut () const                                           {}
	virtual void CalcBoundingBox (BoundingBox &bb) const                  {}

private:
	EdgeText label;
	EdgeClass eqclass;
};



// ------- Hyperedge ---------------------------------------------------------

// A 'Hyperedge' is a labeled PointSequence. The points can be transformed and
// output. The Transform() method of this edge type applies the transformation
// to the point sequence.

typedef PointSequence AttachSequence;

class Hyperedge : public Edge, public AttachSequence 
{
public:
	Hyperedge (const EdgeText& initLabel)
			: Edge(initLabel), AttachSequence()        {color = 0;}
	Hyperedge (const Hyperedge &e) 
		: Edge (e), AttachSequence(e)
		{color = e.color;}
	Edge* Copy () 
		{return new Hyperedge(*this);}
	Hyperedge &operator= (const PointSequence &s) 
		{PointSequence::operator= (s); return *this;}
		
	virtual int operator== (const Edge& compare) const;

	void Transform (const Transformation &T);

	Transformation *FindTrans 
		(const PointSequence *s, const TransFinder *finder, Boolean);
	void FreeTrans (Transformation *t);

	void Draw () const;
	void PSOut () const;
	void VROut () const;

	void CalcBoundingBox (BoundingBox &bb) const
		{AttachSequence::CalcBoundingBox (bb); }

	virtual ColorIndex GetColor() const {return color;}
	virtual void SetColor (ColorIndex c) {color = c;}
private:
	ColorIndex color;

};

ostream& operator<< (ostream& s, const Hyperedge &h);


// ------- AnnotatedEdge -----------------------------------------------------

// An AnnotatedEdge is a labeled transformation. The Transform() method
// of this edge type replaces the stored transformation t such that it becomes
// t(T), where T is the parameter of Transform().

class AnnotatedEdge : public Edge
{
public:
	AnnotatedEdge (const EdgeText &initLabel, Transformation *t) 
		: Edge(initLabel) {trans = t;}
	AnnotatedEdge (const AnnotatedEdge &copy) : Edge(copy) {
		trans = copy.trans->Copy(); }
	~AnnotatedEdge () {delete trans;}
	Edge* Copy () {return new AnnotatedEdge (*this);}

	virtual int operator== (const Edge& compare) const;
	
	void Transform (const Transformation &T); 

	Transformation *FindTrans 
		(const PointSequence *s, const TransFinder *finder, Boolean inverse) {
		return trans;
	}

	void FreeTrans (Transformation *t) {}

private:
	Transformation *trans;

};


// ------- EdgeSet ------------------------------------------------------------

typedef SList<Edge *> EdgePList;

class EdgeSet : private EdgePList
//-----------------------------------------------------------------------------
// An EdgeSet is a container for edges with restricted access.
{
public:
	EdgeSet ();
	EdgeSet (const EdgeSet &theSet); // makes deep copy
	EdgeSet &operator= (const EdgeSet &theSet); // makes deep copy
	virtual ~EdgeSet () {}

	virtual void Include (Edge *newEdge); // pointer is stored!
	virtual void Exclude (Edge* theEdge); // edge is also destroyed
	virtual void Add (const EdgeSet &theSet); // makes deep copy
	
	Edge* Contains (const Edge &edge) const; // uses operator==
	
	int Length () {return EdgePList::Length();}
	virtual Edge *GetAt (uint at) const;
	virtual Edge *GetNext () const;
	int GetIndex () const {return EdgePList::GetIndex();}

	void Transform (const Transformation &T);

	void Draw () const;
	void PSOut () const;
	void VROut () const;
	void CalcBoundingBox (BoundingBox &bb) const;
};


#endif /* !defined(__HYPER_H__) */

