//
// matrix.cc  --  Collage System Collage-Two
//
// Matrix class
//
// Author            : Nils Schwabe
// Date of creation  : Feb 94
// Last modification : 20 Nov 95
//
// University of Bremen, Germany
//

#include <iostream.h>
#include <string.h>
#include <math.h>
#include "matrix.h"


Matrix::Matrix (uint cntRows, uint cntColumns)
{
	e = new ME[cntRows*cntColumns];
	for (uint i=0; i < cntRows*cntColumns; i++)
		e[i] = 0;
	rows = _rows = cntRows;
	columns = _columns = cntColumns;
}


Matrix::Matrix (ME a1,ME a2,ME a3, ME b1,ME b2,ME b3, ME c1,ME c2,ME c3)
{
	e = new ME[9];
	e[0]=a1; e[1]=a2; e[2]=a3;
	e[3]=b1; e[4]=b2; e[5]=b3;
	e[6]=c1; e[7]=c2; e[8]=c3;
	rows = _rows = 3;
	columns = _columns = 3;
}


Matrix::~Matrix ()
{
	delete [] e;
}


Matrix::Matrix (const Matrix &source)
{
	e = new ME[source._rows*source._columns];
	for (uint i=0; i < source._rows*source._columns; i++)
		e[i] = source.e[i];
	rows = source.rows;
	_rows = source._rows;
	columns = source.columns;
	_columns = source._columns;
}


void Matrix::operator=(const Matrix &source)
{
	int cntrows = (rows < source.rows ? rows : source.rows);
	int cntcols = (columns < source.columns ? columns : source.columns);
	for (uint i=0; i < cntrows; i++)
		for (uint j=0; j < cntcols; j++)
			Set (i,j, source.Get(i,j));
}


Boolean Matrix::Resize (uint cntRows, uint cntColumns)
{
	if (_rows < cntRows || _columns < cntColumns) 
		return False;
	rows = cntRows;
	columns = cntColumns;
	return True;
}


void Matrix::Multiply (const Matrix &v, Matrix *result) const
{
	if (columns != v.rows) 
	{
		cerr << "Matrix::Multiply: columns != v.rows" << endl;
		return;
	}
	
	ME s;
	for (uint i = 0; i < rows; i++) 
	{
		for (uint j = 0; j < v.columns; j++) 
		{
			s = 0;
			for (uint k = 0; k < columns; k++)
				s = s + Get(i,k) * v.Get(k,j);
			result->Set (i,j,s);
		}
	}
}


void Matrix::Multiply (const Point &p, Point *result) const
{
	if (columns != p.GetDim()) 
	{
		cerr << "Matrix::Multiply: columns != p.GetDim()" << endl;
		return;
	}
	
	ME s;
	for (uint i = 0; i < rows; i++) 
	{
		s = 0;
		for (uint k = 0; k < columns; k++)
			s = s + Get(i,k) * p.Get(k);
		result->Set (i,s);
	}
}


void Matrix::PivotStep (uint prow, uint pcol)
{
	uint i,j;
	ME alpha;

	alpha = 1 / Get(prow,pcol);
	Set (prow, pcol, 1);

	// Apply rectangle-rule first, because it needs the unchanged
	// information of the pivot row/column elements:
	for (i=0; i < rows; i++)
		for (j=0; j < columns; j++)
			if (i != prow && j != pcol)
				Set (i, j, Get(i,j) - alpha * Get(prow,j) * Get(i,pcol));

	// Apply pivot row/column rules:
	for (i=0; i < rows; i++)
		for (j=0; j < columns; j++)
			if (i == prow)
				Set (i, j, alpha * Get(i,j));
			else if (j == pcol)
				Set (i, j, -alpha * Get(i,j));
}

#if 1

// This implementation uses a backtracking algorithm
// to find a valid pivot column permutation. Switch the 1 above
// to 0 to generate the alternate implementation (see below).

Boolean Matrix::FindPivotColumns (uint row, signed char *pcol)

// Finds a permutation of nonzero columns in 'row' and all rows below.
// The column indices are copied into 'pcol'.
// Result == "a valid permutation was found". 

{
	uint j;

	pcol[row] = -1;
	for (j=0; j < columns; j++)
		if (Get(row,j)) 
		{
			Boolean ok=True;
			for (uint k=0; k < row; k++)
			if (pcol[k]==j) 
			{
				ok=False;
				break;
			};
			if (ok) 
			{
				pcol[row] = j;
				if (row == rows-1 || FindPivotColumns (row+1, pcol))
					return True;
			}
		}
	pcol[row] = -1;
	return False;
}

#else

// The following is an implementation of the algorithm 18
// described in 'Fundamental algorithms, The art of computer
// programming', second edition, p. 304, by D.E.Knuth 

Boolean Matrix::FindPivotColumns (uint dummy, signed char *pcol)

// Finds a permutation of nonzero columns.
// The column indices are copied into 'pcol'.
// Result == "a valid permutation was found". 

{
	uint i, j;

	for (i=0; i < rows; i++) 
	{
		ME max=0/*mEpsilon*/, e;

		pcol[i] = -1;

		for (j=0; j < columns; j++)
			if ((e=fabs(Get(i,j))) > /* zuvor: >= */ max) 
			{
				Boolean valid=True;
				for (int k=0; k < i; k++)
					if (pcol[k]==j) 
					{
						valid = False;
						break;
					}
				if (valid) 
				{
					max = e;
					pcol[i] = j; // optimistic assumption
				}
			}
		if (pcol[i] < 0) 
			return False; // matrix is singular
	}
	return True;
}

#endif  // alternate implementations


Boolean Matrix::Invert ()
{
	signed char *pcol = new signed char[rows];
	memset (pcol, -1, rows);   
  
	if (FindPivotColumns (0,pcol)) 
	{
		uint i, j;

		// do pivot step in each row:
		for (i=0; i < rows; i++) 
		{
			// cout << "Pivot " << i << ", " << (int)pcol[i] << "\n"; 
			PivotStep (i, pcol[i]);
		}

		// Permutate rows in columns according to pivot elements:
   
		Matrix cur(rows,columns);

		for (i=0; i < rows; i++)
			for (j=0; j < columns; j++)
				cur.Set (pcol[i],j, Get(i,j));				
		*this = cur;

		for (j=0; j < columns; j++)
			for (i=0; i < rows; i++)
				cur.Set (i,j, Get(i,pcol[j]));
		*this = cur;

		return True;
	}
	return False;
}


void Matrix::TestOut(char *header)
{
	uint i,j;
	cout << "---" << header << "-----------------\n";
	for (i=0; i < rows; i++) 
	{
		for (j=0; j < columns; j++)
			cout << (fabs(Get(i,j)) > mEpsilon ? Get(i,j) : 0) << "\t";
		cout << "\n";
	}
	cout << "\n"; 
}

