//
// point.cc  --  CollageTwo
//
// Point, PointSequence 
//
// Author             : Nils Schwabe
// Date of creation   : Feb 94
// Last modification  : 17 Dec 95
//
// University of Bremen, Germany
//

#include <string.h>
#include <math.h>
#include <iostream.h>
#include "psout.h"
#include "vrout.h"
#include "point.h"


// ------ Point --------------------------------------------------------------- 

Point::Point (ushort dimensions)
{
	if (dimensions > cntEffDim)
		_x = new Coordinate[dimensions-cntEffDim];
	else
		_x = 0;
	dim = dimensions;
	int i;
	for (i=0; i < cntEffDim; i++)
		x[i] = 0;
	for (i=0; i < dim-cntEffDim; i++)
		_x[i] = 0;
}


Point::Point (Coordinate xx, Coordinate yy)
{
	_x = 0;
	dim = 2;
	x[0] = xx;
	x[1] = yy;
}


Point::Point (const Point &source)
{
	if (source.dim > cntEffDim)
		_x = new Coordinate[source.dim-cntEffDim];
	else
		_x = 0;
	dim = source.dim;
	for (int i=0; i < cntEffDim; i++)
		x[i] = source.x[i];
	if (_x) 
		memcpy (_x, source._x, (dim-cntEffDim)*sizeof(Coordinate));
}


void Point::operator=(const Point &source)
{
	ushort less = dim < source.dim ? dim : source.dim;
	for (int i=0; i < cntEffDim; i++)
	x[i] = source.x[i];
	if (less > cntEffDim) 
		memcpy (_x, source._x, (less-cntEffDim)*sizeof(Coordinate));
}


Point::~Point ()
{
	if (_x) 
		delete [] _x;
}


Coordinate Point::Get (uint index) const
{
	if (index < dim) 
	{
		if (index < cntEffDim)
			return x[index];
		else
			return _x[index-cntEffDim];
	}
	// cerr << "Point::Get: Invalid index " << index << "." << endl;
	return 0.0; 
}


void Point::Set (uint index, Coordinate newX)
{
	if (index < dim) 
	{
		if (index < cntEffDim)
			x[index] = newX;
		else
			_x[index-cntEffDim] = newX;
	}
#if 0	
	else
		cerr << "Point::Set: Invalid index " << index << "." << endl;
#endif		
}


int Point::operator==(const Point &cmp) const
{
	if (dim != cmp.dim) 
	{
		cerr << "Point::operator==: points have different dimensions." << endl;
		return 0;
	}
	
	ushort upto = (dim < cntEffDim ? dim : cntEffDim);
	ushort i;
	
	for (i=0; i < upto; i++)
		if (fabs(x[i]-cmp.x[i]) > pEpsilon) 
			return 0;
			
	for (i=0; i < dim-cntEffDim; i++)
		if (fabs(_x[i]-cmp._x[i]) > pEpsilon) 
			return 0;
			
	return 1;
}


int Point::operator!=(const Point &cmp) const
{
	if (dim != cmp.dim)
	{
		cerr << "Point::operator!=: points have different dimensions." << endl;
		return 1;
	}
	
	ushort upto = (dim < cntEffDim ? dim : cntEffDim);
	ushort i;
	for (i=0; i < upto; i++)
		if (fabs(x[i]-cmp.x[i]) > pEpsilon) 
			return 1;
		
	for (i=0; i < dim-cntEffDim; i++)
		if (fabs(_x[i]-cmp._x[i]) > pEpsilon) 
			return 1;
			
	return 0;
}


void Point::TestOut (const char *title) const
{
	cout << title << ": (";
	for (int i=0; i < dim-1; i++)
		cout << Get(i) << ", ";
	cout << Get(dim-1) << ")\n";
}


void Point::PSOut () const
{
	if (dim >= 2) 
	{
		thePS.Out (Get(0));
		thePS.Out (Get(1));
	}
	else 
	{
		thePS.Out (Get(0));
		thePS.Out (PSOutDim1Y);
	}
}


void Point::VROut () const
{
	// <M>
}


ostream& operator<< (ostream& s, const Point& p)
{
	s << "(";
	int i;
	for (i=0; i < p.GetDim()-1; i++)
		s << p.Get(i) << ",";
	return s << p.Get(i) << ")";
}


// ------ PointSequence ------------------------------------------------------- 

void PointSequence::PSOut () const
{
	int length=0;
	Point *p = GetAt(0);

	while (p) 
	{
		length++;
		p->PSOut();
		p = GetNext();
	}
	thePS.Out (length);
}

void PointSequence::VROut () const
{
	int length=0;
	Point *p = GetAt(0);

	while (p) 
	{
		length++;
		p->VROut();
		p = GetNext();
	}
}


void PointSequence::CalcBoundingBox (BoundingBox &bb) const
{
	Point *p = GetAt(0);
	while (p) 
	{
		bb.Calc (*p);
		p = GetNext();
	}
}


ostream& operator<< (ostream &s, const PointSequence &ps)
{
	int oldpos = ps.GetIndex();
	Point *p = ps.GetAt(0);
	while (p) 
	{
		s << *p;
		p = ps.GetNext();
	}
  	if (oldpos >= 0) 
  		(void) ps.GetAt (oldpos);	
  	return s;
}


Boolean PointSequence::CyclicIdentical (const PointSequence &cmp) const
{
	uint len, offs, i;
	Point *p, *c;
	
	if ((len = this->Length()) != cmp.Length())
		return False;
		
	if (len == 0)
		return True;
	
	// Check all possible "rotations":
	offs = 0;	
	while (offs < len)
	{
		for (i=0; i < len; i++)
		{
			p = GetAt (i);
			c = cmp.GetAt ((offs+i) % len);
			if (*p != *c)
				break;
		}
		if (i == len)
			return True;
		offs++;
	}
	
	// Check for "mirroring":
	offs = 0;	
	while (offs < len)
	{
		for (i=0; i < len; i++)
		{
			p = GetAt (i);
			c = cmp.GetAt ( len-1 - ((offs+i) % len) );
			if (*p != *c)
				break;
		}
		if (i == len)
			return True;
		offs++;
	}
			
	return False;
}

	
