#ifndef __SLISTS_H__
#define __SLISTS_H__
//
// slists.h
//
// Template class for single linked, non-intrusive lists with
// additional indexing mechanism.
//
// Author            : Nils Schwabe
// Date of creation  : Feb 94
// Last modification : Apr 94
//

#ifndef __BASIC_H__
enum Boolean {False, True};
typedef unsigned int uint;
typedef unsigned short ushort;
// if __BASIC_H__ is defined than these "basic"
// declarations" are expected to be already available.
#endif

class SListBase;
template<class T> struct TLink;

// ------ Single linked list template -----------------------------------------

template<class T>
class SList : private SListBase
{
public:
	SList () : SListBase() {}
	SList (const SList<T> &copy) : SListBase() {Append(copy);}
	SList<T>& operator=(const SList<T> &);
	~SList () {while (GetAt(0)) Delete();}
	
	void Append (T& a) {
		SListBase::Append (new TLink<T>(a)); }
		
	// Appends a new element 'a' to the end of the list.

	void Insert (T& a) {
		SListBase::Insert (new TLink<T>(a)); }
		
	// Inserts a new element 'a' at the beginning of the list.

	void InsertAt (uint index, T& a) {
		SListBase::InsertAt (index, new TLink<T>(a)); }
		
	// Inserts a new element 'a' at position 'index', where
	// index==0 implies the beginning of the list and an index
	// equal to the length of the list implies the end of the list.
	// Note that passing an invalid index is an error.

	inline void Append (const SList<T> &append);
	
	// Appends list 'append' to the list.
	
	inline T* GetAt (uint index) const;
	
	// Gets a pointer to the element at position 'index', where
	// the index range is 0 .. length of list minus 1. Note that
	// passing an invalid index is an error.
	
	inline T* Get () const;
	
	// Gets a pointer to the 'hot element' of the list. This is
	// (re-)defined by all operations. If there's no current hot
	// element or the list is empty then a null-pointer is returned.

	inline T* GetNext () const;
	
	// Gets a pointer to the element next to the hot element, if any.

	T* GetLast () const;
	
	// Gets a pointer to the last element of the list, if any.
 	
	inline Boolean Delete ();
	
	// Deletes the hot element. Answer False, if the list was empty
	// or there was no current hot element. The hot element is set
	// to the next element, if any.
 	
	inline void DeleteAt (unsigned int index);
	
	// Deletes the element at position 'index' (0..length-1). 
	// The hot element is set to the next element, if any.
	// Note that passing an invalid index is an error.

	uint Length () const {return SListBase::Length(); }
		
	// Returns current number of elements in the list. This call is fast.

	int GetIndex () const {return SListBase::GetIndex(); }
	
	// Returns the index of the hot element.

};
// ----------------------------------------------------------------------------

// Example of using the SList template class:
// SList<int> intlist;  // define an integer list
// intlist.Insert (53);
// ...

// The following code is implementation specific and only neccessary for 
// template resolution. Reading and understanding this is not required to 
// use the SList template class.

struct SLink
{
	SLink *next;
	SLink() {next = 0;}
	SLink (SLink *p) {next = p;}
};

template<class T>
struct TLink : public SLink
{
	T data;
	TLink (const T& a) : data (a) {}
#ifdef SLISTS_DIRECT_INSERTION
# error "test"
	TLink () {}
#endif
};

class SListBase
{
public:
	SListBase () {last=0; hot=0; length=0; hotnr=-1;}
	//SListBase (SLink* a) {last = a->next = a; hot=0;}

	void Append (SLink*);
	void Insert (SLink*);
	void InsertAt (uint index, SLink *);	

	SLink* GetAt (uint index);
	SLink* GetNext ();
	SLink* Get () const {return hot;}
	SLink* GetLast () {hotnr = length-1; return hot=last;}

	void DeleteAt (unsigned int index);
	Boolean Delete ();
	uint Length () const {return length;}
	int GetIndex() const {return hotnr;}
	
private:
	SLink *last; // last element, last->next = first element
	SLink *hot;
	int length, hotnr;	
};

template<class T> inline void SList<T>::Append (const SList<T> &append)
{
  T* cur = append.GetAt(0);
  while (cur) {
    Append (*cur);
    cur = append.GetNext();
  }
}

template<class T> inline SList<T>& SList<T>::operator=(const SList<T> &ass)
{
  if (&ass == this) return *this;
  if (GetAt(0)) while (Delete()); // delete all existing elements
  Append (ass);
  return *this;
}

template<class T> inline T* SList<T>::GetAt (uint index) const
{
  TLink<T> *t = (TLink<T> *)((SList<T> *)this)->SListBase::GetAt (index);
  if (t) return &t->data;
  else return 0;
}

template<class T> inline T* SList<T>::Get () const
{
  TLink<T> *t = (TLink<T> *)SListBase::Get ();
  if (t) return &t->data;
  else return 0;
}

template<class T> inline T* SList<T>::GetNext () const
{
  TLink<T> *t = (TLink<T> *)((SList<T> *)this)->SListBase::GetNext ();
  if (t) return &t->data;
  else return 0;
}

template<class T> inline T* SList<T>::GetLast () const
{
  TLink<T> *t = (TLink<T> *)((SList<T> *)this)->SListBase::GetLast ();
  if (t) return &t->data;
  else return 0;
}

template<class T> inline Boolean SList<T>::Delete ()
{
  TLink<T> *t = (TLink<T> *)SListBase::Get ();
  if (! t) return False;
  SListBase::Delete ();
  delete t;
  return True;
}

template<class T> inline void SList<T>::DeleteAt (unsigned int index)
{
  TLink<T> *t = (TLink<T> *)SListBase::GetAt (index);
  SListBase::DeleteAt (index);
  delete t;
}	

#endif /* !defined(__SLISTS_H__) */


