//
// trans.cc  --  Collage System Collage-Two
//
// Transformation classes
//
// Author            : Nils Schwabe
// Date of creation  : Feb 94
// Last modification : 20 Nov 95
//
// University of Bremen, Germany
//

#include <iostream.h>
#include "trans.h"


//----- MatrixTrans::Transform ------------------------------------------------

void MatrixTrans::Transform (const Point &p, Point *res) const
{
	ushort dim = p.GetDim();
	Point t(dim+1), r(dim+1);
	t = p;
	t.Set (dim, 1);
	Multiply (t, &r);
	*res = r;
}

Transformation *MatrixTrans::Chain (const Transformation *outer) const
{
	if (outer->InternalType() != transMatrix) 
		return 0;
	MatrixTrans *outa = (MatrixTrans *)outer;
	if (outa->GetColumns() != GetRows()) 
	{
		cerr << "MatrixTrans::Chain: Can't multiply (column/row mismatch)." << endl;
		return 0;
	}
	MatrixTrans *n = new MatrixTrans (GetRows(), outa->GetColumns());
	Multiply (*outa, n);
	return n;
}


//----- TranslationTrans::Transform -------------------------------------------

void TranslationTrans::Transform (const Point &p, Point *res) const
{
	for (int i=0; i < offset.GetDim(); i++) 
		res->Set (i, p.Get(i) + offset.Get(i));
}

Transformation *TranslationTrans::Chain (const Transformation *outer) const
{
	if (outer->InternalType() != transTranslation) 
		return 0;
	TranslationTrans *outa = (TranslationTrans *)outer;
	if (outa->offset.GetDim() != offset.GetDim()) 
	{
		cerr << "TranslationTrans::Chain: Mixing dimensions." << endl;
		return 0;
	}
	//cout << "This offset  = " << offset << endl;
	//cout << "Outer offset = " << outa->offset << endl;
	TranslationTrans *n = new TranslationTrans (offset.GetDim());
	n->offset = outa->offset; 
	Transform (n->offset, &n->offset); // n := outer + this
	//cout << "New offset   = " << n->offset << endl << endl;
	return n;
}


//----- GetLIColumnPermutation ------------------------------------------------ 

static Boolean GetLIColumnPermutation
    (const Matrix &m,
     ushort perm[])

// find a permutation of the columns in ms such that the 
// linear independent columns are placed at the beginning.
// Assumption: ms is of type (m x m+n), m>0, n>=0.
// perm[] must have at least m+n elements.
// Return values: 
// False = "There is definately no l.i. permutation"
// True = "If there is a l.i. perm., than this is one".

{
	// Implementation note:
	// should be done in a precomputing phase to optimize performance.

	Matrix ms = m;
	uint lenS = ms.GetColumns();
	int k;

	for (k=0; k < lenS; k++)
		perm[k] = k;

	for (k=0; k < ms.GetRows(); k++) 
	{
		int i=0;
		while (i < lenS && ms.Get(k,i) == 0) 
			i++;
			
		if (i == lenS) 
			return False; // not invertable

		// swap columns k and i:
		if (k != i) 
		{
			for (int r=0; r < ms.GetRows(); r++) 
			{
				ME cur = ms.Get(r,k);
				ms.Set (r,k, ms.Get(r,i));
				ms.Set (r,i, cur);
			}
			ushort cur = perm[k];
			perm[k] = perm[i];
			perm[i] = cur;
		} 

		for (int l=k+1; l < ms.GetRows(); l++) 
		{
			ME f = ms.Get(l,k) / ms.Get(k,k);
			for (int c=0; c < lenS; c++)
				ms.Set (l,c, ms.Get(l,c) - f*ms.Get(k,c));
		}
	}
	return True;
}


//----- PermutateColumns ------------------------------------------------------

static void PermutateColumns
    (Matrix &m,
     const ushort perm[])
{
	ushort col;

	for (col=0; col < m.GetColumns(); col++) 
	{
    	// swap columns col and perm[col]:
		if (col < perm[col]) 
		{
			for (int r=0; r < m.GetRows(); r++) 
			{
				ME cur = m.Get(r,col);
				m.Set (r,col, m.Get(r,perm[col]));
				m.Set (r,perm[col], cur);
			}
		} 
	}
}


//----- TestPointSequences ---------------------------------------------------

static Boolean TestPointSequences
	(const PointSequence &s,
	 const PointSequence &t,
	 ushort dim,
	 uint *length)

// Apply a general compatibility test to both lists that is needed by nearly
// all transformation finders. Answer 'True' if successful and length of 
// both lists in 'length'.

{
	// Get length of source sequence and check dimension:
	uint lenS=0;
	if (s.GetAt(0)) 
	{
		do 
		{
			if (dim != s.Get()->GetDim()) 
			{
    			cerr << "TransFinder: Dimension mismatch in source sequence:" << endl;
				cerr << "--> " << s << endl;
				return False;
			}	
			lenS++;
		} while (s.GetNext());
	}
	else 
	{
		cout << "TransFinder: Empty source list" << endl;
		return False;
	}
  
	// Get length of target sequence and check dimension:	
	uint lenT=0;
	if (t.GetAt(0)) 
	{
		do 
		{
			if (dim != t.Get()->GetDim()) 
			{
				cerr << "TransFinder: Dimension mismatch in target sequence:" << endl;
				cerr << "--> " << t << endl;
				return 0;
			}	
			lenT++;
		} while (t.GetNext());
	}
	else 
	{
		cout << "TransFinder: Empty target list" << endl;
		return False;
	}

	// Compare lengths: 
	if (lenS != lenT) 
	{	
		//cout << "TransFinder: Different point sequence lengths." << endl;
		return False;
	}

	*length = lenS;
	return True;
}


//----- TestTransformation ---------------------------------------------------

static Boolean TestTransformation
	(const PointSequence &s,
	 const PointSequence &t,
	 const Transformation *trans)

// Check if 'trans' maps all points of 's' to the corresponding points of 't'.

{
	uint dim = s.GetAt(0)->GetDim();

	Point res(dim);
	Point *ps = s.GetAt(0);	
	Point *pt = t.GetAt(0);
	while (ps && pt) 
	{
		trans->Transform (*ps, &res);
		if (res != *pt) 
		{
			cout << "TransFinder: Transformation test failed:\n";
			ps->TestOut ("  source");
			pt->TestOut ("  target");
        	res.TestOut ("  result");
			return False;
		}
		ps = s.GetNext();
		pt = t.GetNext();
	}
	return True;
}
	

//----- AffineFinder ---------------------------------------------------------

Transformation *AffineFinder::FindTrans
    (const PointSequence &s,
     const PointSequence &t) const
{
	// Get length of source sequence and check dimension:
	uint lenS;
	
	if (TestPointSequences (s, t, dim, &lenS) == False)
		return 0;
	else if (lenS < dim+1) 
	{
		cout << "AffineFinder: Lists not n+1-testable." << endl;
		return 0;
	}

	// Copy source sequence into matrix ms:
	int row, col;
	Point *ps = s.GetAt(0); 
	Matrix ms (dim+1, lenS);

	for (col=0; col < lenS; col++) 
	{
		for (row=0; row < dim; row++)
			ms.Set (row, col, ps->Get(row));
		ms.Set (dim, col, 1);
		ps = s.GetNext();
	}

	// Permutate the matrix columns::
	ushort *perm = new ushort[lenS];
	if (! GetLIColumnPermutation (ms, perm)) 
		return 0;
	PermutateColumns (ms, perm);

	// Copy target sequence into matrix mt:
	Point *pt = t.GetAt(0);
	Matrix mt (dim+1, lenS);

	for (col=0; col < lenS; col++) 
	{
		for (row=0; row < dim; row++)
			mt.Set (row, col, pt->Get(row));
		mt.Set (dim, col, 1);
		pt = t.GetNext();
	}

	// Apply the same permutations to mt:
	PermutateColumns (mt, perm);
	delete [] perm;

	// Resize both matrices to square matrices:
	ms.Resize (dim+1, dim+1);
	mt.Resize (dim+1, dim+1);

	// Find solution of m x ms = mt which is m = mt x inv(ms)
	if (! ms.Invert()) 
	{
		cout << "AffineFinder: Matrix not invertable.";
		return 0;
	}

	// Create the resulting transformation(matrix) and initialize
	// it with mt x ms:
	MatrixTrans *result = new MatrixTrans (dim+1, dim+1);
	mt.Multiply (ms, result);

	if (TestTransformation (s, t, result) == False) 
	{
		delete result;
		return 0;
	}
	return result; 
}


//----- TranslationFinder ----------------------------------------------------

Transformation *TranslationFinder::FindTrans
    (const PointSequence &s,
     const PointSequence &t) const
{
	uint lenS;
	if (TestPointSequences (s, t, dim, &lenS) == False)
		return 0;
 
	// Create resulting transformation:
	TranslationTrans *result = new TranslationTrans (dim);
	int rowcol;
	Point *ps = s.GetAt(0); 
	Point *pt = t.GetAt(0);
	for (rowcol=0; rowcol < dim; rowcol++) 
	{
		result->offset.Set (rowcol, pt->Get(rowcol) - ps->Get(rowcol));
	}

	if (TestTransformation (s, t, result) == False) 
	{
		delete result;
		return 0;
	}
	return result; 
}


//----- ScaleTranslationFinder ----------------------------------------------

Transformation *ScaleTranslationFinder::FindTrans
    (const PointSequence &s,
     const PointSequence &t) const
{
	uint lenS;
	if (TestPointSequences (s, t, dim, &lenS) == False)
		return 0;

	if (lenS < 2) 
	{
		cout << "ScaleTranslationFinder: Need at least 2 points." << endl;
		return 0;
	}
  	
  	// Create the resulting transformation(matrix): 
	MatrixTrans *result = new MatrixTrans (dim+1, dim+1);

	// Construct transformation matrix by subtraction and division
	// of coordinates of the first two points of target and source:
	int rowcol;
	Point *ps = s.GetAt(0); 
	Point *pt = t.GetAt(0);
	Point *ps2 = s.GetAt(1); 
	Point *pt2 = t.GetAt(1);
	double factor;
  
	if (ps->Get(0) - ps2->Get(0))
		factor = (pt->Get(0) - pt2->Get(0)) / ((ps->Get(0) - ps2->Get(0)));
	else 
	{
		cerr << "ScaleTranslationFinder: Division by zero." << endl;
		delete result;
		return 0;
	}
  	 
 	for (rowcol=0; rowcol < dim; rowcol++) 
	{
		result->Set (rowcol, rowcol, factor);
		result->Set (rowcol, dim, pt-> Get(rowcol) - (factor * ps->Get(rowcol)));
	}
	result->Set (dim, dim, 1);

	if (TestTransformation (s, t, result) == False) 
	{
		delete result;
		return 0;
	}
	return result;
}

