#ifndef __TRANS_H__
#define __TRANS_H__
//
// trans.h  --  Collage System Collage-Two
//
// Transformation classes
//
// Author            : Nils Schwabe
// Date of creation  : Feb 94
// Last modification : 20 Nov 95
//
// University of Bremen, Germany
//

#ifndef __MATRIX_H__
#include "matrix.h"
#endif


// Transformations and transformation finders
// ==========================================

// The term 'transformation' in the Collage System is formalized in two steps:
// A 'transformation' is a thing that applies a concrete transformation such 
// as "+3" or "multiply with matrix M" to a given point. 
// A 'transformation finder' is something that represents one class of 
// transformations such as "all affine transformationss" or "all translations"
// and is able to create a concrete transformation out of two point lists.



class Transformation  
//-----------------------------------------------------------------------------
// A virtual base class that declares the transformation function that all
// transformation classes must support.
{

// Each new transformation class that inherits directly
// from 'Transformation' and/or uses non-existing representations
// must add its own enum identifier here and implement an appropriate
// InternalType() method. This is needed to implement the Chain()-
// methods which rely on well-known internal representations.

public:
	typedef enum 
	{
		transMatrix,		// matrix multiplication used
		transTranslation	// translation transformation

	} TransInternal;

	virtual Transformation* Copy () const = 0; 

		// Virtual copy constructor.

	virtual ~Transformation()                                              {}

	virtual void Transform (const Point &p, Point *res) const = 0;

		// Applies the transformation to a point: res := t(p).
		// Note: 'p' and 'res' may be the same variables.

	virtual Transformation *Chain (const Transformation *outer) const
								      {return 0;} 

		// Returns a pointer to a new transformation that represents
		// outer(this) or NULL if no such representation could be 
		// generated.

	virtual TransInternal InternalType () const = 0;

		// Returns the internal implementation type of the
		// transformation.	
};


class TransFinder
//-----------------------------------------------------------------------------
// A virtual base class that declares a function that finds a transformation
// given two point lists such that the transformation maps the first point
// seuquence to the second.
{
public:
	virtual ~TransFinder() {}
	virtual TransFinder* Copy () const = 0; // virtual copy constructor
	
	virtual Transformation *FindTrans (const PointSequence &pSeqSource,
	                                   const PointSequence &pSeqTarget) 
		const = 0;
	// Answer a pointer to a newly created instance of an appropriate 
	// transformation or NULL if no such transformation could be found.

	virtual const char *GetName() const = 0;
	// Answer the (printable) name of the finder.
};
	

// Common transformation classes:
// ==============================

class MatrixTrans : public Matrix, public Transformation
//-----------------------------------------------------------------------------
// A transformation based on a matrix that transforms points (i.e. vectors)
// by multiplying the matrix with the vector.
{
public:
	MatrixTrans (ushort r, ushort c) : Matrix(r,c) {}
	MatrixTrans (const MatrixTrans &copy) : Matrix(copy) {}
	Transformation *Copy () const {return new MatrixTrans(*this);}
	void Transform (const Point &p, Point *res) const;
	// res := Matrix * p

	Transformation *Chain (const Transformation *outer) const;
	TransInternal InternalType () const {return transMatrix;}
};


class TranslationTrans : public Transformation
//-----------------------------------------------------------------------------
// A transformation that adds a constant vector to a point.
{
public:
	TranslationTrans (ushort dimensions) : offset(dimensions) {}
	TranslationTrans (const TranslationTrans &copy) : offset(copy.offset) {}
	Transformation *Copy () const {return new TranslationTrans (*this);}
	void Transform (const Point &p, Point *res) const;
	// res := p + offset
	
	Transformation *Chain (const Transformation *outer) const;
	TransInternal InternalType () const {return transTranslation;}
	
	Point offset;	
};




// Common transformation finders:
// ==============================

class AffineFinder : public TransFinder
//-----------------------------------------------------------------------------
// Supports affine transformations.
{
public:
	AffineFinder (ushort dimensions) {dim = dimensions;}
	~AffineFinder () {}
	Transformation *FindTrans (const PointSequence &pSeqSource,
	                           const PointSequence &pSeqTarget) const;
	TransFinder *Copy () const
		{TransFinder *n = new AffineFinder(dim); return n;}

	const char *GetName() const {return "Affine";}
private:
	ushort dim;
};


class TranslationFinder : public TransFinder
//-----------------------------------------------------------------------------
// Supports translations.
{
public:
	TranslationFinder (ushort dimensions) {dim = dimensions;}
	~TranslationFinder () {}
	Transformation *FindTrans (const PointSequence &pSeqSource,
	                           const PointSequence &pSeqTarget) const;
	TransFinder *Copy () const
		{TransFinder *n = new TranslationFinder(dim); return n;}

	const char *GetName() const {return "Translations";}
private:
	ushort dim;
};


class ScaleTranslationFinder : public TransFinder
//-----------------------------------------------------------------------------
// Supports translations and scalings.
{
public:
	ScaleTranslationFinder (ushort dimensions) {dim = dimensions;}
	~ScaleTranslationFinder () {}
	Transformation *FindTrans (const PointSequence &pSeqSource,
	                        const PointSequence &pSeqTarget) const;
	TransFinder *Copy () const
		{TransFinder *n = new ScaleTranslationFinder(dim); return n;}

	const char *GetName() const {return "Scale-translations";}
private:
	ushort dim;
};


#endif /* !defined(__TRANS_H__) */

