/*
 * @(#)Part.java
 * 
 */

package collage.generic;

import openstar.generic.*;
import openstar.util.*;
import collage.kernel.*;

/**
 * Generic "part" class.
 * <p>
 * For a conceptual overview, refer to the description of the base class,
 * <code>Edge</code>, in the kernel package, and to the class 
 * <code>PartFactory</code> in this package.
 *
 * @version 30 Apr 1997
 * @author 	Nils Schwabe
 */ 
public class Part extends Edge
{

/**
 * The number of dimensions of this part.
 * <p>
 * Currently only 2- and 3-dimensional parts are supported by this class.
 */
protected int dimensions;

/**
 * The "shape" of the generic part is given by a transformation.
 */
protected MatrixTrans trans;

/**
 * A generic part may contain arbitrary parameters, none of which
 * has any meaning to the part itself. If a generic part is used
 * directly, the parameters may be used to apply specific higher
 * level semantics to the part.
 */
protected ParamSet params;


public Part(ParamSet params, int dimensions) {

	this.dimensions = dimensions;
	this.params = params;
	initTransformation();
}

protected Part() {
}


protected void initTransformation() {

	if (dimensions == 2) {
		trans = new MatrixTrans(3,3);

		trans.set(0, 0, 1.0);
		trans.set(0, 1, 0.0);
		trans.set(0, 2, 0.0);

		trans.set(1, 0, 0.0); 
		trans.set(1, 1, 1.0);
		trans.set(1, 2, 0.0);

		trans.set(2, 0, 0.0); 
		trans.set(2, 1, 0.0); 
		trans.set(2, 2, 1.0);	
	}
	else if (dimensions == 3) {
		trans = new MatrixTrans(4,4);

		trans.set(0, 0, 1.0);
		trans.set(0, 1, 0.0);
		trans.set(0, 2, 0.0);
		trans.set(0, 3, 0.0);

		trans.set(1, 0, 0.0); 
		trans.set(1, 1, 1.0);
		trans.set(1, 2, 0.0);
		trans.set(1, 3, 0.0);

		trans.set(2, 0, 0.0); 
		trans.set(2, 1, 0.0); 
		trans.set(2, 2, 1.0);
		trans.set(2, 3, 0.0);

		trans.set(3, 0, 0.0); 
		trans.set(3, 1, 0.0); 
		trans.set(3, 2, 0.0);
		trans.set(3, 3, 1.0);
	} else {
		throw new IllegalArgumentException("Generic parts currently support either 2 or 3 dimensional parts.");
	}
}


public int getDimensions() {
	return dimensions;
}


public ParamSet getParams() {

	return params;
}

 
public void transform(Transformation t) {

	if (trans == null) {
		trans = (MatrixTrans)((MatrixTrans)t).clone();
	} else {
		trans = (MatrixTrans)t.chain(trans);
	}
}
 
 
/**
 * This implementation always returns <code>null</code>, indicating
 * that a generic part cannot be used as an hyperedge per se.
 * <p>
 * Of course, derived classes can implement replacement behaviour.
 */ 
public Transformation findTrans(PointSequence s,
                                TransFinder finder,
                                boolean inverse) {
                                
	return null;
}                        
 
 
public boolean equals (Object obj) {

	if (! (obj instanceof Part))
		return false;
		
	return false; // <M> test transformations against 3 or 4 linear indep. points
}		
 
 
protected void cloneInto(Part into) { 

	MatrixTrans newTrans;
	if (trans != null) {
		newTrans = (MatrixTrans)trans.clone();
	} else {
		newTrans = null;
	}
	ParamSet newParams = (ParamSet)params.clone();
	into.dimensions = dimensions;
	into.trans = newTrans;
	into.params = newParams;	
}
 
public Object clone() {

	Part newPart = new Part();
	cloneInto(newPart);
	return newPart;
}
 
} // class Part

