/*
 * @(#)Base.java
 * 
 */

package collage.kernel;

import java.util.Random;
import openstar.util.*;

/**
 * Derivation base class.
 * <p>
 * A derivation base is a list of <code>BaseElement</code> objects,
 * and describes the replacement of one or more hyperedges in a collage.
 * <p>
 * A <code>Base</code> object can be passed to a <code>Collage</code>
 * for execution.
 *
 * @version 14 Apr 1997
 * @author 	Nils Schwabe
 */ 
public class Base extends TypedList
{

private Random random;

public Base() {

	super("collage.kernel.BaseElement", true);
	random = new Random();
}


public boolean build(Collage collage, 
                  RuleSet rules, 
                  TableInfo tableInfo,
                  boolean kInverse) {

	Edge curEdge = (Edge)collage.getEdges().getAt(0);
	Rule[] curRule = new Rule[1];
	Transformation[] trans = new Transformation[1];
	boolean gotRule;
	int edgeListIndex = 0;
	
	if (curEdge == null) {
		return false;
	}
	
	while (curEdge != null) {
	
		curRule[0] = null;
		trans[0] = null;
		if (tableInfo.isExhaustProb()) {
			gotRule = getEdgeRuleExhaust(collage, rules, tableInfo, curEdge, kInverse, curRule, trans);
		} else {
			gotRule = getEdgeRuleNormal(collage, rules, tableInfo, curEdge, kInverse, curRule, trans);
		}
		
		if (gotRule) {
			Collage delContext = null;
			if (curRule[0].getContext() != null && curRule[0].isCDelete()) {
				delContext = curRule[0].getContext();
			}
			insert(curEdge, curRule[0].getRight(), trans[0], delContext);
			if (tableInfo.isSequential() && ! tableInfo.isOneFromBase()) {
				return true;
				// only one edge should be replaced, and we can take the
				// first one in the order				
			}
		}
		
		curEdge = (Edge)collage.getEdges().getAt(++edgeListIndex);	
	}
	
	int numElems = length();
	if (numElems > 0 && tableInfo.isOneFromBase()) {
		//
		// Select one of the base elements randomly and discard all others.
		//
		double normrand = getNormRandom();		
		int elemIndex = (int)(normrand * (double)numElems);
		BaseElement survivor = (BaseElement)getAt(elemIndex);
		removeAllElements();
		addElement(survivor); 
	}                  
	
	return true;
}


protected void insert(Edge edge, 
                 Collage collage, 
                 Transformation trans, 
                 Collage delContext) {

	addElement(new BaseElement(edge, collage, trans, delContext));
}


protected boolean getEdgeRuleNormal(Collage collage, 
									RuleSet rules,
									TableInfo tableInfo,
									Edge curEdge,
									boolean kInverse,
									Rule[] rule,
									Transformation[] trans) {
	double normrand, totalProb;
	Rule curRule;
	
	// Throw the dice only once per edge
	//
	normrand = getNormRandom();
	totalProb = 0.0;
	curRule = (Rule)rules.getAt(0);
	int ruleIdx = 0;
	trans[0] = null;
	
	while (curRule != null) {
		
		// Same equivalence class ?
		//
		if (curRule.getEqClass() != curEdge.getEqClass()) {
			curRule = (Rule)rules.getAt(++ruleIdx);
			continue;
		}
		
		// Correct table ?
		//
		if (   curRule.getTable() != 0 
		    && curRule.getTable() != tableInfo.getTable()) {
			curRule = (Rule)rules.getAt(++ruleIdx);
			continue;
		}
		    
		// Probability ok ?
		//
		totalProb += curRule.getProb();
		if (totalProb < normrand) {
			curRule = (Rule)rules.getAt(++ruleIdx);
			continue;
		}
				
		if (false) {
			// After this point, the edge will be replaced by the current rule or not at all,
			// i.e. no more attempts are made to find another rule if this rule fails after it
			// has been chosen by the random generator.

			// Transformation found ?
			//
			if ((trans[0] = curEdge.findTrans(curRule.getRight().getPins(),
		                                	  curRule.getTransFinder(), kInverse)) == null) {
				return false;
			}

			if (contextOk(collage, curRule, trans[0])) {
				rule[0] = curRule;
				return true;
			}
			return false;
		} else {
			// 
			// Check transformation and context and choose another rule if it doesn't match
			//
			
			// Transformation found ?
			//
			if ((trans[0] = curEdge.findTrans(curRule.getRight().getPins(),
		                                	  curRule.getTransFinder(), kInverse)) == null) {
				trans[0] = null;
				curRule = (Rule)rules.getAt(++ruleIdx);
				continue;
			}

			if (contextOk(collage, curRule, trans[0])) {
				rule[0] = curRule;
				return true;
			}
			curRule = (Rule)rules.getAt(++ruleIdx);
			continue;	
		}
	}
	return false;
}                             
                  
protected boolean getEdgeRuleExhaust(Collage collage, 
                            		 RuleSet rules,
                            		 TableInfo tableInfo,
                            		 Edge curEdge,
                            		 boolean kInverse,
                            		 Rule[] rule,
                            		 Transformation[] trans) {

	// <M>
	return false;
}                             


/**
 * Returns true if the rule satisfies all context conditions.
 */
protected boolean contextOk(Collage collage, 
                            Rule rule, 
                            Transformation trans) {
                           
	boolean pContextOk, nContextOk;
	
	if (rule.getContext() == null) {
		pContextOk = true;
	} else {
		Collage transContext = (Collage)rule.getContext().clone();
		transContext.transform(trans);
		pContextOk = (collage.findContext(transContext, rule.isCAnyOf()));
	}
	
	if (rule.getNContext() == null) {
		nContextOk = true;
	} else {
		Collage transContext = (Collage)rule.getNContext().clone();
		transContext.transform(trans);
		nContextOk = (! collage.findContext(transContext, rule.isNCAnyOf()));
	}
	
	return (pContextOk && nContextOk);	
}                            

/**
 * Returns a random value between 0.0 and 1.0.
 */
protected double getNormRandom() {

	return random.nextDouble();
}



} // class base

