/*
 * @(#)Collage.java
 * 
 */

package collage.kernel;

import openstar.util.*;

/**
 * Collage class.
 * <p>
 * A collage consists primarily of the <em>pin-points</em> and
 * a list of <em>edges</em>, which can be hyperedges or parts.
 * <p>
 * The primary operations on a collage are
 * <ul>
 * <li>transform the collage with a given transformation
 * <li>replace an edge in the collage (using a given transformation,
 * and probably also a context).
 * <li>execute a derivation step using a given derivation base object.
 * </ul>
 *
 * @version 14 Apr 1997
 * @author 	Nils Schwabe
 */ 
public class Collage implements Transformable
{

protected PointSequence pins;
protected EdgeSet edges;

public Collage(PointSequence pins,
               EdgeSet edges) {
               
	this.pins = pins;
	this.edges = edges;
}


public Object clone() {

	Collage newCol = new Collage(null, null);
	newCol.pins = (PointSequence)pins.clone();
	newCol.edges = (EdgeSet)edges.clone();
	return newCol;
}

public PointSequence getPins() {
	return pins;
}

public EdgeSet getEdges() {
	return edges;
}


public void transform(Transformation t) {

	pins.transform(t);
	edges.transform(t);
}


public void replEdge(Edge edge,
                     Collage repl,
                     Transformation t,
                     boolean kInverse,
                     Collage delContext) {

	Collage c = (Collage)repl.clone();
	
	// Normal or inverse replacement
	//
	if (kInverse) {
		transform(t);
	} else {
		c.transform(t);
	}
	
	// Delete context (type-0 grammar) ?
	//
	if (delContext != null) {
		Collage transDelContext = (Collage)delContext.clone();
		transDelContext.transform(t);
		removeSubCollage(transDelContext);	
	}
	
	// Replace the edge
	//
	edges.removeElement(edge);
	
	// Add 'c' to the collage
	//
	add(c);
	
                     
}


public void deriveWithBase(Base base,
                           boolean kInverse) {
                           
	int count = 0;
	BaseElement cur = (BaseElement)base.getAt(0);
	while (cur != null) {
		replEdge(cur.getTarget(), cur.getSource(), cur.getTrans(), kInverse, cur.getDelContext());
		cur = (BaseElement)base.getAt(++count);
	}
}                           


public boolean findContext(Collage context, boolean anyOf) {

	Edge contextEdge = (Edge)context.edges.getAt(0);
	boolean contains;
	int i = 0;
	
	if (contextEdge == null) {
		return true; // empty context always matches
	}
	while (contextEdge != null) {
		contains = (edges.findEqualEdge(contextEdge) != null);
		if ((! anyOf) && (! contains)) {
			return false;
		}
		if (anyOf && contains) {
			return true;
		}
		contextEdge = (Edge)context.edges.getAt(++i);
	}
	return (! anyOf);
}


protected void add(Collage collage) {

	edges.add(collage.edges);
}



protected void removeSubCollage(Collage subCollage) {

	// <M>
}
                     

} // class Collage


