/*
 * @(#)Edge.java
 * 
 */

package collage.kernel;

import openstar.util.*;

/**
 * Abstract edge class.
 * <p>
 * In a technical sense, an <code>Edge</code> is a labeled object in a given 
 * equivalence class that
 * can be transformed and that can evaluate a transformation given
 * an arbitrary point sequence.
 * <p>
 * In the Java version of the CollageSystem, the <code>Edge</code>
 * class is also the base class for all <em>Parts</em> of a collage,
 * i.e. there is no distinction between hyperedges and parts in the
 * kernel of the Java CollageSystem. More precisely, the kernel does
 * not know anything about parts at all. Because the CollageSystem
 * supports type-0 grammars, all produced objects (parts or hyperedges)
 * are a possible target for a replacement operation. Thus, there is
 * no need for the kernel to separate parts and hyperedges.
 *
 * @version 09 Apr 1997
 * @author 	Nils Schwabe
 */ 
abstract public class Edge implements Transformable
{

protected String label;
protected int eqClass;	
	
public Edge() {
	label = "";
	eqClass = 0;
}	
	
public Edge(String initLabel) {	

	label = initLabel;
	eqClass = 0;
}


/**
 * Transforms the edge with the given transformation.
 * <p>
 * The semantic of this operation depends on the implementation
 * and the needs of the derived class. A classical hyperedge or
 * part will apply the transformation to its own shape.
 */
abstract public void transform(Transformation t);

/**
 * Returns a transformation that maps the PointSequence to this
 * edge using the given TransFinder. This method is typically used to
 * determine the transformation that must be applied to a collage in
 * order to replace this <code>Edge</code> object by the collage.
 * <p>
 * <code>null</code> is returned if no such transformation can be found. 
 * If the
 * boolean parameter is <code>true</code> then it is tried to find
 * the inverse mapping instead.
 * <p>
 * Depending on the actual type, i.e. derived class, of this
 * <code>Edge</code> object, an overridden implementation of this
 * method may decide to ignore specific, or all, parameters and
 * return an arbitrary transformation, or <code>null</code>.
 */
abstract public Transformation findTrans(PointSequence s,
                                         TransFinder finder,
                                         boolean inverse);
                       
    
/**                                     
 * This is supposed to make a deep copy of the <code>Edge</code>. This
 * method <em>must</em> be overridden in all subclasses. 
 * <p>
 * This implementation of the method always returns <code>null</code>,
 * indicating that it is not possible to make a copy of an abstract 
 * object.
 */
public Object clone() {
	return null;
}                                         

/**
 * Sets the label string of the <code>Edge</code>
 * <p>
 * The label is the character or string that the user sees and uses in 
 * a grammar to refer to a specific set (or class, but not meant in the
 * usual programmatic sense) of objects. 
 *
 * Usually, the label uniquely identifies objects with equal replacement 
 * behaviour.
 */
public void setLabel(String label) {

	this.label = label;
}


public String getLabel() {

	return label;
}

/**
 * Sets the equivalence class of the <code>Edge</code>.
 * <p>
 * The CollageSystem automatically normalizes every collage grammar. As a
 * part of the normalization process, each <code>Edge</code> object is
 * assigned an equivalence class. The equivalence class is the <em>internal
 * label</em> of an edge. In the special case of <code>
 * Hyperedge</code> objects, two hyperedges with the same label may be
 * assigned different equivalence classes (under some rarely used
 * special conditions). For all other derived <code>Edge</code> sub-classes, 
 * two edges belong to the same equivalence class simply if they have
 * the same label.
 */ 
public void setEqClass(int eqClass) {
	
	this.eqClass = eqClass;
}


public int getEqClass() {

	return eqClass;
}



} // class Edge



