/*
 * @(#)Hyperedge.java
 * 
 */

package collage.kernel;

import openstar.util.*;

/**
 * Standard hyperedge class.
 * <p>
 * This <code>Edge</code> subclass uses a <code>PointSequence</code>
 * to represent the <em>attached points</em> of hyperedges as known from 
 * the collage grammar theory.
 *
 * @version 09 Apr 1997
 * @author 	Nils Schwabe
 */ 
public class Hyperedge extends Edge
{

/** The attached points of the hyperedge. */
protected PointSequence ap;

public Hyperedge(String initLabel) {
	
	super(initLabel);
	ap = new PointSequence();
}


public void transform(Transformation t) {

	Point cur = (Point)ap.getAt(0);
	int i = 0;
	while (cur != null) {
		t.transform(cur, cur);
		ap.setElementAt(cur, i);
		i++;
		cur = (Point)ap.getAt(i);
	}
}

/**
 * This implementation of the abstract <code>findTrans</code> method returns 
 * the result of finding a transformation that maps the
 * given <code>PointSequence</code> to the <em>attached points</em>
 * of this hyperedge, using the given <code>TransFinder</code>.
 */
public Transformation findTrans(PointSequence s,
                                TransFinder finder,
                                boolean inverse) {
 
	if (inverse) {
		return finder.findTrans(ap, s);
	} else {
		return finder.findTrans(s, ap);
	}
}                                


public Object clone() {

	Hyperedge newHyper = new Hyperedge(this.label);
	newHyper.eqClass = this.eqClass;
	newHyper.ap = (PointSequence)this.ap.clone();
	return newHyper;
}


public boolean equals (Object obj) {
	if (! (obj instanceof Hyperedge))
		return false;
	Hyperedge hyper = (Hyperedge)obj;
	return    hyper.eqClass == this.eqClass 
	       && hyper.ap.equals(this.ap);
}


public PointSequence getAttachedPoints() {
	return ap;
}


} // class Hyperedge


