/*
 * @(#)RuleSet.java
 * 
 */

package collage.kernel;

import openstar.util.*;

/**
 * A class representing a set of rules.
 * <p> 
 * Besides storing a number of rules in a list, this class also provides
 * the <em>normalization</em> algorithm that must be applied to the rules
 * before they can be used in derivations.
 *
 * @version 30 Apr 1997
 * @author 	Nils Schwabe
 */ 
public class RuleSet extends TCTList
{

public RuleSet() {
	
	super("collage.kernel.Rule", true);
}


/**
 * Makes a deep copy of the RuleSet. All rules of the set
 * are also cloned.
 */
public Object clone() {

	RuleSet newRS = new RuleSet();
	copyInto(newRS);
	return newRS;
}


/**
 * Normalizes all rules.
 * <p>
 * This function assigns an <em>equivalence class</em> to every
 * rule in the set. Two rules are in the same equivalence class,
 * if they have the same label, and compatible pin-points, i.e.
 * there exists a transformation (found by the <code>TransFinder</code>
 * of the rule in question) that maps the pin-points of 
 * (the right-hand-side of) the rule to the pin-points of (the right-hand-side of) 
 * the other rule. 
 * <p>
 * If the <code>boolean</code> parameter is <code>true</code>
 * then the found transformation is actually applied to (the right-hand-side of) 
 * the rule in question, making all pin-point sequences in one equivalence class
 * equal. (Note that as long as <em>annotated collage grammars</em> are not
 * supported in the Java version of the CollageSystem, it makes no difference
 * for the generated results whether the parameter is <code>true</code> or <code>false</code>.)
 */
public void normalize(boolean transRightSide) {

	Rule rule;
	int curClass = 0; // first valid equivalence class is 1
	
	//
	// Step 1: Iterate once through all rules and assign equivalence classes
	//
	rule = (Rule)getAt(0);
	int ruleIdx = 0;
	while (rule != null) {
		if (rule.getEqClass() != 0) {
			rule = (Rule)getAt(++ruleIdx);
			continue; // rule has already been classified
		}
		
		// This rule becomes the representant of a new equivalence class.
		curClass++;
		rule.setEqClass(curClass);
		if (rule.getName() != null) {
			System.out.println("Rule '"+rule.getName()+"' defines class "+curClass);
		}
		
		// Check all remaining rules if they belong to this eq. class
		//
		int remainIdx = ruleIdx+1;
		Rule remain = (Rule)getAt(remainIdx);
		while (remain != null) {
			if (   remain.getEqClass() == 0
			    && remain.getText().equals(rule.getText())) {
				//
				// If there's a mapping from the pin-points of 'remain'
				// to the pin-points of the current rule, then both rules
				// belong to the same equivalence class.
				// 
				PointSequence source = remain.getRight().getPins();
				PointSequence target = rule.getRight().getPins();
				Transformation trans;
				if ((trans = remain.findTrans(source, target)) != null) {
					remain.setEqClass(curClass);
					if (remain.getName() != null) {
						System.out.println("Rule '"+remain.getName()+"' gets class "+curClass);
					}		
					if (transRightSide) {
						remain.getRight().transform(trans);
					}
				} else {
					if (remain.getName() != null) {
						System.out.println("Rule '"+remain.getName()+"' not compatible with "+curClass);
					}				
				}
			}
			remain = (Rule)getAt(++remainIdx);
		} // while
		rule = (Rule)getAt(++ruleIdx);	
	} // while
	
	//
	// Step 2: Iterate through all rules and all hyperedges and
	//         assign the new equivalence classes to the hyperedges.
	//
	rule = (Rule)getAt(0);
	ruleIdx = 0;
	while (rule != null) {
		classifyEdges(rule.getRight());
		if (rule.getContext() != null) {
			classifyEdges(rule.getContext());
		}
		if (rule.getNContext() != null) {
			classifyEdges(rule.getNContext());
		}
		rule = (Rule)getAt(++ruleIdx);
	} // while
}

/**
 * Calls <code>classifyEdge()</code> for all edges of the collage.
 */
public void classifyEdges(Collage collage) {

	EdgeSet edges = collage.getEdges();
	Edge edge = (Edge)edges.getAt(0);
	int edgeIdx = 0;
	while (edge != null) {
		classifyEdge(edge);
		edge = (Edge)edges.getAt(++edgeIdx);
	}
}

/**
 * Assigns an equivalence class to the given edge.
 * <p>
 * This method may only be called if the rules have been normalized.
 */
protected void classifyEdge(Edge edge) {

	Rule rule;
	
	// Iterate through the rules to find an applicable equivalence
	// class, which, by definition, is unique.
	//
	rule = (Rule)getAt(0);
	int ruleIdx = 0;
	while (rule != null) {
		PointSequence source = rule.getRight().getPins();
		if (   edge.getLabel().equals(rule.getText())
		    && edge.findTrans(source, rule.getTransFinder(), false) != null) {
			edge.setEqClass(rule.getEqClass());
			break;
		}    
		rule = (Rule)getAt(++ruleIdx);
	}
}

/**
 * <em>Not yet implemented in the Java version.</em>
 */
public void annotateEdges(Collage collage) {

	// <M> not yet implemented in the Java version
}



} // class RuleSet


