/*
 * @(#)TableElement.java
 * 
 */

package collage.kernel;

import openstar.util.*;

/**
 * Derivation table element class.
 * <p>
 * A <code>TableElement</code> contains a list of <code>TableElemEntry</code>
 * objects and also a state that indicates the current position in a
 * derivation.
 *
 * @version 14 Apr 1997
 * @author 	Nils Schwabe
 */ 
public class TableElement
{

/** Until which derivation step this TableElement is defined. */
protected int until;

/** The list of <code>TableElemEntry</code> objects. */
protected TypedList entries;

/** The current index into the list of <code>TableElemEntry</code> objects. */
protected int actEntry;

/** The current counter that counts the applications of the current table. */
protected int actTimes;

public TableElement(int until) {

	this.until = until;
	this.entries = new TypedList("collage.kernel.TableElemEntry", false);
	this.actEntry = 0;
	this.actTimes = 0;
}


public void addElement(TableElemEntry entry) {

	entries.addElement(entry);
}

/** 
 * Returns the number of the next table to use. 
 * <p>
 * This changes the internal state of the object.
 */
int nextTable() {

	TableElemEntry entry = (TableElemEntry)entries.getAt(actEntry);
	
	if (entry == null) {
		//
		// wrapped around. Start again at beginning
		//
		resetCounters();
		entry = (TableElemEntry)entries.getAt(actEntry);
	}
	
	if (entry == null) {
		System.err.println("CollageSystem: internal error in derivation table.");
		return 0;
	}
	
	actTimes++;
	if (entry.times >= actTimes) {
		//
		// one more application of the current table
		//
		return entry.table;
	} else {
		//
		// next table (or wrap around)
		//
		actEntry++;
		actTimes = 0;
		return nextTable();
	}
}

/**
 * Resets the internal state.
 */
void resetCounters() {

	actEntry = 0;
	actTimes = 0;
}



} // class TableElement


