/*
 * @(#)TransUtil.java
 * 
 */

package collage.kernel;

import openstar.util.*;

/**
 * Transformation utility functions.
 *
 * @version 09 Apr 1997
 * @author 	Nils Schwabe
 */ 
public class TransUtil
{


/**
 * Finds a permutation of the columns in ms such that the 
 * linear independent columns are placed at the beginning.
 *
 * Assumptions:<br> ms is of type (m x m+n), m>0, n>=0.
 * perm[] must have at least m+n elements.
 * @return <code>false</code> - There is no linear independent permutation<br>
 * <code>true</code> - If there is such a linear indepdent permutation, then this is one.
 */
public static boolean getLIColumnPermutation(Matrix m, int[] perm) {

	// Implementation note:
	// should be done in a precomputing phase to optimize performance.

	Matrix ms = (Matrix)m.clone();
	int lenS = ms.getColumns();
	int k;

	for (k = 0; k < lenS; k++) {
		perm[k] = k;
	}

	for (k = 0; k < ms.getRows(); k++) {
		int i = 0;
		while (i < lenS && ms.get(k, i) == 0) {
			i++;
		}
			
		if (i == lenS) {
			return false; // not invertable
		}

		// swap columns k and i:
		if (k != i) {
			for (int r = 0; r < ms.getRows(); r++) {
				double cur = ms.get(r, k);
				ms.set(r, k, ms.get(r, i));
				ms.set(r, i, cur);
			}
			int cur = perm[k];
			perm[k] = perm[i];
			perm[i] = cur;
		} 

		for (int l = k+1; l < ms.getRows(); l++) {
			double f = ms.get(l, k) / ms.get(k,k);
			for (int c = 0; c < lenS; c++) {
				ms.set(l, c, ms.get(l, c) - f * ms.get(k, c));
			}
		}
	}
	return true;
}


/**
 * Matrix column permutation.
 *
 * Permutates the columns in a given matrix according to a given
 * permutation vector.
 */
public static void permutateColumns(Matrix m, int perm[]) {

	for (int col = 0; col < m.getColumns(); col++) {
    	// swap columns col and perm[col]:
		if (col < perm[col]) {
			for (int r = 0; r < m.getRows(); r++) {
				double cur = m.get(r, col);
				m.set(r, col, m.get(r, perm[col]));
				m.set(r, perm[col], cur);
			}
		} 
	}
}


/** 
 * Compatibilty test.
 *
 * Applies a general compatibility test to both lists that is needed by nearly
 * all transformation finders. Answers <code>true</code> if successful.<br>
 * Note: In the Java version, only the length of both lists is compared.
 * @param s source sequence
 * @param d destination sequence
 * @param length length of (both) lists (only index 0 is used to store the result)
 */
public static boolean testPointSequences(PointSequence s,
                                         PointSequence t,
                                         int[] length) {
                      
	// Note: all dimension checks are not implemented in the Java version 
                                                               
	// Compare lengths: 
	int lenS = s.size();
	int lenT = t.size();
	if (lenS != lenT) {	
		return false;
	}
	length[0] = lenS;
	return true;
}


/**
 * Transformation test.
 *
 * Checks if <code>trans</code> maps all points of <code>s</code> to the 
 * corresponding points of <code>t</code>.
 */
public static boolean testTransformation(PointSequence s,
                                         PointSequence t,
                                         Transformation trans) {

	int dim = ((Point)s.getAt(0)).getDim();

	Point ps = (Point)s.getAt(0);	
	Point pt = (Point)t.getAt(0);
	Point res;
	int i = 0;
	while (ps != null && pt != null) {
		res = new Point(ps.getDim());
		trans.transform (ps, res);
		if (! res.equals(pt)) {
			System.err.println("TransFinder: Transformation test failed:");
			System.err.println("source = " + ps);
			System.err.println("target = " + pt);
			System.err.println("result = " + res);
			return false;
		}
		i++;
		ps = (Point)s.getAt(i);
		pt = (Point)t.getAt(i);
	}
	return true;
}


/**
 * Calculates a 2D rotation about a given angle.
 * <p>
 * The generated transformation is a 3x3 <code>MatrixTrans</code>.
 */
public static Transformation rotation2d(double rotAngle) {

	MatrixTrans trans = new MatrixTrans(3,3);
	
	trans.set(0, 0, Math.cos(rotAngle));
	trans.set(0, 1, -Math.sin(rotAngle));
	trans.set(0, 2, 0.0);
			
	trans.set(1, 0, Math.sin(rotAngle)); 
	trans.set(1, 1, Math.cos(rotAngle));
	trans.set(1, 2, 0.0);
			
	trans.set(2, 0, 0.0); 
	trans.set(2, 1, 0.0); 
	trans.set(2, 2, 1.0);	
	
	return trans;
}


/**
 * Calculates a 3D rotation about a given rotation axis and a given
 * angle.
 * <p>
 * The generated transformation is a 4x4 <code>MatrixTrans</code>.
 */
public static Transformation rotation3d(Point rotDir, double rotAngle) {

	double d = Math.sqrt(  rotDir.get(0) * rotDir.get(0)
	                     + rotDir.get(1) * rotDir.get(1)
	                     + rotDir.get(2) * rotDir.get(2));
	                     
	double n0 = rotDir.get(0) / d;
	double n1 = rotDir.get(1) / d;
	double n2 = rotDir.get(2) / d;
	
	double s = Math.sin(rotAngle);
	double c = Math.cos(rotAngle);
	double t = 1-c;
	
	MatrixTrans m = new MatrixTrans(4,4);

	m.set(0, 0, t * n0 * n0 + c);
	m.set(0, 1, t * n0 * n1 - s * n2);
	m.set(0, 2, t * n0 * n2 + s * n1);
	m.set(0, 3, 0.0);
	
	m.set(1, 0, t * n0 * n1 + s * n2);
	m.set(1, 1, t * n1 * n1 + c);
	m.set(1, 2, t * n1 * n2 - s * n0);
	m.set(1, 3, 0.0);
	
	m.set(2, 0, t * n0 * n2 - s * n1);
	m.set(2, 1, t * n1 * n2 + s * n0);
	m.set(2, 2, t * n2 * n2 + c);
	m.set(2, 3, 0.0);
	
	m.set(3, 0, 0.0);
	m.set(3, 1, 0.0);
	m.set(3, 2, 0.0);
	m.set(3, 3, 1.0);
	
	return m;
}




} // class TransUtil

