/*
 * @(#)CollageApplet.java
 * 
 */

package collage.ui;

import java.util.*;
import java.io.*;
import java.awt.*;
import java.awt.event.*;
import java.applet.Applet;

import openstar.generic.*;
import collage.generic.*;
import collage.parser.*;


/**
 * CollageSystem applet.
 * <p>
 * This class implements a GUI interface and controller for
 * all components of the CollageSystem.
 *
 * @version 09 Jun 1997
 * @author 	Nils Schwabe
 */  
public class CollageApplet extends Applet
{

//
// GUI
//

protected Panel mainPanel;
protected Previewer preview;
protected TextArea textArea;
protected String cgrDirectory = "./cgr/";
protected String fileName;
protected boolean showingEditor = true;
protected Button switchButton;

/**
 * The environment object. The environment is created by this object.
 */
protected Envi envi = new Envi();

/**
 * The derivation result object (created by this object).
 */
protected DerivationResult derivationResult = new DerivationResult(envi);

/**
 * The derivation result selection object (created by this object).
 */
protected DerivationResultSelection selection = new DerivationResultSelection(derivationResult); 

public void init() {

	setLayout(new BorderLayout());

	//
	// mainPanel
	//
	mainPanel = new Panel();
	mainPanel.setLayout(new CardLayout());
	add("Center", mainPanel);

	//
	// textArea
	//
	textArea = new TextArea();
	mainPanel.add("editor", textArea);	
	
	//
	// preview
	//
	preview = new Previewer(selection);
	mainPanel.add("previewer", preview);

	//
	// Button bar
	//		
	Panel p = new Panel();
	add("South", p);
	p.add(new LoadButton("Load..."));
	p.add(new SaveButton("Save"));
	p.add(new SaveAsButton("Save as..."));
	p.add(new CompileButton("Compile"));
	p.add(switchButton = new SwitchViewButton("->View"));

	setSize (700,500);
}

public void start() 
{
	showTitle();	
}

public void stop() 
{
} 

/**
 * Returns the nearest <code>Frame</code> instance that is a
 * parent of this applet.
 */
protected Frame getNextFrame()
{
    Container parent = getParent();
    while (parent != null) {
    	if (parent instanceof Frame)
    		return (Frame)parent;
    	parent = parent.getParent();
    }
    return null;
}

/**
 * Shows the window title, containing the name of the current collage grammar file.
 */
protected void showTitle() {

    Frame frame = getNextFrame();
    if (frame != null) {
		if (fileName != null) {
			frame.setTitle("CollageSystem - [" + fileName + "]");
		} else {
			frame.setTitle("CollageSystem - untitled");
		}
	}
}

/**
 * Displays a file selector and loads a new grammar into the editor window.
 */
protected void loadGrammar()
{
    Frame frame = getNextFrame();
    if (frame != null) {
    	FileDialog fdial = new FileDialog(frame, "Grammatik laden", FileDialog.LOAD);
    	fdial.setDirectory (cgrDirectory);
    	fdial.setFilenameFilter(new CgrFilenameFilter());
    	fdial.show();
    	fileName = fdial.getFile ();
    	if (fileName != null) {
    		cgrDirectory = fdial.getDirectory();
    		File file = new File(cgrDirectory, fileName);

    		//
    		// Check if file is ok.
    		//
    		if (! file.exists())
    		{
    			preview.showMessage ("File '" + cgrDirectory + fileName + "' doesn't exist.");
    			return;
    		}
    		if (! file.canRead())
    		{
    			preview.showMessage ("No read access to '" + cgrDirectory + fileName + "'.");
    			return;    				
    		}

    		//
    		// Open and read input stream.
    		//
    		FileInputStream istream;
    		try {
    			istream = new FileInputStream(file);
    		} catch (FileNotFoundException e) {
    			preview.showMessage ("Cannot open '" + cgrDirectory + fileName + "'.");
    			return;
    		} catch (IOException e) {
    			preview.showMessage ("I/O exception while opening '" + cgrDirectory + fileName + "'.");
    			return;    			
    		}

    		int length = 0;
    		try {length = istream.available();} catch (IOException e) {}
    		if (length <= 0)
    		{
    			try {istream.close();} catch (IOException e) {}
    			preview.showMessage ("File '" + cgrDirectory + fileName + "' has zero length.");
    			return;     			
    		}
    		byte[] buffer = new byte[length];
    		try {istream.read (buffer, 0, length);} catch (IOException e) {}
    		textArea.setText (new String (buffer));
 			switchToEditor();
			showTitle();
    	}
    	else
    		preview.showMessage ("No file chosen");    	
    }
    else
    	preview.showMessage ("Sorry - can't find frame window");
}


/**
 * Saves the currently edited grammar into a file.
 * @param saveAs Display a file selector, save under current name otherwise.
 */
protected void saveGrammar(boolean saveAs) {

	if (saveAs || fileName == null) {
		Frame frame = getNextFrame();
		if (frame != null) {
			FileDialog fdial = new FileDialog(frame, "Grammatik sichern", FileDialog.SAVE);
			fdial.setDirectory (cgrDirectory);
			fdial.setFilenameFilter(new CgrFilenameFilter());
			fdial.show();
			fileName = fdial.getFile ();
			if (fileName != null) {
				cgrDirectory = fdial.getDirectory();
			}
		} else {
    		preview.showMessage ("Sorry - can't find frame window");
    		return;
    	}
    }
    
	File file = new File(cgrDirectory, fileName);
	
    //
   	// Open and write output stream and print writer.
    //
    FileOutputStream ostream;
    try {
   		ostream = new FileOutputStream(file);
	} catch (IOException e) {
    	preview.showMessage ("I/O exception while creating '" + cgrDirectory + fileName + "'.");
    	return;    			
	}
	PrintWriter writer = new PrintWriter(ostream);
	writer.print(textArea.getText());
	writer.close();
    try {	
		ostream.close();
	} catch (IOException e) {
    	preview.showMessage ("I/O exception while closing '" + cgrDirectory + fileName + "'.");
    	return;    			
	}		
	preview.showMessage("File '" + fileName + "' has been saved.");	
	showTitle();
}


/**
 * Parses the currently edited text. If successfully, the environment
 * object is changed, old displays are deregistered and new ones are 
 * registered, and the environment change is announced to registered
 * listeners (which causes all views and displays to update their state).
 */
protected void compileGrammar() {

	// Get the text from the textArea and make a stream from it:
	String text = textArea.getText();
	StringBufferInputStream in = new StringBufferInputStream(text);
	
	// Create a new parser and let it use the stream and our environment:
	CSParser parser = new CSParser(in); 
	parser.setEnvi(envi);
	
	//
	// Reset the environment
	//	
	// Iterate through the display list of the environment
	// and de-register all previously registered displays from the
	// derivationResult and/or the selection:
	Display display;
	for (int i=0; (display = (Display)envi.getDisplays().getAt(i)) != null; i++) {
		if (display instanceof ModelChangeListener) {
			derivationResult.removeModelChangeListener((ModelChangeListener)display);
		}
		if (display instanceof ModelSelectionListener) {
			selection.removeModelSelectionListener((ModelSelectionListener)display);
		}
	}
	// Now reset the environment (no notifications sent):
	envi.reset(false);
	envi.setErrorMsg(null);
	
	try {
		parser.ParseCollageGrammar();
		
		// Iterate through the display list of the environment
		// and register all eligble displays for the derivationResult
		// and/or the selection:
		for (int i=0; (display = (Display)envi.getDisplays().getAt(i)) != null; i++) {
			if (display instanceof ModelChangeListener) {
				derivationResult.addModelChangeListener((ModelChangeListener)display);
			}
			if (display instanceof ModelSelectionListener) {
				selection.addModelSelectionListener((ModelSelectionListener)display);
			}
		}		
		envi.setModelChanged(this);  // -> triggers listeners
		//
		// Implementation note: The above two lines are a good
		// example of the MVC concept within OpenStar: First, the
		// model is changed in some way. Secondly, the modification
		// is announced to the interested public. Nothing else is
		// neccessary to start a derivation, update all displays, etc. 
		//
	} catch (ParseError e) {
		if (envi.getErrorMsg() != null) {
			preview.showMessage("ERROR: " + envi.getErrorMsg());
		} else {
			preview.showMessage("ERROR: see terminal window for details.");
		}
	}	
	switchToPreviewer();
}

/**
 * Switches to the editor window.
 */
protected void switchToEditor() {

	((CardLayout)mainPanel.getLayout()).show(mainPanel, "editor");
	textArea.requestFocus();				
	switchButton.setLabel("->View");
	showingEditor = true;
}


/**
 * Switches to the preview (2D view) window.
 */
protected void switchToPreviewer() {

	((CardLayout)mainPanel.getLayout()).show(mainPanel, "previewer");
	switchButton.setLabel("->Edit");
	showingEditor = false;
}


//
// Button handler classes
//

class LoadButton extends Button implements ActionListener {
   	public LoadButton(String title) {
   		super(title); addActionListener(this);
	} 
	public void actionPerformed(ActionEvent event) {
		loadGrammar();
	}
}

class SaveButton extends Button implements ActionListener {
   	public SaveButton(String title) {
   		super(title); addActionListener(this);
	} 
	public void actionPerformed(ActionEvent event) {
		saveGrammar(false);
	}
}

class SaveAsButton extends Button implements ActionListener {
   	public SaveAsButton(String title) {
   		super(title); addActionListener(this);
	} 
	public void actionPerformed(ActionEvent event) {
		saveGrammar(true);
	}
}

class CompileButton extends Button implements ActionListener {
   	public CompileButton(String title) {
   		super(title); addActionListener(this);
	} 
	public void actionPerformed(ActionEvent event) {
		compileGrammar();
	}
}


class SwitchViewButton extends Button implements ActionListener {
   	public SwitchViewButton(String title) {
   		super(title); addActionListener(this);
	} 
	public void actionPerformed(ActionEvent event) {
		if (showingEditor) {
			switchToPreviewer();
		} else {
			switchToEditor();
		}
	}
}   


    
} // class CollageApplet

