/*
 * @(#)Previewer.java
 * 
 */

package collage.ui;

import java.util.*;
import java.io.*;
import java.awt.*;
import java.awt.event.*;
import java.applet.Applet;

import openstar.generic.*;
import collage.parser.*;
import collage.kernel.*;
import collage.generic.*;

/**
 * 2D collage display class.
 *
 * This class implements a collage <code>Display</code> that
 * outputs the collages (in 2D only) to the screen using
 * AWT graphics functions.
 *
 * @version 11 Jun 1997
 * @author 	Nils Schwabe
 */ 
public class Previewer extends Panel 
                       implements Display {

/**
 * The result of the current derivation (back-reference).
 */
protected DerivationResult derivationResult;

/**
 * The current selection (back-reference).
 */
protected DerivationResultSelection selection; 

/**
 * The canvas that displays the current collage.
 */
protected PreviewDisplay canvas; 

/**
 * The list that displays the currently available Display objects.
 */
protected ListOfDisplays list; 


//
// We delegate the view-related work to an inner class
//

class View implements ModelChangeListener, ModelSelectionListener {  
 
/** 
 * The currently displayed collage.
 */
private Collage showCollage;


/**
 * Registers this view object to the derivation result and to the
 * derivation result selection of the enclosing object.
 */
public View() {
	derivationResult.addModelChangeListener(this);
	selection.addModelSelectionListener(this);
}


/**
 * Called when the derivation result has changed.
 * <p>
 * Redisplays the currently selected collage.
 */
public void modelChange(ModelChangeEvent event) {

	// We know the model in this event is actually the DerivationResult object
	// that we already know.
		
	showMessage("Number of available derivation steps: "
	            + derivationResult.getCollageList().size());
	redisplay();
}


/**
 * Called when the derivation result selection has changed.
 * <p>
 * Redisplays the currently selected collage.
 */
public void modelSelection(ModelSelectionEvent event) {

	// We know the model in this event is actually the 
	// DerivationResultSelection object that we already know.
		
	redisplay();
}


private void redisplay() {
	String msg = "No generated collages";
	CollageList list = derivationResult.getCollageList();
	showCollage = null;
	if (selection.getIndex() < list.size()) {
		showCollage = (Collage)list.getAt(selection.getIndex());
		if (showCollage != null) {
			if (selection.getIndex() == 0) {
				msg = "Start";
			} else {
				msg = selection.getIndex() + ". Step";
			}
			msg += ": " + showCollage.getEdges().size();
			if (showCollage.getEdges().size() == 1) {
				msg += " object.";
			} else {
				msg += " objects.";
			}
		}			     		
	}
	
	if (showCollage != null) {	
		canvas.displayCollage(derivationResult.getEnvi(), showCollage, 
		                      selection.getIndex(), list.size());
	}
	showMessage(msg);
}

} // inner class View



public Previewer(DerivationResultSelection selection) {

	// Store a backlink to the selection and a shorthand for the derivation result:
	this.selection = selection;
	this.derivationResult = selection.getResult();
	
	// Create the View object:
	new View();
	
	setLayout(new BorderLayout());

	//
	// Canvas
	//
	canvas = new PreviewDisplay();
	add("Center", canvas);

	//
	// Button bar
	//		
	Panel p = new Panel();
	add("South", p);
	p.add(new FirstButton("First"));
	p.add(new LastButton("Last"));
	p.add(new PrevButton("Prev"));
	p.add(new NextButton("Next"));
	p.add(new ZoomInButton("Zoom in"));
	p.add(new ZoomOutButton("Zoom out"));
	p.add(new ResetViewButton("Reset"));
	
	//
	// Display list
	//
	list = new ListOfDisplays(derivationResult.getEnvi());
	add("East", list);
	
	//
	// Register for event notifications.
	// Whenever the environment changes (grammar, parameters, etc.)
	// we want to be informed.
	// (Note that we are not interested in changes to the
	// derivation result, because we assume that direct changes to
	// the list of generated collages do not happen.)
	//
	derivationResult.getEnvi().addModelChangeListener(this);	
}


/**
 * Called when the environment (<code>Envi</code> object) has changed.
 * <p>
 * In response to any change in the environment model (grammar, parameters,
 * whatever), a new derivation is carried out by this method, and the derivation
 * result object, which is also a model, is changed. This change
 * triggers all active views that display the result of a derivation,
 * including the internal View object of this object.
 */
public void modelChange(ModelChangeEvent event) {

	// We know the model in this event is actually an Envi object:
	Envi envi = (Envi)event.getModel();
	
	// Create a new collage list and perform a derivation:
	CollageList list = new CollageList();	
	new StdDerivator().doDerivation(envi, list);
	
	// Reset the selection:
	selection.setIndex(DerivationResultSelection.NONE_SELECTED); // -> triggers...
	
	// Replace the old list with new list:
	derivationResult.setCollageList(list); // -> triggers views	
	
	// Reset the selection:
	selection.setIndex(0); // -> triggers...	
}


public void showMessage(String msg) {
	canvas.showMessage(msg);
}



//
// Button handler classes (all inner classes)
//

class FirstButton extends Button implements ActionListener {
   	public FirstButton(String title) {
   		super(title); addActionListener(this);
	} 
	public void actionPerformed(ActionEvent event) {
		selection.setIndex(0);
	}
}

class LastButton extends Button implements ActionListener {
   	public LastButton(String title) {
   		super(title); addActionListener(this);
	} 
	public void actionPerformed(ActionEvent event) {
		CollageList list = derivationResult.getCollageList();
		if (list.size()-1 >= 0) {
			selection.setIndex(list.size() - 1);
		} else {
			selection.setIndex(0);
		}
	}
}


class PrevButton extends Button implements ActionListener {
   	public PrevButton(String title) {
   		super(title); addActionListener(this);
	} 
	public void actionPerformed(ActionEvent event) {
		if (selection.getIndex() > 0) {
			selection.setIndex(selection.getIndex()-1);
		}
	}
}


class NextButton extends Button implements ActionListener {
   	public NextButton(String title) {
   		super(title); addActionListener(this);
	} 
	public void actionPerformed(ActionEvent event) {
		CollageList list = derivationResult.getCollageList();
		if (selection.getIndex() < list.size()-1) {
			selection.setIndex(selection.getIndex()+1);
		}		
	}
}


class ZoomInButton extends Button implements ActionListener {
   	public ZoomInButton(String title) {
   		super(title); addActionListener(this);
	} 
	public void actionPerformed(ActionEvent event) {
		canvas.zoomIn();
	}
}

class ZoomOutButton extends Button implements ActionListener {
   	public ZoomOutButton(String title) {
   		super(title); addActionListener(this);
	} 
	public void actionPerformed(ActionEvent event) {
		canvas.zoomOut();
	}
}

class ResetViewButton extends Button implements ActionListener {
   	public ResetViewButton(String title) {
   		super(title); addActionListener(this);
	} 
	public void actionPerformed(ActionEvent event) {
		canvas.resetView();
	}
}

    
}
