/*
 * @(#)VRMLDisplay.java
 * 
 */

package collage.vrml;

import java.io.*;

import openstar.generic.*;
import collage.kernel.*;
import collage.generic.*;

/**
 * CollageSystem VRML display.
 * <p>
 * This display supports all part classes (derived from
 * <code>collage.kernel.Edge</code>) that implement the
 * <code>IVRMLPart</code> interface. One such class is
 * <code>VRMLPart</code> (which is generated, for example,
 * by the part factory <code>VRFactory</code>).
 *
 * @version 18 Jun 1997
 * @author 	Nils Schwabe
 */ 
public class VRMLDisplay implements Display, 
                                    ModelSelectionListener
{

private String name;
private String filename;
private FileOutputStream out; // not null <=> a file is currently open
private PrintWriter writer;
private String viewer; // the VRML viewer program

public VRMLDisplay(String viewer) {
	this.viewer = viewer;
}

public VRMLDisplay() {
	this.viewer = "vrweb -bg3d black -remote";
}


public void setName(String name) {
	this.name = name;
}


public String getName() {
	return name;
}


public void setViewer(String viewer) {
	this.viewer = viewer;
}


public String getFilename() {
	return this.filename;
}


protected String getRealFilename(int index) {

	if (filename == null) {
		return "Collage.wrl";
	}
	if (index >= 0) {
		return filename + "_" + index + ".wrl";
	} else {
		return filename + ".wrl";
	}
}

protected void open(int index) throws IOException {
	if (out != null) {
		// indicates there's already a file open - close it first
		writer.close();
		out.close();
		out = null;
	}	
	out = new FileOutputStream(getRealFilename(index));
	writer = new PrintWriter(out);
}


protected void close() throws IOException {
	if (out == null) {
		// indicates nothing is open
		return;
	}
	writer.close();
	out.close();
}
	
	
protected int outputCollage(Envi envi,
                            Collage collage) {
                        
	EdgeSet edges = collage.getEdges();
	Edge edge;
	int output = 0;
	for (int i=0; (edge = (Edge)edges.getAt(i)) != null; i++) {
		if (edge instanceof IVRMLPart) {
			((IVRMLPart)edge).vrmlOutput(writer);
			output++;
		}
	}
	return output;
}


protected void output(Envi envi,
                      Collage collage,
                      int index) {
    try {                       
		open(index);
		writer.println("#VRML V1.0 ascii");
		writer.println("#Collage generated by OpenCollage display '"+getName()+"'");
		writer.println("#for index "+index+"\n\n");	
		if (outputCollage(envi, collage) == 0) {		
			writer.println("Separator {");
			writer.println("    Material {ambientColor 1 .6 .6  diffuseColor 1 .6 .6}");
			writer.println("    AsciiText {string \"Collage #" + index + " (no VRML parts)\" spacing .5 justification CENTER}");
			writer.println("}");
		}
		close();
		// <M> get viewer name from environment
		Process p = Runtime.getRuntime().exec(viewer+" "+getRealFilename(index));
	} catch (IOException e) {
		e.printStackTrace();
	}
}


public void modelChange(ModelChangeEvent event) {
}


public void modelSelection(ModelSelectionEvent event) {

	DerivationResultSelection selection = (DerivationResultSelection)event.getModelSelection();
	int index = selection.getIndex();
	if (index >= 0) {
		Collage collage = (Collage)selection.getResult().getCollageList().getAt(index);
		output(selection.getResult().getEnvi(), collage, index);
	}
}


} // class VRMLDisplay

