/*
 * @(#)Factory.java
 * 
 */

package openstar.generic;

/**
 * Generic factory base class.
 *
 * This class provides a common framework for the creation of user-defined
 * objects. It is intended to be used by tools that are also "generic",
 * in the sense that they provide services or functions to allow users to
 * create objects by specifying a class name and giving a set of parameters
 * in a predefined format. One example of such a tool could be a parser for
 * a specific picture generation method. To support such tools in their 
 * generic behaviour,
 * it is recommended that the names of concrete factory classes are composed
 * of the class name of the objects they create, followed by the string 
 * "Factory", for example "FooFactory".
 * <p>
 * <code>Factory</code> is a <em>meta-meta factory class</em>, and can also
 * be called the "mother of all factory base classes". <em>meta factory 
 * classes</em> are derived from <code>Factory</code> and may
 * add additional features that are specific to the family of objects being
 * created. Finally, concrete <em>Factory classes</em> are the ones that
 * create specific objects within a given family.
 * 
 * @version 04 Jun 1997
 * @author 	Nils Schwabe
 * 
 */
public abstract class Factory extends ParamSet
{

/** Everything's just fine. */
static public final int OK = 0;

/** The parameter name is not recocgized by this factory. */
static public final int ERR_UNKNOWN_PARAMETER = -1;

/** The parameter value is not an instance of a class that
 * is expected for this parameter. */
static public final int ERR_WRONG_TYPE = -2;

/** The specified factory class was not found. */
static public final int ERR_CLASS_NOT_FOUND = -3;

/** An internal error occurred while instantiating a factory. */
static public final int ERR_INSTANCE_ERROR = -4;

/** The specified class is not derived from <code>Factory</code>. */
static public final int ERR_NOT_A_FACTORY = -5;

/** A parameter with the same name already existed and it has 
 * been replaced by the new parameter. */
static public final int WARN_PARAM_EXISTED = 1;


/** 
 * Creates and returns an instance of a factory.
 * <p>
 * The factory class is loaded dynamically (if not already loaded
 * by the system) by using the method <code>Class.forName()</code>.
 * The name of the class is composed of the given string, and the
 * string "Factory". Thus, classes whose names do not end with 
 * "Factory" cannot be loaded with this method.
 * @param name Name of the factory class (not including the trailing
 * "Factory")
 * @param error If the method returns <code>null</code> then the first
 * element of this array contains one of the defined error codes.
 * @return New factory instance or <code>null</code> if an error occured.
 */
static public Factory getFactory(String name, int[] error) {

	try {
		Class factoryClass = Class.forName(name + "Factory");
		Object factInst;
		try {
			factInst = factoryClass.newInstance();
		} catch (InstantiationException e) {
			error[0] = ERR_INSTANCE_ERROR;
			return null;
		} catch (IllegalAccessException e) {
			error[0] = ERR_INSTANCE_ERROR;
			return null;
		}
		if (! (factInst instanceof Factory)) {
			error[0] = ERR_NOT_A_FACTORY;
			return null;
		}
		return (Factory)factInst;
	} catch (ClassNotFoundException e) {
		error[0] = ERR_CLASS_NOT_FOUND;
		return null;	
	}
}


/**
 * Adds a new parameter. 
 */
public int newParam(String name, Object value) {

	if (put(name, value) != null) {
		return WARN_PARAM_EXISTED;
	}
	return OK;
}

/**
 * Returns an array of strings that contain the names
 * of missing parameters, or <code>null</code> if no parameters
 * are missing. 
 * <p>
 * A generic tool should check this
 * array before trying to create() an instance. If the returned array
 * is not <code>null</code>, the user should be asked to provide values
 * for the missing parameters. 
 * <p>
 * The default implementation of this method returns <code>null</code>.
 */
public String[] missingParams() {
	
	return null;
}


/**
 * Creates a new instance.
 * <p>
 * This method must be implemented by all <em>meta factory classes</em>.
 */
abstract public Object create();


} // class Factory

