/*
 * @(#)IModel.java
 * 
 */

package openstar.generic;

import openstar.util.*;

/**
 * Generic model interface.
 * <p>
 * A <em>model</em> in the OpenStar system is an object,
 * probably consisting of many different other objects,
 * that has the following characteristics:
 * <ul>
 * <li>The object lives for a relatively long time
 * (compared to other objects in the system).
 * <li>Other objects may register themselves as <em>Listeners</em>
 * at the model and get notifed when the model (or parts of the model) 
 * change.
 * <li>The model has neither a visualization or a GUI-based user interaction 
 * component, nor does it have any explicit knowledge of such components.
 * </ul>
 * <p>
 * Note: In most cases, applications will use the class <code>Model</code>
 * as the base for own model classes instead of always implementing all the 
 * methods of this interface from scratch. The interface definition is
 * provided, so own model classes can be derived from application specific
 * domain classes, while also implementing the <code>IModel</code>
 * interface. Just remember, when implementing own extensions to the
 * framework, that wherever a model object is expected, it must be
 * an <code>IModel</code>, and not a <code>Model</code>.
 *
 * @version 16 Jun 1997
 * @author 	Nils Schwabe 
 */
public interface IModel
{

/**
 * Adds a listener to this model.
 * <p>
 * The listener gets notified when the model changes.
 */
public void addModelChangeListener(ModelChangeListener listener);


/**
 * Removes a listener from the model.
 */
public void removeModelChangeListener(ModelChangeListener listener);


/**
 * Fires a <code>ModelChangeEvent</code> to all registered listeners.
 * <p>
 * Nothing happens if the notifications switch of this model is "off"
 * (@see getEnableNotifications(), @see getEnableNotifications()).
 * @param source The originator of the change (often the calling object)
 */
public void setModelChanged(Object source);


/**
 * Enable or disable notification of registered event listeners. 
 * <p>
 * If notifications are turned off, the <code>setModelChanged()</code>
 * methods do nothing. The default setting is "on". You can turn
 * notifications off to make extensive changes to a model without
 * every change triggering an event. When finished,
 * you should turn notifications "on" again and make one call to
 * the <code>setModelChanged()</code> method to inform all listeners
 * of the changes.
 */
public void setEnableNotifications(boolean enable);

/**
 * @return 
 * <code>true</code> -- notifications are currently enabled.<br>
 * <code>false</code> -- notifications are currently disabled.
 */
public boolean getEnableNotifications();




} // interface IModel
