/*
 * @(#)ModelSelection.java
 * 
 */

package openstar.generic;

import openstar.util.*;

/**
 * Generic model selection class.
 * <p>
 * A model selection is the description of an arbitrary subset
 * of a specific model instance. Because nothing is known about
 * the internal structure of models at this generic level, all
 * details of a model selection must be represented through classes
 * that are derived from this class.
 * <p>
 * As with models, interested parties may register with a selection
 * to get notified of any changes in the selection.
 *
 * @version 16 Jun 1997
 * @author 	Nils Schwabe 
 */
public abstract class ModelSelection implements EventDelivery
{

/** The model of this model selection. */
protected Model model;

protected EventSupport support;

public ModelSelection(Model model) {
	this.model = model;
	support = new EventSupport();
}


public Model getModel() {
	return model;
}


public void addModelSelectionListener(ModelSelectionListener listener) {
	support.addEventListener(listener);
}


public void removeModelSelectionListener(ModelSelectionListener listener) {
	support.removeEventListener(listener);
}


/**
 * Fires a <code>ModelSelectionEvent</code> to all registered listeners.
 */
public void fireModelSelection(Object source) {
	support.fireEvent(new ModelSelectionEvent(source, this), this);
}

/**
 * Fires a <code>ModelSelectionEvent</code> to all registered listeners.
 */
protected void fireModelSelection() {
	support.fireEvent(new ModelSelectionEvent(this, this), this);
}


public void deliverEvent(java.util.EventListener target, Event event) {
	((ModelSelectionListener)target).modelSelection((ModelSelectionEvent)event);
}


} // class ModelSelection
