/*
 * @(#)Parameter.java
 * 
 */

package openstar.generic;

/**
 * A <code>Parameter</code> consists of a <em>name</em> (a <code>String</code> object)
 * and a current <em>value</em> (an arbitrary <code>Object</code>).
 * <p>
 * The name of a parameter is used (within a given context) to uniquely
 * identify the parameter. Therefore, it cannot be changed after the parameter
 * has been created. 
 * <p>
 * The value of a parameter can be changed during the lifetime of the
 * parameter object.
 * <p>
 * Two parameter objects are equal if they have equal names and equal 
 * values. 
 * <p>
 * The type of a parameter name is always <code>String</code>, and the
 * string will usually have the form of an identifier in common programming
 * languages, e.g. "color", "width", etc. However, while the type of a
 * parameter's value is defined to be an <code>Object</code>, one should
 * always pass around complete <code>Parameter</code> objects and avoid
 * to make copies of these objects. This is essential for the correct
 * behaviour of the notification mechanism associated with parameters.
 *
 * @version 11 Jun 1997
 * @author 	Nils Schwabe
 */ 
public class Parameter implements java.io.Serializable
{
	
protected String name;
protected Object value;
private transient java.beans.PropertyChangeSupport support;

public Parameter(String name) {
	this.name = name;
	this.value = null;
	support = new java.beans.PropertyChangeSupport(this);
}

public Parameter(String name, Object value) {
	this.name = name;
	this.value = value;
	support = new java.beans.PropertyChangeSupport(this);
}


public String getName() {
	return this.name;
}

/**
 * Changes the value of the parameter.
 * <p>
 * A <code>java.beans.PropertyChangeEvent</code> is fired to all registered
 * listeners (only if the new value is different from the current value).
 */ 
public void setValue(Object value) {
	if (value != this.value) {
		Object oldValue = this.value;
		this.value = value;
		support.firePropertyChange(name, oldValue, value);
	}
}

public Object getValue() {
	return this.value;
}

/**
 * Returns <code>true</code> if and only if the given object
 * is a <code>Parameter</code> and both, name and value of the
 * parameter, are equal to the name and value of this object.
 * <p>
 * Note that equality in this sense does not imply that both
 * parameters are the same objects.
 */
public boolean equals(Object object) {
	if (object == null || ! (object instanceof Parameter)) {
		return false;
	}
	Parameter param = (Parameter)object;
	return (   this.name.equals(param.name)
	        && this.value.equals(param.value));
}


/**
 * Returns an independent copy of this parameter.
 * <p>
 * The new parameter has no registered listeners.
 */
public Object clone() {
	return new Parameter(this.name, this.value);
}


public void addPropertyChangeListener(java.beans.PropertyChangeListener listener) {
	support.addPropertyChangeListener(listener);
}


public void removePropertyChangeListener(java.beans.PropertyChangeListener listener) {
	support.removePropertyChangeListener(listener);
}




} // class Parameter

