/*
 * @(#)EventSupport.java
 * 
 */

package openstar.util;

import openstar.generic.*;

/**
 * This is a utility class (similar, for example, to 
 * java.beans.PropertyChangeSupport)
 * that can be used to delegate the event-related work.
 * <p>
 * You can either inherit from this class (as long as you fire only
 * one type of event) or you can use one or more instances
 * of this class as members of your event-generating class.
 *
 * @version 11 Jun 1997
 * @author 	Nils Schwabe  
 */
public class EventSupport
{

/**
 * The list of registered listeners for the event. 
 * <p>
 * All listeners are required to be derived from 
 * <code>java.util.EventListener</code>.
 */
protected TypedList listeners;


public EventSupport() {
	listeners = new TypedList("java.util.EventListener", true);
}

/**
 * Adds a listener to the event listener list. 
 * <p>
 * No checks for multiple occurences are made.
 */
public void addEventListener(java.util.EventListener listener) {
	listeners.addElement(listener);
}


/**
 * Removes the (first matching) listener from the event listener list. 
 * <p>
 * No checks for multiple occurences are made.
 */
public void removeEventListener(java.util.EventListener listener) {
	listeners.removeElement(listener);
}


/**
 * Fires the specified event.
 * <p>
 * The event is delivered using the given <code>EventDelivery</code>
 * object which is responsible to cast the generic <code>Event</code> 
 * to the appropriate derived type and to call the appropriate method of
 * the <code>EventListener</code>. 
 * <p>
 * The event is delivered to the set of listeners that are registered
 * at the moment. Changes to the list of listeners that occur indirectly
 * from the event processing methods do not effect the set of listeners
 * that get the event.
 */
public void fireEvent(Event event, EventDelivery delivery) {

	java.util.EventListener listener;
	int i = 0;
	// Make an independent copy of the current list of listeners.
	// This is to be thread safe.
	TypedList curListeners = (TypedList)listeners.clone();
	while ((listener = (java.util.EventListener)curListeners.getAt(i)) != null) {
		delivery.deliverEvent(listener, event);
		i++;
	}
}


/**
 * Returns a copy of the list of listeners.
 *
 * All objects in this list can be safely casted to
 * <code>java.util.EventListener</code>.
 */
public TypedList getListeners() {
	TypedList curListeners = (TypedList)listeners.clone();
	return curListeners;
}

} // class EventSupport
