/*
 * @(#)Matrix.java
 * 
 */

package openstar.util;

/**
 * A simple general purpose matrix class operating on double values.
 *
 * @version 09 Apr 1997
 * @author 	Nils Schwabe
 */
public class Matrix implements Cloneable
{

private double[] e;		// array containing the matrix values 

private int rows;		// number of currently used rows

private int columns;	// number of currently used columns

private int _rows;		// number of allocated rows

private int _columns;	// number of allocated columns


/**
 * Constructs a matrix with a given number of rows and columns.
 * All elements are initialized with zero.
 * @param cntRows the number of rows
 * @param cntColumns number of columns
 * 
 */
public Matrix(int cntRows, int cntColumns) {

	e = new double[cntRows*cntColumns];
	for (int i=0; i < cntRows*cntColumns; i++) {
		e[i] = 0;
	}
	rows = _rows = cntRows;
	columns = _columns = cntColumns;
}	

/**
 * Constructs and initializes a 3x3 matrix.
 * @param a first row
 * @param b second row
 * @param c third row
 */ 	 	 	
public Matrix(double a1, double a2, double a3,
	          double b1, double b2, double b3,
	          double c1, double c2, double c3) {
	           
	e = new double[9];
	e[0]=a1; e[1]=a2; e[2]=a3;
	e[3]=b1; e[4]=b2; e[5]=b3;
	e[6]=c1; e[7]=c2; e[8]=c3;
	rows = _rows = 3;
	columns = _columns = 3;	
}

/**
 * Makes a deep copy of this matrix object.
 */
public Object clone() {

	Matrix m = new Matrix (this.columns, this.rows);
	m.rows = this.rows;
	m.columns = this.columns;
	m._rows = this._rows;
	m._columns = this._columns;
	for (int i=0; i < this.columns * this.rows; i++) {
		m.e[i] = this.e[i];
	}
	return m;
}

/**
 * Makes a deep copy of this matrix object and stores the
 * values in another given matrix.
 */
public void copyInto(Matrix res) {

	res.e = new double[this._rows*this._columns];
	res.rows = this.rows;
	res.columns = this.columns;
	res._rows = this._rows;
	res._columns = this._columns;
	for (int i=0; i < this.columns * this.rows; i++) {
		res.e[i] = this.e[i];
	}
}

/**
 * Returns true if both matrices have the same number of
 * rows and columns and all elements have equal values.
 */
public boolean equals(Object obj) {

	if (! (obj instanceof Matrix)) {
		return false;
	}
	Matrix m = (Matrix)obj;
	if (m.rows != this.rows || m.columns != this.columns) {
		return false;
	}
	for (int i=0; i < this.columns * this.rows; i++) {
		if (m.e[i] != this.e[i])
			return false;
	}
	return true;
}

public String toString() {

	int i, j;
	String res = "";
	for (i = 0; i < rows; i++) {
		for (j = 0; j < columns; j++) {
			res = res + get(i,j) + "\t";
		}
		res = res + "\n";
	}
	return res;
}		


/**
 * Returns the specified element of the matrix.
 * @param row zero-based row index
 * @param column zero-based column index
 */
public double get(int row, int column) {

	return e[row * _columns + column];
}


/**
 * Sets the specified element of the matrix.
 * @param row zero-based row index
 * @param column zero-based column index
 * @param value new value of the matrix element
 */
public void set(int row, int column, double value) {

	e[row * _columns + column] = value;
}

/**
 * Returns the number of rows of the matrix.
 */
public int getRows() {

	return rows;
}

/**
 * Returns the number of columns of the matrix.
 */
public int getColumns() {

	return columns;
}

/**
 * Resizes the matrix.
 * <br>Resizing is only possible if the matrix has been created
 * with a sufficient size.
 * @param cntRows new number of rows
 * @param cntColumns new number of colums
 * @return true - successful operation<br>
 * false - initial size of matrix not sufficient
 */
public boolean resize(int cntRows, int cntColumns) {

	if (_rows < cntRows || _columns < cntColumns) 
		return false;
	rows = cntRows;
	columns = cntColumns;
	return true;	
}

/**
 * Multiplies this matrix with another matrix and stores
 * the result in yet another matrix.
 * @param v matrix with which this matrix should be multiplied
 * @param result resulting matrix (must have correct size)
 */
public void multiply(Matrix v, Matrix result) {

	if (columns != v.rows) 
	{
		// <EXC>
		return;
	}

	double s;
	for (short i = 0; i < rows; i++) 
	{
		for (short j = 0; j < v.columns; j++) 
		{
			s = 0;
			for (short k = 0; k < columns; k++)
				s = s + get(i,k) * v.get(k,j);
			result.set (i,j,s);
		}
	}		
}

/**
 * Multiplies this matrix with a vector and stores
 * the result in another vector.
 * @param p vector with which this matrix should be multiplied
 * @param result resulting vector (must have correct size)
 */	
public void multiply(Point p, Point result) {

	if (columns != p.getDim()) 
	{
		// <EXC>
		return;
	}

	double s;
	for (short i = 0; i < rows; i++) 
	{
		s = 0;
		for (short k = 0; k < columns; k++)
			s = s + get(i,k) * p.get(k);
		result.set (i,s);
	}
}

/**
 * Inverts the matrix.
 * @return success indicator. 
 */
public boolean invert() {

	int[] pcol = new int[rows];
	for (int i = 0; i < rows; i++) {
		pcol[i] = -1;
	}
	
	if (findPivotColumns(0, pcol)) {
	
		int i, j;
		
		// do pivot step in each row:
		for (i = 0; i < rows; i++) {
			pivotStep(i, pcol[i]);
		}
		
		// permutate rows in columns according to pivot elements:
		Matrix cur = new Matrix(rows, columns);
		for (i = 0; i < rows; i++) {
			for (j = 0; j < columns; j++) {
				cur.set(pcol[i], j, get(i,j));
			}
		}
		
		cur.copyInto(this);
		
		for (j = 0; j < columns; j++) {
			for (i = 0; i < rows; i++) {
				cur.set(i, j, get(i,pcol[j]));
			}
		}
		
		cur.copyInto(this);
		return true;
	}
	return false;
}


private void pivotStep(int prow, int pcol) {

	int i, j;
	double alpha;
	
	alpha = 1.0 / get(prow, pcol);
	set(prow, pcol, 1.0);
	
	// Appy rectangle rule first, because it needs the unchanged
	// information of the pivot row/column elements:
	for (i = 0; i < rows; i++) {
		for (j = 0; j < columns; j++) {
			if (i != prow && j != pcol) {
				set(i, j, get(i,j) - alpha * get(prow,j) * get(i,pcol));
			}
		}
	}

	// Apply pivot row/column rules:
	for (i = 0; i < rows; i++) {
		for (j = 0; j < columns; j++) {
			if (i == prow) {
				set(i, j, alpha * get(i,j));
			}
			else if (j == pcol) {
				set(i, j, -alpha * get(i,j));
			}
		}
	}
}

private boolean findPivotColumns(int row, int[] pcol) {

	int j;
	pcol[row] = -1;
	for (j = 0; j < this.columns; j++) {
		if (get(row, j) != 0.0) {
			boolean ok = true;
			for (int k=0; k < row; k++) {
				if (pcol[k] == j) {
					ok = false;
					break;
				} 
			}
			if (ok) {
				pcol[row] = j;
				if (row == rows-1 || findPivotColumns(row+1, pcol)) {
					return true;
				}
			}
		}
	}
	pcol[row] = -1;
	return false;
}



} // class Matrix

