/*
 * @(#)Point.java
 * 
 */

package openstar.util;

/**
 * A class representing points in the n-dimensional euclidean space.
 *
 * @version 09 Apr 1997
 * @author 	Nils Schwabe
 */ 
public class Point implements Transformable
{

private int dims;
private double[] x;

/**
 * Interval in which two coordinate values are treated as beeing equal.
 */
public double equalityTreshold = 0.0001;


/**
 * Constructs a new n-dimensional point.
 * @param dimensions number of dimensions
 */
public Point(int dimensions) {
	this.dims = dimensions;
	this.x = new double[dimensions];
}

/**
 * Constructs a 2-D point with specified coordinates.
 * @param x first coordinate
 * @param y second coordinate
 */
public Point(double x, double y) {
	this.dims = 2;
	this.x = new double[2];
	this.x[0] = x;  
	this.x[1] = y;
}

/**
 * Constructs a 3-D point with specified coordinates.
 * @param x first coordinate
 * @param y second coordinate
 */
public Point(double x, double y, double z) {
	this.dims = 3;
	this.x = new double[3];
	this.x[0] = x;
	this.x[1] = y;
	this.x[2] = z;
}

/**
 * Makes a deep copy of this point.
 */
public Object clone() {
	Point p = new Point (this.dims);
	for (int i=0; i < this.dims; i++)
		p.x[i] = this.x[i];
	return p;
}


/**
 * Assigns the coordinate values of another point to this point.
 * Dimension conflicts are ignored, i.e. if the other point has
 * too many coordinates then they are ignored, or the remaining
 * coordinates of this point are set to 0.
 */
public void assign(Point p) {
	
	int minDim = Math.min(this.dims, p.dims);
	
	for (int i = 0; i < minDim; i++) {
		this.x[i] = p.x[i];
	}
	int remain = this.dims - minDim;
		
	for (int i = minDim; i < minDim+remain; i++) {
		this.x[i] = 0.0;
	}
}

public String toString() {
	String res = "(";
	for (int i=0; i < this.dims; i++) {
		res = res + this.x[i];
		if (i < this.dims-1)
			res = res + ", ";
	}
	return res + ")";
}
	
public boolean equals (Object obj) {
	if (! (obj instanceof Point))
		return false;
	Point p = (Point)obj;	
	int maxDims = Math.max(this.dims, p.dims);
	double a, b;
	for (int i=0; i < maxDims; i++) {
		a = get(i);
		b = p.get(i);
		if (Math.abs(a-b) > equalityTreshold) {
			return false;
		}
	}
	return true;
}



/**
 * Gets the coordinate value for a given dimension.
 * @param index zero-based index (0.0 is returned if the index is too big)
 */
public double get (int index) {
	if (index >= this.dims)
		return 0.0;
	return this.x[index];
}


/**
 * Sets the coordinate value for a given dimension.
 * @param index zero-based index (ignored if the index is too big)
 * @param newVal new value
 */
public void set (int index, double newVal) {
	if (index < this.dims)
		this.x[index] = newVal;
}

/**
 * Returns the number of dimensions of this point.
 */
public int getDim () {
	return this.dims;
}


/**
 * Transforms this point.
 */
public void transform(Transformation t) {

	t.transform(this, this);
}



} // class Point
