/*
 * @(#)TypedList.java
 * 
 */
 
package openstar.util;

import java.util.*;

/**
 * Vector-like list class that is restricted to objects of a
 * given type.
 *
 * @version 09 Apr 1997
 * @author 	Nils Schwabe 
 */
public class TypedList implements Cloneable
{

protected ClassTest classTest;
protected Vector vector;

/**
 * @param requiredType the fully qualified name of the class to which all
 *        elements of the list must belong.
 * @param polymorphic <code>true</code> indicates that objects belongig
 *        to subclasses of the specified type may also be stored in the list.
 */
public TypedList(String requiredType, boolean polymorphic) {

	try {
		Class c = Class.forName(requiredType);
		classTest = new ClassTest(c, polymorphic);
	} catch (ClassNotFoundException e) {
		e.printStackTrace();
	}
	vector = new Vector();
}


/**
 * @param requiredType the fully qualified name of the class to which all
 *        elements of the list must belong.
 * @param polymorphic <code>true</code> indicates that objects belongig
 *        to subclasses of the specified type may also be stored in the list.
 * @param initialCapacity the initial capacity of the underlying <code>Vector</code>
 *        that is used to store the elements of the list.
 */
public TypedList(String requiredType, boolean polymorphic, int initialCapacity) {

	try {
		Class c = Class.forName(requiredType);
		classTest = new ClassTest(c, polymorphic);
	} catch (ClassNotFoundException e) {
		e.printStackTrace();
	}
	vector = new Vector(initialCapacity);
}


/**
 * Makes an independent copy of this list.
 * <p>
 * The elements of the list are not cloned, i.e. only the references
 * to the objects are copied.
 */
public synchronized Object clone() {

	TypedList cloned = new TypedList(classTest.getClassName(), classTest.isPolymorphic());	
	int aSize = vector.size();
	Object cur;
	for (int i = 0; i < aSize; i++) {
		cur = vector.elementAt(i);
		cloned.vector.addElement(cur);
	}
	return cloned;
}

/**
 * Returns the number of elements in the list.
 */
public int size() {

	return vector.size();
}

/**
 * Same as size().
 */
public int length() {

	return vector.size();
}


public boolean isEmpty() {

	return vector.isEmpty();
}


public int indexOf(Object elem, int startIndex) throws IllegalArgumentException {

	test(elem);
	return vector.indexOf(elem, startIndex);
}	


public Object elementAt(int index) throws ArrayIndexOutOfBoundsException {

	return vector.elementAt(index);
}

/**
 * Same as elementAt() but returns null in case of an invalid index.
 */
public Object getAt(int index) {

	try {
		return vector.elementAt(index);
	} catch (ArrayIndexOutOfBoundsException e) {
		return null;
	}
}

public void setElementAt(Object elem, int index) 
			throws IllegalArgumentException, ArrayIndexOutOfBoundsException {
			
	test(elem);
	vector.setElementAt(elem, index);
}


public void removeElementAt(int index) throws ArrayIndexOutOfBoundsException {
			
	vector.removeElementAt(index);
}


public void removeElement(Object elem) {

	vector.removeElement(elem);
}


public void insertElementAt(Object elem, int index) 
			throws IllegalArgumentException, ArrayIndexOutOfBoundsException {
			
	test(elem);
	vector.insertElementAt(elem, index);
}


public void addElement(Object elem) throws IllegalArgumentException {

	test(elem);
	vector.addElement(elem);
}


public void add(TypedList a) throws IllegalArgumentException {

	int aSize = a.vector.size();
	Object cur;
	for (int i = 0; i < aSize; i++) {
		cur = a.vector.elementAt(i);
		test(cur);
		vector.addElement(cur);
	}
}


public void removeAllElements() {

	vector.removeAllElements();
}


public String toString() {

	return "TypedList<" + classTest.getClassName() + "> " + vector.toString();
}


/**
 * Tests if the given object may be stored in this list.
 * <p>
 * If the object has an invalid type, an <code>IllegalArgumentException</code>
 * is thrown. Otherwise, nothing happens.
 */
protected void test(Object object) throws IllegalArgumentException {

	if (object == null) {
		throw new IllegalArgumentException("null Object not valid in TypedList");
	}
	if (! classTest.isValid(object)) {
		throw new IllegalArgumentException("Type check failed in TypedList");
	}
}


/**
 * Returns <code>true</code> if and only if the given object is also a
 * <code>TypedList</code> (no matter which subclass thereof), and
 * both lists contain the same number of objects, and the 
 * application of the <code>equals</code> method of all elements returns
 * <code>true</code>.
 * <p>
 * Note 1: the above definition implies that two <code>TypedList</code>
 * objects can be equal even if they support two types that are not related
 * to each other in any way, as long as the actual objects in the list
 * are related to both types through an <code>implements</code> or 
 * <code>extends</code> clause.
 * <p>
 * Note 2: the implementation of the <code>equals</code> method should,
 * according to the Java specs, always be <em>symetric</em>. Thus, the
 * information which <code>equals</code> method for a given pair of
 * list elements is actually used should be irrelevant.
 */
public boolean equals (Object obj) {

	if (! (obj instanceof TypedList))
		return false;
		
	TypedList other = (TypedList)obj;
	int otherSize = other.vector.size();
	int mySize = vector.size();
	if (otherSize != mySize) {
		return false;
	}
	
	Object curMine;
	Object curOther;
	for (int i=0; i < mySize; i++) {
		curMine = vector.elementAt(i);
		curOther = other.vector.elementAt(i);
		if (! curMine.equals(curOther)) {
			return false;
		}
	}	
	return true;
}

/**
 * Returns the type name of the elements of this list.
 * <p>
 * All elements are guaranteed to be either of this exact type, or (
 * if <code>isPolymorphic()</code> returns <code>true</code>) to
 * be of this type or a subclass thereof.
 */
public String getElemClassName() {

	return classTest.getClassName();
}


public boolean isPolymorphic() {
	
	return classTest.isPolymorphic();
}

} // class TypedList

